/**** X11 interface for network map program.  ****/

/*	Copyright (C) 1989, 1990, 1991 Massachusetts Institute of Technology
 *		Right to copy granted under the terms described
 *		in the file Copyright accompanying this distribution.
 */


/* This is the X interface for the program that displays network connectivity.
 * The data presently is more or less fixed by values in certain data files.
 * Eventually I hope to be able to do this more dynamically.
 */

#include "map.h"
#include "debug.h"

#include <X11/Xlib.h>
#include <X11/keysym.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>



#ifdef EDIT_HISTORY

Nbr Date	Author		Description
--- ----	------		-----------
 3  20-Feb-89	M. A. Patton	First release version.
 3A  4-Mar-89	M. A. Patton	No longer precompute Vertex list, computed on
				the fly while displaying.  Finally got all X
				stuff in here, moved X includes here from map.h
 4  16-Mar-89	M. A. Patton	Converted to X11.  No custom cursor, dpy arg
				gets set too late, window create is too trivial
				(e.g. it doesn't use spec'd geometry)
 4A 20-Mar-89	M. A. Patton	Added pxl2rgb to remove X from TEK module.
				(From report by P. V. O'Neill)
 4B 21-Mar-89	Chris Tengi	Improved window creation somewhat.
 4C 22-Mar-89	M. A. Patton	More window creation improvements.  Use the
				user specified geometry, pick a cursor.  Open
				display early by prescan of args.
 4D 23-Mar-89	M. A. Patton	Added waiting flag so redisplay in middle of
				things gets correct bottom line.
 4E 23-Mar-89	M. A. Patton	Cosmetics from report by Chris Tengi.
 4F 29-Jun-89	M. A. Patton	Fixed bug in pxl2rgb (called XQueryColor with
				X10 args, changed to X11).  Draw box around a
				title when either dimension given.
 4G  8-Jul-89	M. A. Patton	Added debug options for tracing events and
				color mapping.
 4H 12-Jul-89	M. A. Patton	Foreground color wrong on TEK output, added
				debug for pxl2rgb so I can track it.
 4I  1-Aug-89	M. A. Patton	Fixed up black and white, needs much more. Also
				defaults for IP and CHAOS pixels were wrong.
 4J 24-Jan-90	M. A. Patton	Added line to ignore keboard mapping notify.
				Process ^U.  Drop all links from hosts when
				deleting a network.
 4K 10-Mar-90	M. A. Patton	Added hack to change cursor when in "wait ..."
 5  12-Jun-90	M. A. Patton	Added getOption (for SNMP).
				Added message when taps miss net.
 5A 15-Jun-90	M. A. Patton	Made keypad "Enter" key work.
 5B 17-Jun-90	M. A. Patton	If there were no ConfigureNotify events, we
				don't know the size and prompt clears screen.
 5C  9-Aug-90	M. A. Patton	Networks were never setting the AlreadyPlaced
				flag.
 5D 30-Nov-90	M. A. Patton	Somehow under GWM and no window manager (at
				least), it always shows up at (0, 0) no matter
				where the command line says...  Needed to move
				the window create down after geo parse...
 5E 10-Mar-91	M. A. Patton	Added GravityNotify to ignored event types.
 5F  1-Apr-91	Mike Patton	Added HighlightFG (Highlight is BG of
				highlighted host).  This defaults so that on
				black and white displays the highlighting is
				by reverse video and on color the default is
				to just change the background color.
 5G  5-Apr-91	Mike Patton	Added seperate routine to calculate placement.
				This is done in a different order because of
				the dependency of point-to-point nets.
				Made point-to-point nets look only at hosts
				that are actually displayed (i.e. machines) so
				that for "external connections" from a map you
				can give one host "over there" and show as a
				point-to-point style connection.

#endif
/****	Internal Documentation and External calling sequences	****/


#ifdef DOCUMENTATION

General conventions:
    subroutine and function names:
	Place...	Does recalculations of X internal (i.e. display
				position) parts of the data structure.
	Display...	Actually draws something on the screen, does nothing
				else.
	Erase...	Draws something in the background color, may erase
				parts of other things.
	ReDraw...	Calls appropriate Erase... and Display... routines for
				the given thing and then calls ReDraw... of
				other things that might be affected.

    !!!! Needs More !!!!

#endif
/****	Parameters used by this module.	****/


#define TITLE "Network Mapper"


int border_width = 3;		/* The width of the border */
char *display_name = NULL;	/* The name of the display */
char *border_color = "Black";	/* The color for the border */
char *back_color = "White";	/* The background color */
char *fore_color = "Black";	/* The text color */
char *highlight_color = "LightBlue"; /* Background of the selected host */
char *highlight_fg_color = "Navy"; /* Text in selected host */
char *font_name = "6x10";	/* The font to use */
char *geometry = NULL;		/* Pointer to geometry string for window */


/* A list of colors to use for defaulting.  */
static char *default_color[] = {
    "Blue",
    "Yellow",
    "Red",
    "Orange",
    "LightGray",
    "DimGray",
    "Khaki",
    "Goldenrod",
    NULL			/* Terminator */
};
static int default_color_index = 0;	/* Which one we're on */




/* Global information, used during operation. */
static Display *dpy = NULL;
int screen;

int display_height;
int display_width;

int foreground;
int background;
int border;
int highlight;
int highlight_fg;
int IP_pixel, CHAOS_pixel;

Window wind;
XFontStruct *font_info;

Cursor norm_cursor;
Cursor wait_cursor;

GC line_gc;	/* Used for lines and text (fg and bg normal) */
GC fill_gc;	/* Used with random fills (i.e. nets & protos),
		   fg, width changes as needed */
GC bg_gc;	/* for filling in background color (i.e. fg=bg) */
GC highlight_gc; /* GC for selected host background */
GC highlight_fg_gc; /* GC for selected host text */


XSizeHints xsh = {	/* Size hints for window manager */
    (PPosition|PSize),		/* flags */
    0, 0,			/* position */
    668, 530,			/* dimensions */
    20, 20,			/* minimum dimensions */
    0, 0,			/* max dim. (not used) */
    1, 1,			/* increments (any size is usable) */
    0,0,0,0,			/* aspect ratios (who cares) */
};

XWMHints xwmh = {	/* Other hints for the Window Manager */
    (InputHint|StateHint),      /* flags */
    True,			/* input */
    NormalState,		/* initial state */
    0,                          /* icon pixmap */
    0,                          /* icon window */
    0, 0,			/* icon location */
    0,                          /* icon mask */
    0,                          /* Window group */
};



/* Random state variables */
static int need_redisplay = FALSE;
static waiting = TRUE;

static char *prompt = "Command:";
static char *waitmsg = "Wait ...";
static char input[INPUT_LEN];
static int input_index = 0;
/**** Initialization ****/




caddr_t
InitDisplay()
{   if (dpy == NULL)
	dpy = XOpenDisplay(display_name);
    if (dpy!=NULL)
    {	screen = DefaultScreen(dpy);
	if (!(font_info = XLoadQueryFont(dpy, font_name)))
	    error("XLoadQueryFont failed");
	OpenWindow();
	DisplayWait();
    }
    return((caddr_t)dpy);
}




GetDefaults()
{   char *option;

    /* For now we're going to look ahead to the args for disp name */
    if (dpy==NULL)
    {	int i = 0;
	char *cp = NULL;
	while (++i < Argc)
	{   if (index(Argv[i],':')!=NULL)
	    {	cp = Argv[i];
		continue;
	    }
	    S_ARG("-display",cp);
	}
	dpy = XOpenDisplay(cp);
	if (dpy==NULL) return;
    }
    if ((option = XGetDefault(dpy, Argv[0], "BorderWidth")) !=NULL)
	border_width = atoi(option);
    if ((option = XGetDefault(dpy, Argv[0], "Border")) !=NULL)
	border_color = option;
    if ((option = XGetDefault(dpy, Argv[0], "BorderColor")) !=NULL)
	border_color = option;
    if ((option = XGetDefault(dpy, Argv[0], "Background")) !=NULL)
	back_color = option;
    if ((option = XGetDefault(dpy, Argv[0], "Foreground")) !=NULL)
	fore_color = option;
    if ((option = XGetDefault(dpy, Argv[0], "Highlight")) !=NULL)
	highlight_color = option;
    if ((option = XGetDefault(dpy, Argv[0], "HighlightFG")) !=NULL)
	highlight_fg_color = option;
    if ((option = XGetDefault(dpy, Argv[0], "Font")) !=NULL)
	font_name = option;
}

/* Get one option (for SNMP configuration, eventually better?) */
char *
getOption(optname)
char *optname;
{   if (dpy==NULL) return (NULL);
    return (XGetDefault (dpy, Argv[0], optname));
}
/**** Set up the window ****/



OpenWindow()
{   XGCValues vals;

    /* Deal with colors */
    /* Background */
    background = InterpretColor(back_color, WhitePixel(dpy,screen));
    /* Foreground */
    foreground = InterpretColor(fore_color, BlackPixel(dpy,screen));
    /* Border */
    border = InterpretColor(border_color, foreground);
    /* Highlight (selected host) */
    highlight = InterpretColor(highlight_color, foreground);
    highlight_fg = InterpretColor(highlight_fg_color,
			    (highlight==foreground) ? background : foreground);
    /* IP and CHAOS protocol colors (need to save for later use) */
    IP_pixel = InterpretColor("IP", background);
    CHAOS_pixel = InterpretColor("CHAOS", background);
    /* Do this simply now, snaz up latter!!!! */
    /* Remember these values, in case we don't get told any others */
    display_width = xsh.width;
    display_height = xsh.height;
    /* Set up position and size for the window */
    if (geometry != NULL)
    {	int f = XParseGeometry(geometry,
			       &xsh.x, &xsh.y, &xsh.width, &xsh.height);
	if (f&(XValue|YValue)) xsh.flags |= USPosition;
	if (f&(WidthValue|HeightValue))
	{   xsh.flags |= USSize;
	    if (f&WidthValue) display_width = xsh.width;
	    if (f&HeightValue) display_height = xsh.height;
	}
    }
    /* Create the window with specified location & size */
    wind = XCreateSimpleWindow(dpy, RootWindow(dpy,screen),
			       xsh.x, xsh.y, xsh.width, xsh.height,
			       border_width, border, background);
    XSetStandardProperties(dpy, wind, TITLE, TITLE, None, Argv, Argc, &xsh);
    XSetWMHints(dpy, wind, &xwmh);
    XSelectInput(dpy, wind, ButtonPressMask | KeyPressMask |
			    ExposureMask | StructureNotifyMask );
    /* Make the cursors */
    norm_cursor = XCreateFontCursor(dpy, XC_cross_reverse);
    wait_cursor = XCreateFontCursor(dpy, XC_watch);
    XDefineCursor(dpy, wind, wait_cursor);
    /* Set up the GCs */
    vals.foreground = foreground;
    vals.background = background;
    vals.font = font_info->fid;
    /* normal drawing for lines and text in foreground color */
    line_gc = XCreateGC(dpy,  wind, GCForeground|GCBackground|GCFont, &vals);
    /* foreground changes as needed */
    fill_gc = XCreateGC(dpy, wind, GCForeground|GCBackground, &vals);
    vals.foreground = background;
    bg_gc = XCreateGC(dpy, wind, GCForeground|GCBackground, &vals);
    vals.foreground = highlight;
    highlight_gc = XCreateGC(dpy, wind, GCForeground|GCBackground, &vals);
    vals.foreground = highlight_fg;
    highlight_fg_gc = XCreateGC(dpy, wind,
				GCForeground|GCBackground|GCFont, &vals);
    XMapWindow(dpy, wind);
}
/**** Display management ****/


ForceRedisplay()
{   need_redisplay = TRUE;
    XClearWindow(dpy, wind);
}


ForceDisplay()
{   /* Should be more integrated and handle some kinds of events that	*/
    /* might be in the queue.						*/
    if (need_redisplay) Redisplay();
    XFlush(dpy);
}




Redisplay()
{   register Network *np;
    register Machine *hp;
    register Title *tp;

/** fprintf(stderr,"Beginning full redisplay.\n"); /**/
    need_redisplay = FALSE;
    MAPCAR (np, net_base)
	DisplayNetwork(np);
    MAPCAR (hp, machine_base)
	DisplayHost(hp->host);
    MAPCAR (tp, title_base)
	DisplayTitle(tp);
    if (waiting)
    {	DisplayWait();
    }
    else
    {	DisplayPrompt();
	DisplayInput();
    }
/** fprintf(stderr,"Full redisplay complete.\n");XFlush(dpy);sleep(5); /**/
}




CalculatePlacement()
{   register Network *np;
    register Machine *hp;
    register Title *tp;

    need_redisplay = TRUE;
    MAPCAR (hp, machine_base)
	PlaceHost(hp->host);
    MAPCAR (np, net_base)
	PlaceNetwork(np);
    MAPCAR (tp, title_base)
	PlaceTitle(tp);
}



Display_Message(cp)
char *cp;
{   fputs(cp, stdout);
}
/****	Compute placement for a Host entry	****/


PlaceHost(p)
register Host *p;
{   register Machine *gp;

    if ((gp=p->machine) == NULL)
    {	gp = (Machine *)malloc(sizeof(Machine));
	if (gp==NULL)
	{   XBell(dpy, 70);
	    fprintf(stderr, "Can't allocate!!\n");
	    return;
	}
	p->machine = gp;
	gp->next = machine_base;
	machine_base = gp;
	gp->name = p->preferred_name;
	if (gp->name == NULL)
	    gp->name = p->names->name;
	gp->host = p;
    }
    /* Default to natural size if width or height not given */
    if (!(WidthValue&p->geo_flag))
	p->width = XTextWidth(font_info, gp->name, strlen(gp->name)) + 4;
    if (!(HeightValue&p->geo_flag))
	p->height = font_info->ascent + 4;
    p->x1 = p->x_pos - (p->width/2);
    p->y1 = p->y_pos - (p->height/2);
    p->x2 = p->x1 + p->width;
    p->y2 = p->y1 + p->height;
    p->geo_flag |= AlreadyPlaced;
    PlaceTaps(p);
}
/****	Calculate placement of Network	****/

PlaceNetwork(p)
Network *p;
{   if (p->geo_flag&P_P_NET)
	PlaceP2Pnet(p);
    else
	PlaceFlatNet(p);
}


static
PlaceFlatNet(p)
register Network *p;
{   p->vertical = (p->height > p->width);
    p->x1 = p->x_pos - (p->width/2);
    p->y1 = p->y_pos - (p->height/2);
    p->x2 = p->x1 + p->width;
    p->y2 = p->y1 + p->height;
    /* Set up the bounding box for the net itself. */
    if (p->vertical)
    {	register int phase = 1;		/* 1=Draw on left, 0=Draw on right */
	register AddressList *ap;

	p->side1 = p->x1;
	p->side2 = p->x2;
	MAPCAR (ap, p->addresses)
	{   if (phase)
	    {	phase = 0;
		p->x1 -= 3;
	    }
	    else
	    {	phase = 1;
		p->x2 += 3;
	    }
	}
	p->label_x = p->x2+2;
	p->label_y = p->y_pos+(font_info->ascent/2);
    }
    else
    {	register int phase = 1;		/* 1=Draw on top, 0=Draw on bottom */
	register AddressList *ap;

	p->side1 = p->y1;
	p->side2 = p->y2;
	MAPCAR (ap, p->addresses)
	{   if (phase)
	    {	phase = 0;
		p->y1 -= 3;
	    }
	    else
	    {	phase = 1;
		p->y2 += 3;
	    }
	}
	p->label_x = p->x_pos-(XTextWidth(font_info, p->name, strlen(p->name))/2);
	p->label_y = p->y1-2;
    }
    p->geo_flag |= AlreadyPlaced;
}


static
PlaceP2Pnet(p)
register Network *p;
{   /* This assumes all hosts are loaded, need to find exactly 2 with addresses
     *	on this net...
     */
    Host *h1=NULL;
    Host *h2=NULL;
    Machine *tp;

    MAPCAR(tp,machine_base)
    {	AddressList *ap;
	MAPCAR(ap,tp->host->addresses)
	{   if (AddrInList_p(ap->address,p->addresses))
	    {	if (h1==NULL)
		{   h1 = tp->host;
		    break;
		}
		if (h2==NULL)
		{   h2 = tp->host;
		    break;
		}
		if (DBG_FILES_P)
		{   fprintf(stderr,
		       "Point-to-point network %s has more than two hosts.\n",
			    p->name);
		    fprintf(stderr,"  First three are: %s, %s, %s\n",
			    h1->preferred_name,
			    h2->preferred_name,
			    tp->host->preferred_name);
		}
		return;
	    }
	}
    }
    if (h2==NULL)
    {	if (DBG_FILES_P)
	    fprintf(stderr,
		    "Couldn't find two hosts on point-to-point net %s.\n",
		    p->name);
	return;
    }
    if ( (h1->geo_flag&AlreadyPlaced) && (h2->geo_flag&AlreadyPlaced) )
    {	p->x1 = h1->x_pos;
	p->y1 = h1->y_pos;
	p->x2 = h2->x_pos;
	p->y2 = h2->y_pos;
	p->geo_flag |= AlreadyPlaced;
    }
}
/****	Compute placement for a Title entry	****/


PlaceTitle(p)
register Title *p;
{   int w=XTextWidth(font_info, p->text, strlen(p->text));
    int h=font_info->ascent;

    p->text_x = p->x_pos - (w/2);
    p->text_y = p->y_pos + (h/2);
    /* Default to natural size if width or height not given */
    if (!(WidthValue&p->geo_flag))
	p->width = w + 4;
    if (!(HeightValue&p->geo_flag))
	p->height = h + 4;
    p->x1 = p->x_pos - (p->width/2);
    p->y1 = p->y_pos - (p->height/2);
    p->x2 = p->x1 + p->width;
    p->y2 = p->y1 + p->height;
    p->geo_flag |= AlreadyPlaced;
}
/****	Compute placement of the taps from a Host to the Networks	****/



PlaceTaps(p)
register Host *p;
{   register AddressList *ap;
    Network *FindNetwork();

    MAPCAR (ap, p->addresses)
    {   Network *np = FindNetwork(ap->address);
	int ok;
	if (np != NULL)
	{   if (np->geo_flag&P_P_NET)
		continue;
	    if (!(np->geo_flag&AlreadyPlaced)) PlaceNetwork(np);
	    if (np->vertical)
		ok = (p->y_pos >= np->y1+5 && p->y_pos <= np->y2-5);
	    else
		ok = (p->x_pos >= np->x1+5 && p->x_pos <= np->x2-5);
	    if (ok)
		ap->net_p = np;
	    else
	    {	if (DBG_FILES_P)
		{   AddressList *tp;
		    char c='(';
		    fprintf(stderr, "Tap from host %s to ",
			    p->preferred_name);
		    if (np->name==NULL  || *(np->name)=='\0' )
			printf("unnamed network ");
		    else
			printf("%s ",np->name);
		    MAPCAR (tp,np->addresses)
		    {	printf("%c",c);
			WriteAddress(stdout,tp->address,FALSE);
			c=',';
		    }
		    if (c=='(') printf("(No addresses");
		    printf(") cannot be aligned.\n");
		}
		ap->net_p = NULL;
	    }
	}
    }
}
/****	Draw a host	****/


DisplayHost(hp)
Host *hp;
{   if (! (hp->geo_flag&AlreadyPlaced))
	PlaceHost(hp);
    UpdateHost(hp, line_gc);
}


EraseHost(hp)
Host *hp;
{   /* Will eraseing the selected_host (to move) leave a blob? ???? */
    UpdateHost(hp, bg_gc);
}



static
UpdateHost(hp, pix)
Host *hp;
GC pix;
{   register int drawn = FALSE;
    register AddressList *ap;

    MAPCAR (ap, hp->addresses)
    {	if ( (show_chaos && ap->address.class==AF_CHAOS) ||
	     (show_IP && ap->address.class==AF_INET) )
	{   if (!drawn)
	    {	XFillRectangle(dpy, wind,
			       (hp == selected_host)?highlight_gc:bg_gc,
				hp->x1, hp->y1, hp->width, hp->height );
		XDrawRectangle(dpy, wind, pix,
			       hp->x1, hp->y1, hp->width, hp->height);
		XDrawString(dpy, wind,
			    (hp == selected_host)?highlight_fg_gc:line_gc,
			    hp->x1+2, hp->y2-2,
			    hp->machine->name, strlen(hp->machine->name) );
		drawn = TRUE;
	    }
	    if (ap->net_p != NULL && !(ap->net_p->geo_flag&P_P_NET))
	    {   Network *np = ap->net_p;
		short x[5], y[5];
		if (np->vertical)
		{   y[0] = y[1] = y[3] = hp->y_pos;
		    y[2] = hp->y_pos - 4;
		    y[4] = hp->y_pos + 4;
		    if (np->x_pos < hp->x1)
		    {	x[0] = hp->x1;
			x[3] = np->side2 + 8;
			x[1] = x[2] = x[4] = np->side2;
		    }
		    else
		    {	x[0] = hp->x2;
			x[3] = np->side1 - 8;
			x[1] = x[2] = x[4] = np->side1;
		    }
		}
		else
		{   x[0] = x[1] = x[3] = hp->x_pos;
		    x[2] = hp->x_pos - 4;
		    x[4] = hp->x_pos + 4;
		    if (np->y_pos < hp->y1)
		    {   y[0] = hp->y1;
			y[3] = np->side2 + 8;
			y[1] = y[2] = y[4] = np->side2;
		    }
		    else
		    {   y[0] = hp->y2;
			y[3] = np->side1 - 8;
			y[1] = y[2] = y[4] = np->side1;
		    }
		}
		XDrawLine(dpy, wind, line_gc, x[0], y[0], x[1], y[1] );
		XDrawLine(dpy, wind, line_gc, x[2], y[2], x[3], y[3] );
		XDrawLine(dpy, wind, line_gc, x[3], y[3], x[4], y[4] );
	    }
	}
    }
}


ReDrawHost(hp)
Host *hp;
{   if (hp->geo_flag&AlreadyPlaced)
	EraseHost(hp);
    DisplayHost(hp);
}
/****	Draw a Title	****/


DisplayTitle(tp)
Title *tp;
{   if (! (tp->geo_flag&AlreadyPlaced))
	PlaceTitle(tp);
    UpdateTitle(tp, line_gc);
}


EraseTitle(tp)
Title *tp;
{   UpdateTitle(tp, bg_gc);
}



static
UpdateTitle(tp, pix)
Title *tp;
GC pix;
{   XDrawString(dpy, wind, line_gc,
		tp->text_x, tp->text_y, tp->text, strlen(tp->text));
    if ( (WidthValue|HeightValue)&tp->geo_flag)
	XDrawRectangle(dpy, wind, pix,
		       tp->x1, tp->y1, tp->width, tp->height);
}


ReDrawTitle(tp)
Title *tp;
{   if (tp->geo_flag&AlreadyPlaced)
	EraseTitle(tp);
    DisplayTitle(tp);
}
/****	Draw a network	****/


DisplayNetwork(np)
register Network *np;
{   if (! (np->geo_flag&AlreadyPlaced))
	PlaceNetwork(np);
    UpdateNetwork(np, foreground);
}
    

EraseNetwork(np)
register Network *np;
{   UpdateNetwork(np, background);
}

static
UpdateNetwork(np, pix)
register Network *np;
int pix;
{   if (! (np->geo_flag&AlreadyPlaced))
	return;
    if (np->geo_flag&P_P_NET)
	UpdateP2Pnetwork(np, pix);
    else
	UpdateFlatNetwork(np, pix);
}


static
UpdateFlatNetwork(np, pix)
register Network *np;
int pix;
{   register int draw = FALSE;
    register AddressList *ap;
    int a, b, t;

    t = 3;
    a = np->side1 - t;
    b = np->side2;
    MAPCAR (ap, np->addresses)
    {   if ( (show_chaos && ap->address.class==AF_CHAOS) ||
	     (show_IP && ap->address.class==AF_INET) )
	{   XSetForeground(dpy, fill_gc,
			   (ap->address.class==AF_INET)?IP_pixel:CHAOS_pixel );
	    if (np->vertical)
		XFillRectangle(dpy, wind, fill_gc, a, np->y1, abs(t), np->height);
	    else
		XFillRectangle(dpy, wind, fill_gc, np->x1, a, np->width, abs(t));
	    {int tmp=a-t;a=b;b=tmp;}
	    t = -t;
	    draw = TRUE;
	}
    }
    if (draw||np->addresses==NULL)
    {   XSetForeground(dpy, fill_gc, np->media->pixel );
	if (np->vertical)
	    XFillRectangle(dpy, wind, fill_gc,
			   np->side1, np->y1, np->width, np->height);
	else
	    XFillRectangle(dpy, wind, fill_gc,
			   np->x1, np->side1, np->width, np->height);
	XDrawString(dpy, wind, line_gc,
		    np->label_x, np->label_y, np->name, strlen(np->name));
    }
}


static
UpdateP2Pnetwork(np, pix)
register Network *np;
int pix;
{   XGCValues vals;

    /* for now we only draw the "media" part (and always draw it). */
    vals.foreground = np->media->pixel;
    vals.line_width = np->width;
    XChangeGC(dpy, fill_gc, GCForeground|GCLineWidth, &vals );
    XDrawLine(dpy, wind, fill_gc, np->x1, np->y1, np->x2, np->y2 );
}



ReDrawNetwork(np)
Network *np;
{   if (np->geo_flag&AlreadyPlaced)
	EraseNetwork(np);
    DisplayNetwork(np);
    /* !!!!>>>	Should also find host taps that might need redrawing <<<!!!! */
}
/****	Maintenance of interaction area	****/


static
DisplayWait()
{   waiting = TRUE;
    XDefineCursor(dpy, wind, wait_cursor);
    XClearArea(dpy, wind, 0, display_height-(font_info->ascent+2), 0,0,FALSE);
    XDrawString(dpy, wind, line_gc,
		5, display_height-2,  waitmsg, strlen(waitmsg) );
    XFlush(dpy);		/* Anticipation!! */
}



static
DisplayPrompt()
{   waiting = FALSE;
    XDefineCursor(dpy, wind, norm_cursor);
    XClearArea(dpy, wind, 0, display_height-(font_info->ascent+2), 0,0,FALSE);
    XDrawString(dpy, wind, line_gc,
		5, display_height-2, prompt, strlen(prompt) );
}




static
DisplayInput()
{   XClearArea(dpy, wind,
	       XTextWidth(font_info, prompt, strlen(prompt))+9,
	       display_height-(font_info->ascent+2),
	       0, 0, FALSE );
    if (input_index != 0)
	XDrawString(dpy, wind, line_gc,
		    XTextWidth(font_info, prompt, strlen(prompt))+10,
		    display_height-2,
		    input, input_index);
}
/****	External representations.	****/


int
InterpretColor(cp, dflt)
char *cp;			/* Name for the color */
int dflt;			/* Default value if not available */
{   XColor color;

    if (DBG_COLOR_P)
	printf("Interpreting \"%s\" as a color (default is %d).",cp,dflt);
    switch ( DefaultVisual(dpy, screen)->class )
    {
    default:
	fprintf(stderr, "Don't understand Visual class %d.\n",
		DefaultVisual(dpy, screen)->class );
	fprintf(stderr, "Assuming Black and White display.\n");
    case StaticGray:
    case GrayScale:
	if (DBG_COLOR_P)
	    printf("  B&W display, returning default %d.",dflt);
	/* can't do any better with only gray scale */
	return (dflt);
    case StaticColor:
    case PseudoColor:
    case TrueColor:
    case DirectColor:
	break;
    }
    if (!XParseColor(dpy, DefaultColormap(dpy,screen), cp, &color))
    {   if (DBG_COLOR_P)
	    printf("  Not directly a color, look for mapping.");
	if ((cp = XGetDefault(dpy, Argv[0], cp)) == NULL)
	{   if (DBG_COLOR_P)
		printf("  No mapping, pick next default.");
	    if ((cp = default_color[default_color_index++]) == NULL)
	    {	if (DBG_COLOR_P)
		    printf("  No defaults left, give up (returning %d).\n",
			   dflt);
		default_color_index--;
		return(dflt);
	    }
	}
	if (DBG_COLOR_P)
	    printf("  Trying for color \"%s\".",cp);
	if (!XParseColor(dpy, DefaultColormap(dpy,screen), cp, &color))
	{   if (DBG_COLOR_P)
		printf("  Not a color either, give up (returning %d).\n",dflt);
	    return(dflt);
	}
    }
    if (DBG_COLOR_P)
	printf("  Mapped to r=%d, g=%d, b=%d.",
	       color.red,color.green,color.blue);
    if (!XAllocColor(dpy, DefaultColormap(dpy,screen), &color))
    {	if (DBG_COLOR_P)
	    printf("  This display can't do that, give up (returning %d).\n",
		   dflt);
	return(dflt);
    }
    if (DBG_COLOR_P)
	printf("  Using pixel value %d.\n",color.pixel);
    return(color.pixel);
}



fprintgeo(fd, flag, width, height, x_pos, y_pos)
FILE *fd;
int flag, width, height, x_pos, y_pos;
{   if (WidthValue&flag)
	fprintf(fd, "%d", width);
    if (HeightValue&flag)
	fprintf(fd, "x%d", height);
    fprintf(fd, "%c%d%c%d", (XNegative&flag)?'-':'+', abs(x_pos),
			    (YNegative&flag)?'-':'+', abs(y_pos));
}



int
pxl2rgb(p,r,g,b)
int p, *r, *g, *b;
{   XColor color;

    if (DBG_COLOR_P)
	printf("Turning pixel %d back into seperate colors, got ",p);
    color.pixel = p;
    if (!XQueryColor(dpy,DefaultColormap(dpy,screen),&color))
    {	if (DBG_COLOR_P)
	    printf("an error.\n");
	return (FALSE);
    }
    if (DBG_COLOR_P)
	printf("R=%d, G=%d, B=%d.\n",color.red,color.green,color.blue);
    *r = color.red;
    *g = color.green;
    *b = color.blue;
    return(TRUE);
}
/****	Background Loop for display interaction	****/


Display_BackgroundLoop()
{   XEvent event;

    DisplayPrompt();
    FOREVER
    {	if (need_redisplay && !XPending(dpy))
	    Redisplay();
	XNextEvent(dpy, &event);
	if (DBG_EVENTS_P)
	    fprintf(stderr,"===>Got an event, type #%d.\n",event.type);
	switch(event.type)
	{
	default:		/* Don't know what this is. */
	    /* Should probably only print this if some DEBUG flag on... */
	    fprintf(stderr,"Unknown event type %d.\n",event.type);

	    /* These are the ones we can't block, but don't care about. */
	case ReparentNotify:
	case MapNotify:
	case UnmapNotify:
	case MappingNotify:
	case GravityNotify:
	    break;

	case ConfigureNotify:
	    display_width = event.xconfigure.width;
	    display_height = event.xconfigure.height;
	    if (DBG_EVENTS_P)
		fprintf(stderr,"It's a Configure, new dimens=%dx%d.\n",
			display_width,display_height);
	    break;

	case Expose:
	    if (DBG_EVENTS_P)
		fprintf(stderr,"It's an Exposure!  (Yow!  Are we over-exposed yet?)\n");
	    need_redisplay = TRUE;
	    break;

	case ButtonPress:
	    HandleMouseButton(&event);
	    break;

	case KeyPress:
	    HandleKey(&event);
	    break;

	}
    }
}
/**** input handling ****/



static
HandleMouseButton(event_p)
XButtonEvent *event_p;
{   
    if (DBG_EVENTS_P)
	fprintf(stderr,"It's button 0x%X (with modifiers 0x%X)\n",
		event_p->button,event_p->state);
    DisplayWait();
    switch (event_p->button)
    {
    case Button1:
	Describe(event_p->x, event_p->y, event_p->state);
	break;

    case Button2:
	Adjust(event_p->x, event_p->y, event_p->state);
	break;

    case Button3:
	XBell(dpy, 00);
	break;

    default:
	fprintf(stderr, "Don't get it: MouseButton (button=0x%X).\n",
		event_p->button);
	break;
    }
    DisplayPrompt();
    DisplayInput();
}
/* Describe(x, y, flags)
 *
 * Locate an object in the data base given display coordinates and describe it.
 */
static
Describe(x, y, flags)
int x, y;
int flags;
{   Machine *gp;
    Network *np;
    Title *tp;
    Machine *LocateMachine();
    Network *LocateNetwork();
    Title *LocateTitle();

    flags = ( (flags&ShiftMask)?MOD_VERBOSE:0 )
	  | ( (flags&ControlMask)?MOD_GEOMETRY:0 )  ;
    DeselectHost();
    printf("----------------------------------\n");
    if ( (gp=LocateMachine(x,y)) != NULL)
    {   SelectHost(gp->host);
	DescribeMachine(gp, flags);
    }
    else if ( (np=LocateNetwork(x,y)) != NULL)
    {	DescribeNetwork(np, flags);
    }
    else if ( (tp=LocateTitle(x,y)) != NULL)
    {	HandleInput(tp->command);
    }
    else
    {	fprintf(stderr, "Can't figure out what you're pointing at.\n");
	XBell(dpy, 00);
    }
}

static Machine *
LocateMachine(x, y)
int x, y;
{   Machine *gp;

    MAPCAR (gp, machine_base)
    {	Host *hp = gp->host;

	if ((x >= hp->x1) && (x <= hp->x2) && (y >= hp->y1) && (y <= hp->y2))
	    return (gp);
    }
    return (NULL);
}



/* Returns true if the three integers are ordered (i.e. the second is between
 * the first and third).  Either increasing order or decreasing.
 */
ordered_p(i,j,k)
int i,j,k;
{   if (i<j)
	return(j<k);
    else
	return(j>k);
}


/* Computes square of distance from point (ax,ay)
 *	to line segment (ax1,ay1)(ax2,ay2).
 *  Only works if reasonably close (otherwise gets overflow)
 */
int
dist2(x,y,x1,y1,x2,y2)
int x,y,x1,y1,x2,y2;
{   int xd=x2-x1;		/* x diff */
    int yd=y2-y1;		/* y diff */
    int n=(x1*y2+x2*y+x*y1-y1*x2-y2*x-y*x1);
    int ans = n*n/(xd*xd+yd*yd);
    return(ans);
}

static Network *
LocateNetwork(x, y)
int x, y;
{   Network *np;

    MAPCAR (np, net_base)
    {	if (ordered_p(np->x1, x, np->x2) &&
	    ordered_p(np->y1, y, np->y2)  )
	{   if (np->geo_flag&P_P_NET)
	    {   int d=dist2(x,y,np->x1,np->y1,np->x2,np->y2);
		if (d<(np->width)*(np->width)/2)
		    return (np);
	    }
	    else
		return (np);
	}
    }
    return(NULL);
}

static Title *
LocateTitle(x, y)
int x, y;
{   Title *p;

    MAPCAR (p, title_base)
    {	if ((p->command != NULL) &&
	    (x >= p->x1) && (x <= p->x2) && (y >= p->y1) && (y <= p->y2))
	    return (p);
    }
    return(NULL);
}
/**** Screen modification ****/


static
Adjust(x, y, flags)
int x, y;
int flags;
{   Machine *gp, *LocateMachine();
    Network *np, *LocateNetwork();

    switch (flags)
    {
    default:
	XBell(dpy, 00);
	return;
    case 0:
	if (selected_host==NULL)
	{   XBell(dpy, 00);
	    fprintf(stderr, "No host to place.\n");
	    return;
	}
	AddHost(x, y, selected_host);
	return;
    case ShiftMask:
	DeselectHost();
	if ( (gp=LocateMachine(x,y)) != NULL)
	{   DeleteMachine(gp);
	    return;
	}
	if ( (np=LocateNetwork(x,y)) != NULL)
	{   DeleteNetwork(np);
	    return;
	}
	printf("Can't figure out what you want me to delete.\n");
    }
}


static
AddHost(x, y, hp)
int x, y;
Host *hp;
{   Machine *gp;
    AddressList *ap;
    Network *np;

    if ((gp=hp->machine) == NULL)
    {	gp = (Machine *)malloc(sizeof(Machine));
	if (gp==NULL)
	{   XBell(dpy, 70);
	    fprintf(stderr, "Can't allocate!!\n");
	    return;
	}
	hp->machine = gp;
	gp->next = machine_base;
	machine_base = gp;
	gp->name = hp->preferred_name;
	if (gp->name == NULL)
	    gp->name = hp->names->name;
	gp->host = hp;
    }
    else
    {	XClearWindow(dpy, wind);
	need_redisplay = TRUE;
    }
    hp->geo_flag = XValue | YValue;
    hp->x_pos = x;
    hp->y_pos = y;
    PlaceHost(hp);
    MAPCAR(ap,hp->addresses)
	if ( (np=ap->net_p) != NULL &&
	     (np->geo_flag&P_P_NET) )
	    PlaceP2Pnet(np);
    if (!need_redisplay)
	DisplayHost(hp);
}


int
DeleteMachine(gp)
Machine *gp;
{   Machine *p, *op;

    /* First entry is a special case */
    if ( (p=machine_base) == NULL) return (FALSE);
    if (gp==p)
    {	machine_base = gp->next;
	if (gp->host != NULL)
	    gp->host->machine = NULL;
	/* FreeMachine(gp); */
	XClearWindow(dpy, wind);
	need_redisplay = TRUE;
	return (TRUE);
    }
    for (op=p,p=op->next; p!=NULL; op=p,p=op->next)
    {	if (gp==p)
	{   op->next = gp->next;
	if (gp->host != NULL)
	    gp->host->machine = NULL;
	    /* FreeMachine(gp); */
	    XClearWindow(dpy, wind);
	    need_redisplay = TRUE;
	    return (TRUE);
	}
    }
    return (FALSE);
}



DeleteNetwork(np)
Network *np;
{   Network *p, *op;
    Host *hp;
    AddressList *ap;

    /* First entry is a special case */
    if ( (p=net_base) == NULL) return;
    if (np==p)
    	net_base = np->next;
    else
	for (op=p,p=op->next; p!=NULL; op=p,p=op->next)
	{   if (np==p)
	    {	op->next = np->next;
		break;
	    }
	}
    MAPCAR(hp,host_base)
    	MAPCAR(ap,hp->addresses)
	    if (ap->net_p == np)
		ap->net_p = NULL;
    /* FreeNetwork(np); */
    XClearWindow(dpy, wind);
    need_redisplay = TRUE;
    return;
}
/**** Keyboard input, buffer it up. ****/



static
HandleKey(event_p)
XKeyEvent *event_p;
{   char buff[INPUT_LEN];
    KeySym keysym;
    int n;

    n = XLookupString(event_p, buff, INPUT_LEN, &keysym, NULL);
    if (DBG_EVENTS_P)
    {	buff[n] = 0;
	fprintf(stderr,"Keycode 0x%X => keysym 0x%X",event_p->keycode,keysym);
	fprintf(stderr," (string \"%s\", %d chars)",
		(buff[0]>=' '&&buff[0]<='~')?buff:"??",n);
	fprintf(stderr,", modifiers 0x%X\n",event_p->state);
    }
    if (IsModifierKey(keysym))
	return;
    /* Handle normal characters */
    if (n>0 && (event_p->state&~(ShiftMask|LockMask)) == 0 &&
			buff[0] >= ' ' && buff[0] <= '~')
    {   if (input_index+n > INPUT_LEN)
	{   XBell(dpy, 00);
	    return;
	}
	bcopy(buff,&input[input_index],n);
	input_index += n;
	DisplayInput();
	return;
    }
    switch (keysym)
    {
    case XK_Help:
	DisplayWait();
	if (input_index==0)
	    HelpOn("");
	else
	{   input[input_index] = '\0';
	    HelpOn(input);
	}
	DisplayPrompt();
	DisplayInput();
	return;
    case XK_Linefeed:
    case XK_Return:
    case XK_KP_Enter:
    case XK_Execute:
	if (input_index != 0)
	    input[input_index] = '\0';
	DisplayWait();
	HandleInput(input);
	input_index = 0;
	DisplayPrompt();
	DisplayInput();
	return;
    case XK_u:    case XK_U:
	if ( !(event_p->state&ControlMask) )
	{   XBell(dpy, 00);
	    return;
	}
    /* Should be a case for "Clear input" like keysyms. */
	input_index = 0;
	DisplayInput();
	return;
    case XK_l:    case XK_L:
    case XK_w:    case XK_W:
	if ( !(event_p->state&ControlMask) )
	{   XBell(dpy, 00);
	    return;
	}
    case XK_Clear:
	ForceRedisplay();
	return;
    case XK_Delete:
    case XK_BackSpace:
	if (--input_index >= 0)
	{   DisplayInput();
	    return;
	}
	input_index = 0;
	DisplayInput();
	XBell(dpy, 00);
	return;

	/* Don't beep about this one, shouldn't it be MOD1, not multi? */
    case XK_Multi_key:
	return;

    default:
	if (DBG_EVENTS_P)
	    fprintf(stderr,"Barfing over keycode 0x%X\n",keysym);
	XBell(dpy, 00);
	return;
    }
}
