#ifndef _JSysTime_H
#define _JSysTime_H


/*!
 \class JSysTime JSysTime.h
 \brief encapsulation for struct timeval

*/

#include <sys/time.h>
#include "assert.h"
#include "jam/JTime.h"

class ostream;

class JSysTime:public timeval
{
public:
  
  JSysTime(){;}       /// Careful here
  JTime timeFrom(const JSysTime &from);
  void  subtract(const JTime &t);
  void  setFrom(const JSysTime &start,JTime when);
};



inline
void 
JSysTime::setFrom(const JSysTime &start,JTime when)
{
  tv_sec  =
    when.glitchs()/JTime::GLITCHPERSEC + start.tv_sec;
  
  tv_usec = 
    (1000000/JTime::GLITCHPERSEC)*(when.glitchs()%JTime::GLITCHPERSEC) +
    start.tv_usec ;
  
  tv_sec += tv_usec/1000000;
  tv_usec = tv_usec%1000000;
  
  assert(tv_sec >= 0);
  assert(tv_usec >= 0);
}


ostream & operator << (ostream &out, const JSysTime &t);


/*!
 \class JSysInterval JSysTime.h
 \brief encapsulation of struct itimer

*/


class JSysInterval:public itimerval
{
public:

  ///
  JSysInterval(const JSysTime &now,const JSysTime &start,const JTime &when);

  JSysInterval(const JTime val);

  JSysInterval(const JTime val,const JTime inter);

  double realInterval() const;

  static JSysInterval   & smallest() { return _smallest;}
private:
  ///
  static JSysInterval   _smallest;

};


inline
double
JSysInterval::realInterval() const
{
  return  ((double)(it_value.tv_sec)) + ((double)(it_value.tv_usec))/1000000.0;
}

inline
bool operator >= (const JSysTime &a,const JSysTime &b)
{
  if ( a.tv_sec > b.tv_sec ) return true;
  if ( a.tv_sec < b.tv_sec ) return false;
  if ( a.tv_usec < b.tv_usec ) return false;
  return true;
}

ostream & operator << (ostream &out, const JSysInterval &t);

inline
JSysInterval::JSysInterval(const JTime val)
{
  it_value.tv_sec=val.glitchs()/JTime::GLITCHPERSEC;
  it_value.tv_usec=
    (val.glitchs()%JTime::GLITCHPERSEC)*1000;
  
  it_interval.tv_sec=0;
  it_interval.tv_usec=0;
}


inline
JSysInterval::JSysInterval(const JTime val,const JTime inter)
{
  it_value.tv_sec=val.glitchs()/JTime::GLITCHPERSEC;
  it_value.tv_usec=
    (val.glitchs()%JTime::GLITCHPERSEC)*1000;
  
  it_interval.tv_sec=inter.glitchs()/JTime::GLITCHPERSEC;
  it_interval.tv_usec=
    (inter.glitchs()%JTime::GLITCHPERSEC)*1000;
  
}



inline
JSysInterval::JSysInterval(const JSysTime &now,
			   const JSysTime &start,
			   const JTime &when)
{ 

  it_interval.tv_sec=0;
  it_interval.tv_usec=0;

  long vtv_sec  =
    when.glitchs()/JTime::GLITCHPERSEC + start.tv_sec - now.tv_sec-1;

  long vtv_usec = 
    (1000000/JTime::GLITCHPERSEC)*(when.glitchs()%JTime::GLITCHPERSEC) +
    start.tv_usec - now.tv_usec +1000000;

  vtv_sec += vtv_usec/1000000;
  
  if (vtv_sec < 0 )
    {
      it_value.tv_sec =0;
      it_value.tv_usec=1;
    }
  else if (vtv_sec == 0 && vtv_usec == 0)
    {
      it_value.tv_sec =0;
      it_value.tv_usec=1;
    }
  else
    {
      vtv_usec = vtv_usec%1000000;
      
      it_value.tv_sec =vtv_sec;
      it_value.tv_usec=vtv_usec;
    }
  
} 
  

inline
void
JSysTime::subtract(const JTime &t)
{ 
  tv_usec = tv_usec + 1000000 - (1000000/JTime::GLITCHPERSEC)*(t.glitchs()%JTime::GLITCHPERSEC);
  tv_sec  = tv_sec  -1 + tv_usec/1000000 - t.glitchs()/JTime::GLITCHPERSEC;
  tv_usec  =  tv_usec%1000000;
} 

inline
JTime
JSysTime::timeFrom(const JSysTime &start)
{
  return JTime::GLITCHPERSEC*(tv_sec-start.tv_sec) + 
    (tv_usec-start.tv_usec)/(1000000/JTime::GLITCHPERSEC); 
}

#endif









