
/*
*   this file contains the symbol table management routines:
*     struct symbol* install (name,scope,type) 
*     struct symbol* mnew     (name)
*     struct symbol* lookup  (name)
*     char*          strsave (name)
*                    kill_scope (scope)
*                    free_node  (symptr) 
*                    hash       (name)
*                    newscope ()
*/
#include <stdio.h>
#include "defs.h"

struct symbol *htab[HASHSIZE] = {0};	/* hash table */

static struct str {			/* string table */
	char *str;			/* string */
	struct str *next;		/* next entry */
} *stab[HASHSIZE] = {0};

static struct symbol *freelist = NULL;	/* free list of unused entries */

/*
*   remove all entries of the scope from the symbol table
*/
kill_scope (scope)
int scope;
{
	register int i;
	register struct symbol *p, **q, *r;

	for (i = 0; i < HASHSIZE; i++) {
		q = &htab[i];
		for (p = htab[i]; p; p = r) {
			r = p->next;
                        if (p->scope == scope){
				*q = p->next;
				free_node (p);
                                }
			else
				q = &p->next;
			}
		}
}

/* install - allocate a new entry for name and install it */
struct symbol *install(name,scope,type)
char *name;
int scope,type;
{
	struct symbol *p;

	p = mnew(name);
	p->next = htab[hash(name)];
	p->type = type;
	p->scope = scope;
	htab[hash(name)] = p;
	return (p);
}

/* lookup - lookup name, return pointer to entry */
/* WARNING: it can only find symbol if name is a ptr returned by strsave */
struct symbol *lookup(name)
char *name;
{
	struct symbol *p;

	for (p = htab[hash(name)]; p != NULL ; p = p->next)
		if (name == p->name)
			return (p);
	return (NULL);
}

/* mnew - allocate and initialize and new symbol table entry for name */
struct symbol *mnew(name)
char *name;
{
	struct symbol *p;
	char *emalloc();

	if (p = freelist)
		freelist = p->next;
	else
		p = (struct symbol *) emalloc(sizeof (struct symbol));
	p->name = name;
	p->type = mktype(T_FLOAT, T_SCALAR);
	p->scope = S_GLOBAL;
/*
	p->defined = p->offset = 0;
	p->list = NULL;
*/
	return (p);
}

/* strsave - lookup str and install if necessary, return pointer */
char *strsave(char *str)
{
	int h, len;
	char *s, *strcpy();
	struct str *p;
	char *emalloc();

	h = 0;
	for (p = stab[hash(str)]; p != NULL ; p = p->next)
		if (strcmp(str, p->str) == 0)
			return (p->str);
	p = (struct str *) emalloc(sizeof(struct str));
	p->str = (char *)malloc(strlen(str)+1);
	strcpy(p->str, str);
	p->next = stab[hash(str)];
	stab[hash(str)] = p;
	return (p->str);
}

/* free_node  - free storage  for reuse  */
/* is very closely connected to mnew()    */
/* TBD:  only allow a maximum freelist length */
free_node (p)
struct symbol *p;
{
	if (p==NULL) {
	  i_warning (" free_node was called with NULL ptr ");
	  return;
	}

        if (freelist==NULL)    freelist = p;
        else {
		p->next = freelist;
		freelist = p;
	}

}  /*   free_node */



/* dname - return string representation of type, shape, or scope */
char *dname(x)
int x;
{
	static struct tname {
		int val;
		char *name;
	} tnames[] = { T_INT, "int", T_FLOAT, "float", 
		T_COND, "conditional", T_SCALAR, "", T_FUNC, "()",
		T_ARRAY, "[]", S_GLOBAL, "global", S_PARAM, "parameter",
		S_LOCAL, "local", 0, 0 };
	static char buf[30];
	int i;

	for (i = 0; tnames[i].name; i++)
		if (tnames[i].val == x)
			return (tnames[i].name);
	sprintf(buf, "<%d>", x);
	return (buf);
}

/* dump - print entire symbol table or one entry */
dump(p, fp)
struct symbol *p;
FILE *fp;
{
	int i;

	if (fp == NULL)
		fp = stderr;
	if (p == NULL)
		for (i = 0; i < HASHSIZE; i++)
			for (p = htab[i]; p; p = p->next)
				dump(p, fp);
	else {
		fprintf(fp, "%s ", dname(xtype(p->type)));
/*
		fprintf(fp, "%s%s, %sscope=%s, offset=%d\n", p->name,
			dname(xshape(p->type)),	p->defined?"defined, ":"",
			dname(p->scope), p->offset);
*/
		}
}




/*  this routine has the error checking for malloc built in */
char *emalloc (i)
int i;
{
	char *s;
        char *malloc ();

	s =  malloc (i);
	if (s==NULL) {
		sys_error ("system out of memory");
		exit (1);
	}
	else    return s;
}


/*
* returns an index to a hash bucket
*/
hash (s)
char *s;
{
	int i=0;

	while (*s) {
		i =  (((unsigned) *s+i) % HASHSIZE);
		s++;
	}
	return i;
}

