// crossfader.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/
#ifdef __GNUG__
#pragma implementation
#endif

#include "application.h"
#include "controller.h"
#include "crossfader.h"
#include "curvegen.h"
#include "data.h"
#include "editor.h"
#include "envelope.h"
#include "localdefs.h"
#include "query.h"
#include "request.h"

Combiner::Combiner(Data* target, Data* src)
		: DataModifier(target), theSource(src) {
	theSource->ref();
}

Combiner::~Combiner() {
	Resource::unref(theSource);
}

boolean
Combiner::confirmValues(Controller *) {
	boolean warning = false;
	char msg[128];
	// warn user about format mismatches
	if(target()->channels() != source()->channels()) {
		sprintf(msg, "Warning:  source channels (%d) != target channels (%d)",
			source()->channels(), target()->channels());
		warning = true;
	}
	else if(target()->sRate() != source()->sRate()) {
		sprintf(msg, "Warning:  source samp rate (%d) != target samp rate (%d)",
			source()->sRate(), target()->sRate());
		warning = true;
	}
	return (!warning ||
		Application::confirm(msg, "Continue anyway?", nil, Cancel) == Yes);
}

//********

Mixer::Mixer(Data* target, Data* source) : Combiner(target, source) {
	initialize();
}

Modifier *
Mixer::create(DataEditor*) {
	return nil;
}

Mixer::~Mixer() {}

const QueryInfo *
Mixer::requestInfo() { return nil; }

boolean
Mixer::setValues(Request &) { return true; }

int
Mixer::doApply(Data* data) {
	data->add(source());
	return true;
}

//********

Replacer::Replacer(Data* target, Data* source) : Combiner(target, source) {
	initialize();
}

Modifier *
Replacer::create(DataEditor*) {
	return nil;
}

Replacer::~Replacer() {}

const QueryInfo *
Replacer::requestInfo() { return nil; }

boolean
Replacer::setValues(Request &) { return true; }

int
Replacer::doApply(Data* data) {
	return data->replaceWith(source());
}

//********

Splicer::Splicer(Data* target, Data* source) : Combiner(target, source) {
	initialize();
}

Modifier *
Splicer::create(DataEditor*) {
	return nil;
}

Splicer::~Splicer() {}

const QueryInfo *
Splicer::requestInfo() { return nil; }

boolean
Splicer::setValues(Request &) { return true; }

int
Splicer::doApply(Data* data) {
	data->deferUpdate(true);			// dont do update twice
	int status = (data->spliceIn(source()->frameRange())
		&& data->replaceWith(source()));
	data->deferUpdate(false);
	return status;
}

//********

Crossfader::Crossfader(Data* target, Data* source, Envelope *evp, double amp)
		: Combiner(target, source), envelope(evp), gain(amp),
		  reverseEnvelope(false) {
	envelope->ref();
	initialize();
}

Modifier *
Crossfader::create(DataEditor*) {
	return nil;
}

Crossfader::~Crossfader() {
	Resource::unref(envelope);
}

int
Crossfader::doApply(Data* data) {
	data->crossfade(source(), envelope);
	return true;
}

const QueryInfo *
Crossfader::requestInfo() {
	static QueryLabelInfo labels[] = {
		{ "Crossfade Selected Region: " }, { nil }
	};
	static QueryValueInfo values[] = {
		{ "Amplitude Factor: ", "1.0", CharCheck::numsOnly },
		{ nil }
	};
	static QueryChoiceInfo choices[] = {
		{ "Envelope for Crossfade:", "|Linear|Exponential|File on Disk|",
			Linear, true },
		{ "Envelope Direction:", "|Forward|Reverse|", Forward, true },
		{ nil }
	};
	static QueryInfo info[] = {
		{ labels, "", values, choices },
		{ nil }
	};
	return info;
}

boolean
Crossfader::setValues(Request &request) {
	QueryValue v;
	request.retrieveValues(v);
	gain = v;
	static const int nChoices = 2;
	QueryChoice choices[nChoices];
	request.retrieveChoices(choices, nChoices);
	switch(choices[0]) {
		case Linear:
			envelope = new Envelope;
			envelope->ref();
			{
			LinearCurveGenerator curve(envelope, 0.0, 1.0);
			curve.apply();
			}
			break;
		case Exponential:
			envelope = new Envelope;
			envelope->ref();
			{
			ExponentialCurveGenerator curve(envelope, 0.0, 1.0, 2.0);
			curve.apply();
			}
			break;
		case OnDisk:
			break;
	}
	reverseEnvelope = (choices[1] == Reverse);
	return true;
}

boolean
Crossfader::confirmValues(Controller *c) {
	boolean status = Super::confirmValues(c);
	if(status) {
		Controller* newController = nil;
		if(envelope == nil) {
			if((newController = c->editor()->openNewFile(
					"Select Envelope for Crossfade:")) != nil) {
				Data* source = newController->model();
				source->ref();
				envelope = new Envelope(source->length());
				envelope->ref();
				envelope->copyFrom(source);
				Resource::unref(source);
			}
			else
				status = false;
		}
		if(envelope && reverseEnvelope)
			envelope->reverse();
		}
	return status;
}
