// fileselector.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


// Specialized InterViews FileChooser class which loads its filename into a
// QueryValue.  Uses pathname passed in in the QueryValue unless the global
// parameter BrowserUseLastPath is set to true, in which case it uses whatever
// was the last opened path.

#ifdef __GNUG__
#pragma implementation
#endif

#include <sys/param.h>

#include <InterViews/box.h>
#include <InterViews/button.h>
#include <InterViews/event.h>
#include <InterViews/filebrowser.h>
#include <InterViews/frame.h>
#include <InterViews/glue.h>
#include <InterViews/streditor.h>

#include "application.h"
#include "eventforwarder.h"
#include "fileselector.h"
#include "query.h"

// NOTE:  remember to implement these on browser:
//      void SetDirectoryTextFilter(const char*)
//      void SetModeFilter(int)
//      void SetDirectoryModeFilter(int)
//		boolean Acceptable(const char*) -- this needs to be redefined

char FileSelector::savedDir[MAXPATHLEN+1];

FileSelector::FileSelector(ButtonState *state, QueryValue *path,
	const char* suffixes, int rows, int cols)
	: FileChooser(
		state,
		Application::globalResourceIsTrue("BrowserUseLastPath") ?
			retrieveDirectory() : (const char *) *path,
		rows, cols, Center
	  ),
	    myValue(path), forwarder(nil) {
	initialize(suffixes);
}

FileSelector::~FileSelector() {
	Resource::unref(forwarder);
}

void
FileSelector::initialize(const char* suffixes) {
	SetClassName("FileSelector");
	title = subtitle = nil;						// I dont use these
	Insert(createInterior());
	browser()->SetTextFilter(strlen(suffixes) == 0 ? nil : suffixes);
}

// all this is copied from base class impl. simply because the Forward()
// method was not declared to be virtual!

boolean
FileSelector::doAccept() {
    int v = NullResponse;
    state->SetValue(v);
    _focus = _sedit;

    HandleFocus();
    state->GetValue(v);
    while (v == NullResponse || v == '\t') {
        if (v == '\t') {
            UpdateEditor();
            UpdateBrowser();
            SwitchFocus();
            HandleFocus();

        } else {
    		Event e;
            Read(e);
            if (e.target != _focus && CanFocus(e.target)) {
                SwitchFocus();
            }
			if(forwarder != nil)
				forwarder->handle(e);
			else
				Forward(e);
        }
        state->GetValue(v);
    }

    boolean accepted = (v == '\r' || v == Yes);

    if (accepted) {
        UpdateEditor();
        UpdateBrowser();
    }
    return accepted;
}

boolean
FileSelector::Accept() {
    boolean accepted, dirSelected;

    do {
        accepted = doAccept();
        dirSelected = browser()->IsADirectory(Choice());
    } while (accepted && dirSelected);

	if(accepted) {
		storeDirectory(browser()->GetDirectory());	
		setValue();
	}
    return accepted;
}

boolean
FileSelector::setValue() { return value()->set(Choice()); }

void
FileSelector::setForwarder(EventForwarder* ef) {
	forwarder = ef;
	forwarder->ref();
}

Interactor* 
FileSelector::createInterior() {
    const int space = round(.5*cm);
    return new MarginFrame(
        new VBox(
            new Frame(new MarginFrame(_sedit, 2)),
            new VGlue(space, 0),
            new Frame(AddScroller(browser())),
            new VGlue(space, 0)
         ), space, space/2, 0
    );
}

void
FileSelector::storeDirectory(const char* dir) {
	strcpy(savedDir, dir);
}

const char *
FileSelector::retrieveDirectory() {
	if(*savedDir == '\0')
		storeDirectory(".");
	return savedDir;
}
