// framedwindow.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include <InterViews/bitmap.h>
#include <InterViews/border.h>
#include <InterViews/box.h>
#include <InterViews/canvas.h>
#include <InterViews/cursor.h>
#include <InterViews/event.h>
#include <InterViews/glue.h>
#include <InterViews/style.h>
#include <InterViews/window.h>
#include <InterViews/world.h>
#include <IV-X11/xwindow.h>

#include "localdefs.h"
#include "application.h"
#include "framedwindow.h"
#include "menus.h"
#include <mxv.bitmap>

const int	FramedWindow::viewStartingX = 50;
const int	FramedWindow::viewStartingY = 300;
const int	FramedWindow::viewXIncrement = 30;
const int	FramedWindow::viewYIncrement = 50;
World *		FramedWindow::globalWorld = nil;
Bitmap*		FramedWindow::globalIconBitmap = nil;
Coord		FramedWindow::currentViewX = viewStartingX;
Coord		FramedWindow::currentViewY = viewStartingY;
boolean		FramedWindow::placeWindows = false;

// interior musr be nil here so as to allow Insert() call from derived class,

FramedWindow::FramedWindow(const char* title, Interactor* interior, int border)
		: Frame(title, nil, border), windowname(nil), isBusy(false) {
	init();
	storeName(title);
}

// redefining Interactor::Wrap lets me accrete all the additional interactors
// around the specific 'insides', and these other interactors are common to all
// FramedWindow subclasses

Interactor *
FramedWindow::Wrap(Interactor* insides) {
	menus = makeMenuBar();
	VBox *interior = new VBox(
		new HBox(menus, new HGlue),
		new HBorder,
		insides
	);
	return interior;
}

void
FramedWindow::init() {
	SetClassName("FramedWindow");
	placeWindows = Application::globalResourceIsTrue("AutoPlaceWindows");
	if(globalIconBitmap == nil) {
		globalIconBitmap = new Bitmap(mxv_bits, mxv_width, mxv_height);
	}
	globalIconBitmap->ref();
}

FramedWindow::~FramedWindow() {
	delete [] windowname;
	Resource::unref(globalIconBitmap);
}

void
FramedWindow::Reconfig() {
	GetTopLevelWindow()->icon_bitmap(globalIconBitmap);
	GetTopLevelWindow()->wm_delete(getWMDeleteHandler());
	Frame::Reconfig();
	SetName(windowname);
}

void
FramedWindow::storeName(const char *name) {
	delete [] windowname;
	windowname = newstr(name);
}

void
FramedWindow::changeName(const char *name) {
	if(windowname != nil && !strcmp(name, windowname))
		return;
	storeName(name);
	SetName(windowname);
}

Coord
FramedWindow::viewXLocation() {
	return currentViewX;
}

Coord
FramedWindow::viewYLocation() {
	World *w = getWorld();
	return w ? w->Height() - currentViewY : currentViewY;
}

void
FramedWindow::incrementViewLocation() {
	currentViewX += viewXIncrement;
	currentViewY += viewYIncrement;
	World *w = getWorld();
	if(w) {
		// wrap to keep on screen
		if(currentViewX > int(w->Width()) - 500)
			currentViewX = viewStartingX;
		if(currentViewY > int(w->Height()) - 20)
			currentViewY = viewStartingY;
	}
}

World *
FramedWindow::getWorld() {
	World *w = GetWorld();
	return w ? w : globalWorld;
}

boolean
FramedWindow::checkForControlKeyEvent() {
	long uSecInterval = 10000;		// .01 seconds
	Event e;
	static const char delChar = 127;
	static const char bSpaceChar = 8;
	if(Read(0L, uSecInterval, e)) {
		if(e.eventType == KeyEvent && e.control_is_down())
			return (*e.keystring == delChar || *e.keystring == bSpaceChar);
	}
	return false;
}

void
FramedWindow::display(World *world) {
	if(placeWindows) {
		world->InsertApplication(this, viewXLocation(), viewYLocation());
		incrementViewLocation();
	}
	else
		world->Insert(this);
	setWorld(world);	
}

void
FramedWindow::unDisplay() {
	World* w = GetWorld();
	if(w)
		w->Remove(this);
}

void
FramedWindow::makeVisible() {
	ManagedWindow *top = GetTopLevelWindow();
	if(top)
		top->deiconify();
}

boolean
FramedWindow::isMapped() {
	return (canvas != nil && canvas->status() == Canvas::mapped);
}

void
FramedWindow::SetName(const char *name) {
	ManagedWindow *top = GetTopLevelWindow();
	if(top != nil) {
		top->style()->attribute("name", name);
		top->style()->attribute("title", name);
		top->style()->attribute("iconName", name);
		ManagedWindowRep* mwRep = top->rep();
		mwRep->do_set(top, &ManagedWindowRep::set_name);
		mwRep->do_set(top, &ManagedWindowRep::set_icon_name);
	}
}

void
FramedWindow::busy(boolean bsy) {
	SetCursor(bsy ? hourglass : defaultCursor);
	isBusy = bsy;
}

Control *
FramedWindow::makeMenu(Menu *menu, MenuInfo *item) {
	for(MenuInfo* i = item; i->str != nil; i++) {
		boolean enabled = (i->inactive == false);
		Control* menuItem = (i->submenuInfo != nil) ?
			makeMenu(new PullrightCommandMenu(i->str, enabled), i->submenuInfo)
			: new Command(i->str, i->keystr, makeMenuItemAction(i), enabled);
		menu->Include(menuItem);
	}
	return menu;
}

void
FramedWindow::closeAndDestroy() {
	unDisplay();
	Resource::unref_deferred(this);
}
