// header.h

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


// This class contains the interface, implementation and state common to all
// Header subclasses.  A Header object is responsible for reading, parsing, and
// validating the on-disk header information for each data file, and well as
// performing the reverse operation at save time.  Header subclass instances
// are created within the Data subclass objects, and in the current implemen-
// tation, they only exist during the read and write operations.  In future
// versions, they may be kept as a permenant member.

#ifndef HEADER_H
#ifdef __GNUG__
#pragma interface
#endif
#define HEADER_H

#include <InterViews/resource.h>
#include "localdefs.h"
#include "requester.h"

class Comment;
class Controller;
class DataFile;
class Data;
class Request;

class Header : public Resource {
public:
	static int readMagicNumber(DataFile *);
public:
	Header(int magic=0, DataType type=NoData, int chans=0) 
		: magic_number(magic), data_offset(0), data_size(0), data_type(type),
		nchans(chans), comment(nil), raw(false) {
	}
	virtual ~Header();
	virtual boolean configure(Controller *);
	virtual int read(DataFile *);
	virtual int write(DataFile *);
	virtual int magic() const { return magic_number; }
	int dataOffset() const { return data_offset; }
	DataType dataType() const { return data_type; }
	int dataSize() const { return data_size; }
	int nChans() const { return nchans; }
	void setComment(const char *com);
	void setComment(Comment *com);
	Comment *getComment() { return comment; }
	void setDataSize(int size) { data_size = size; }
protected:
	virtual boolean isLittleEndian() { return false; }
	boolean needsByteSwapping();
	void setDataOffset(int offset) { data_offset = offset; }
	virtual int seekInFile(DataFile*, int);
	int sampleSize();
	boolean isRaw() { return raw; }
	void setRaw() { raw = true; }
	virtual int diskHeaderSize() { return 0; }
	virtual int diskHeaderCommentSize() { return 0; }
	virtual int diskHeaderInfoSize() {
		return diskHeaderSize() - diskHeaderCommentSize();
	}
	int checkMagic(DataFile *);
	virtual int readMagic(DataFile *);
	virtual boolean isMagic() { return true; }
	virtual int bytesPerSecond() { return 0; }
	virtual int checkHeader() { return true; }
	virtual int readInfo(DataFile *) { return true; }
	virtual int readComment(DataFile *);
	virtual int writeInfo(DataFile *) { return true; }
	virtual int writeComment(DataFile *);
	virtual int commentLength();
	virtual const char* magicError() { return ""; }
protected:
	class ConfigRequester : public Requester {
	public:
		ConfigRequester(Header* header);
		virtual ~ConfigRequester() {}
	protected:
		Header* myHeader;
	private:
		redefined const QueryInfo *requestInfo() { return nil; }
	};
	friend ConfigRequester;
	virtual Requester* configRequester() { return nil; }
protected:
	static const int defaultSampRate;
	int magic_number;
	int data_offset;
	int data_size;
	DataType data_type;
	int nchans;
	Comment *comment;
private:
	boolean raw;		// if raw is true, header will not be read from file
	void initialize();
};


// abstract "intermediate" base class

class FrameDataHeader : public Header {
public:
	virtual int sampleRate() { return samprate; }
	virtual double frameRate() { return framerate; }
	virtual double duration() { return dur; }
protected:
	FrameDataHeader(int magic, DataType type) : Header(magic, type, 0),
		framerate(0), samprate(0), dur(0) {}
	FrameDataHeader(int magic, DataType type, int chans, double frate,
			int srate, double duration=0)
		: Header(magic, type, chans),
		framerate(frate), samprate(srate), dur(duration) {}
protected:
	class FrameConfigRequester : public Header::ConfigRequester {
	public:
		FrameConfigRequester(Header* header) : ConfigRequester(header) {}
	protected:
		redefined void configureRequest(Request &);
	};
	double framerate;
	int samprate;
	double dur;
};

class LPCHeader : public FrameDataHeader {
	typedef FrameDataHeader Super;
	enum { LP_MAGIC = 999 };
public:		// class portion
	enum Type { None=0x1, With=0x2 };
	enum Poles { defaultPoles = 34, maxPoles = 64 };
	static Type defaultHeaderType() { return default_HeaderType; }
	static void setDefaultHeaderType(Type t) { default_HeaderType = t; }
public:		// object portion
	LPCHeader(int poles=defaultPoles, double framert=200, int srate=44100,
                        double dur=0);
	virtual ~LPCHeader() {}
	virtual int nPoles() { return npoles; }
	static int readMagicNumber(DataFile *);
protected:
	redefined int diskHeaderSize() { return sizeof(DiskHeader); }
	redefined int diskHeaderCommentSize() { return 4; }
	redefined int readMagic(DataFile *);
	redefined boolean isMagic() { return magic() == LP_MAGIC; }
	redefined int checkHeader();
	redefined int readInfo(DataFile *);
	redefined int writeInfo(DataFile *);
	redefined const char* magicError();
protected:
	class LPCConfigRequester : public FrameConfigRequester {
	public:
		LPCConfigRequester(Header* header) : FrameConfigRequester(header) {}
	protected:
		redefined void configureRequest(Request &);
    	redefined boolean setValues(Request &);
	};
	friend boolean LPCConfigRequester::setValues(Request &);
public:
	redefined Requester* configRequester() {
		return new LPCConfigRequester(this);
	}
private:	// class members
	static Type default_HeaderType;
private:
	class DiskHeader;
	friend DiskHeader;
	// The structure of the LPC header on disk
	class DiskHeader {
	public:
		DiskHeader(int h=0, int m=LP_MAGIC, int np=0, int nv=0,
			float fr=0, float sr=0, float dur=0) : headersize(h), lpmagic(m),
			npoles(np), nvals(nv), framrate(fr), srate(sr), duration(dur) {}
		int	headersize,
			lpmagic,
			npoles,	// number of filter pole coefficients per frame
			nvals;	// total number of channels of information
		float	framrate,	// number of frames per second
			srate,		// sampling rate of orig data
			duration;	// duration of orig data analyzed
		char	text[4];
	};
	int npoles;
};

class PCHHeader : public FrameDataHeader {
	typedef FrameDataHeader Super;
	enum { PCH_MAGIC = 900 };
public:
	PCHHeader() : FrameDataHeader(PCH_MAGIC, FloatData, 2, 0, 0, 0),
		framesize(0) {}
	PCHHeader(double framert=200, int fsize=350, int srate=44100,
			double duration=0)
		: FrameDataHeader(PCH_MAGIC, FloatData, 2, framert, srate,
		duration), framesize(fsize) {}
	virtual ~PCHHeader() {}
	virtual int frameSize() { return framesize; }
protected:
	redefined int readMagic(DataFile *);
	redefined boolean isMagic() { return true; }
	redefined int readInfo(DataFile *);
private:
	int framesize;
};

class FFTHeader : public FrameDataHeader {
	typedef FrameDataHeader Super;
	enum { FFT_MAGIC = 8888 };
	enum Points { fftMinPoints = 64 };
public:
	FFTHeader(int npoints, double frmrate=200,
			int srate=defaultSampRate, double duration=0)
		: FrameDataHeader(FFT_MAGIC, DoubleData, npoints,
			frmrate, srate,  duration) {}
	virtual ~FFTHeader() {}
	int nPoints() { return nChans(); }
protected:
	redefined int diskHeaderSize() { return sizeof(DiskHeader); }
	redefined int diskHeaderCommentSize() { return 4; }
	redefined boolean isMagic() { return magic() == FFT_MAGIC; }
	redefined int readInfo(DataFile *);
	redefined int checkHeader();
	redefined int writeInfo(DataFile *);
	redefined const char* magicError();
protected:
	class FFTConfigRequester : public FrameConfigRequester {
	public:
		FFTConfigRequester(Header* header) : FrameConfigRequester(header) {}
	protected:
		redefined void configureRequest(Request &);
    	redefined boolean setValues(Request &);
	};
	friend boolean FFTConfigRequester::setValues(Request &);
public:
	redefined Requester* configRequester() {
		return new FFTConfigRequester(this);
	}
private:
	class DiskHeader;
	friend DiskHeader;
	// This is the struct for the FFT header on disk
	class DiskHeader {
	public:
		DiskHeader(int m=FFT_MAGIC, int hs=0, int np=0, float sr=0)
			: fftmagic(m), headersize(hs), npoints(np), srate(sr) {}
		int	fftmagic,
			headersize,
			npoints;		// number of points in fft
		float	srate; 		// sampling rate of orig data
		char	text[4];
	};
};

class EnvelopeHeader : public Header {
	typedef Header Super;
	enum { EVP_MAGIC = 777 };
	enum Points { evpMinPoints = 8 };
public:		// class portion
	enum Type { None=0x1, With=0x2 };
	static Type defaultHeaderType() { return default_HeaderType; }
	static void setDefaultHeaderType(Type t) { default_HeaderType = t; }
public:		// object portion
	EnvelopeHeader() : Header(EVP_MAGIC, FloatData, 1),
		samprate(0), npoints(0) {}
	EnvelopeHeader(int npts=evpMinPoints, int srate=defaultSampRate)
		: Header(EVP_MAGIC, FloatData, 1), samprate(srate),
		npoints(npts) {}
	virtual ~EnvelopeHeader() {}
	virtual int nPoints() { return npoints; }
	virtual int sampleRate() { return samprate; }
protected:
	redefined int diskHeaderSize() { return sizeof(DiskHeader); }
	redefined int diskHeaderCommentSize() { return 4; }
	redefined boolean isMagic() { return magic() == EVP_MAGIC; }
	redefined int readInfo(DataFile *);
	redefined int checkHeader();
	redefined int writeInfo(DataFile *);
	redefined const char* magicError();
protected:
	class EvpConfigRequester : public Header::ConfigRequester {
	public:
		EvpConfigRequester(Header* header) : ConfigRequester(header) {}
	protected:
		redefined void configureRequest(Request &);
	};
public:
	redefined Requester* configRequester() {
		return new EvpConfigRequester(this);
	}
private:
	class DiskHeader;
	friend DiskHeader;
	// The struct for the envelope header on disk
	class DiskHeader {
	public:
		DiskHeader(int m=EVP_MAGIC, int hs=0, int np=0, float sr=0)
			: evpmagic(m), headersize(hs), npoints(np), srate(sr) {}
		int	evpmagic,
			headersize,
			npoints;	// number of points in evp
		float	srate; 		// sampling rate of orig data
		char	text[4];
	};
	int samprate;
	int npoints;
private:
	static Type default_HeaderType;
};

#endif
