// next_converter.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include <InterViews/action.h>
#include "application.h"
#include "next_dac.h"
#include "typeconvert.h"
#include "next_sound.h"
#include "statusaction.h"

SNDSoundStruct NextConverter::internalSound;
int NextConverter::myTag = 0;

NextConverter::NextConverter() {
}

NextConverter::~NextConverter() {
	stop();
}

int
NextConverter::currentPlayLevel() {
	int left = -1, dummy;
	SNDGetVolume(&left, &dummy);
	return round((left - 1)/42.0 * 100);		// 1 - 43 -> 0 - 100
}

int
NextConverter::currentRecordLevel() {
	return 100;									// not adjustable
}

int
NextConverter::setPlayLevel(int volume) {
	int left = round(42 * volume/100.) + 1;		// 0 - 100 -> 1 - 43
	int right = left;
	return (SNDSetVolume(left, right) == SND_ERR_NONE);
}

int
NextConverter::setRecordLevel(int volume) {
	return true;								// not adjustable
}

int
NextConverter::setSpeakerOutput(boolean toOn) {
	return (SNDSetMute(int(toOn)) == SND_ERR_NONE);
}

boolean
NextConverter::isPlayableFormat(DataType type) {
	return (type == ShortData || type == MuLawData);
}

int
NextConverter::tag() { return myTag; }

int
NextConverter::doConversion(StatusAction* askedToStop) {
	int err;
	myTag = 1;
//	Application::inform("Playing...");
	if((err = SNDStartPlaying(&internalSound, myTag, 0, true,
			start_fun, end_fun)) != SND_ERR_NONE) {
		Application::alert(SNDSoundError(err));
		fail();
		return false;
	}
	if((*askedToStop)())
		return stop();
	return true;
}

int
NextConverter::doRecording(StatusAction* askedToStop) {
	int err;
	myTag = 1;
	if((err = SNDStartRecording(&internalSound, myTag, 0, true,
			start_fun, end_fun)) != SND_ERR_NONE) {
		Application::alert(SNDSoundError(err));
		fail();
		return false;
	}
	if((*askedToStop)())
		return stop();
	return true;
}

int
NextConverter::pause() {
	return true;
}

int
NextConverter::resume() {
	return true;
}

int
NextConverter::stop() {
	if(tag() > 0) {
		SNDStop(tag());
		Super::stop();
		myTag = 0;	// to avoid duplicate stops
	}
	return true;
}

int
NextConverter::doConfigure() {
	return createInternalSound(sampleRate(), channels(), dataType());
}

int
NextConverter::checkDataType(DataType type) {
	if(type != MuLawData && type != ShortData) {
		Application::alert("Only MuLaw and short integer sounds may be played or recorded.");
		return false;
	}
	return true;
}

int
NextConverter::checkSampleRate(int rate) {
	if(dataType() == MuLawData && rate != 8012) {
		Application::alert("Sample rate for MuLaw files must be 8012");
		return false;
	}
	return true;
}

int
NextConverter::checkChannels(int chans) {
	int status = true;
	if(chans != 1 && chans != 2) {
		Application::alert("Only monaural and stereo files may be played or recorded.");
		status = false;
	}
	else if(dataType() == MuLawData && chans != 1) {
		Application::alert("MuLaw files must be monaural for this operation.");
		status = false;
	}
	return status;
}

boolean
NextConverter::ready() const {
	return good();
}

int
NextConverter::createInternalSound(int samprate, int channels, DataType type) {
	SNDSoundStruct *snd;
	int err;
	if((err = SNDAlloc(&snd, 1, type_to_format(type), 
			(samprate < 30000) ? ((samprate < 10000) ?
			SND_RATE_CODEC : SND_RATE_LOW) : SND_RATE_HIGH,
			channels, 4)) != SND_ERR_NONE) {
		Application::alert(SNDSoundError(err));
		fail();
		return false;
	}
	// copy data from new header to static header
	bcopy((char *) snd, (char *) &internalSound, sizeof(internalSound));

	// set data offset to beginning of sample data
	internalSound.dataLocation = (char *) pointerToData() - (char *) &internalSound;
	internalSound.dataSize = dataSize();
	SNDFree(snd);	/* and free the temporary sound struct */
	return 1;
}

int
NextConverter::start_fun(SNDSoundStruct *sp, int tag, int err) {
	myTag = (err == SND_ERR_NONE) ? 1 : -1;
	return 1;
}

int
NextConverter::end_fun(SNDSoundStruct *sp, int tag, int err) {
	myTag = (err == SND_ERR_NONE) ? 0 : -1;
	if(recording())
		doRecordDoneAction();
	setStatus((err == SND_ERR_NONE) ? Stopped : Error);
//	Application::inform();
	return 1;
}
