// pitchtracker.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/

#ifdef __GNUG_
#pragma implementation
#endif

#include "application.h"
#include "pitchtracker.h"
#include "pchdata.h"
#include "query.h"
#include "request.h"

PitchTracker::PitchTracker(Data* output, const Range &freqs, int framesize,
		int frameoffset) : ArrayFunction(output, framesize, frameoffset),
		freqRange(freqs) {
	initialize();
}

const QueryInfo *
PitchTracker::requestInfo() {
	static QueryLabelInfo labels[] = {
		{ "Pitch Track Analysis of Selected Region:" },
		{ nil }
	};
	static QueryValueInfo values[] = {
		{ "Frame size (samples):", "350", CharCheck::posIntsOnly },
		{ "Frame offset (samples):", "200", CharCheck::posIntsOnly },
		{ "Frame rate (Hz):", "0", CharCheck::posNumsOnly },
		{ "High Freq. Boundry (Hz):", "1000", CharCheck::posNumsOnly },
		{ "Low Freq. Boundry (Hz):", "100", CharCheck::posNumsOnly },
		{ nil }
	};
	static QueryInfo info[] = {
		{ labels, "Set framesize to 0 to have it auto-sized.", values },
		{ nil }
	};
	return info;
}

boolean 
PitchTracker::setValues(Request& request) {
	const int nVals = 5;
	QueryValue v[nVals];
	request.retrieveValues(v, nVals);
	int framesize = v[0];
	int offset = v[1];
	double lowFreq = v[4];
	double hiFreq = v[3];
	framesize = (framesize == 0) ? round(sampRate() / lowFreq) : framesize;
	if(offset >= framesize) {	// only true for pitch tracking
		char msg[128];
		sprintf(msg, "Frame offset (%d) must be less than framesize (%d).",
			offset, framesize);
		Application::alert(msg);
		return false;
	}	
	double framerate = v[2];
	int status = true;
	if((status = setBaseValues(framesize, framerate, offset)) == true)
		freqRange.set(lowFreq, hiFreq).check();
	return status;
}

boolean
PitchTracker::confirmValues(Controller *) {
	boolean status = true;
	int neededFrameSize = round(1.0 / freqRange.min() * sampRate());
	if(framesize() < neededFrameSize) {
		char msg[128];
		sprintf(msg,
			"To track frequencies down to %0.0f hz, a frame size of %d is needed.",
			freqRange.min(), neededFrameSize);
		status = Application::confirm("For Your Information:", msg, "Continue anyway?");
	}
	return status;
}

void
PitchTracker::initialize() {
	JMAX = framesize()/10;
	MM = (JMAX+1)/2;
	for(int i=0; i<50; ++i) {
		freq[i] = freqRange.min() + i * freqRange.spread()/50.;
		int nt = int(nyquist() / (10.0 * freq[i]));
		int n = (nt < 5 ? nt : 5);
		double omega = (freq[i]*TWO_PI) / (sampRate()/10.0);
		trigpo(omega,tphi[i],tpsi[i],tgamps[i],tgamph[i],n);
	}
	Super::initialize();
	setAnalysis(
		new PCHData(analysisLength(), sampRate(), framerate(),  framesize())
	);
}

int
PitchTracker::doApply(Data *data) {
	boolean was = Data::deferRescan(true);	// no need to do updating on copy
	Data* copy = data->copyOf();
	copy->ref();
	PitchTrackFilter filter(copy);	// apply lowpass filter first
	filter.apply();
	int status = Super::doApply(copy);
	Resource::unref(copy);
	Data::deferRescan(was);
	return status;
}

int
PitchTracker::operator () (double *input, Data *frame) {
	frame->set(getPitch(input), 0, 0);
	frame->set(getRMS(input), 0, 1);
	return true;
}

double 
PitchTracker::getRMS(double *sig) {
	int frameSize = framesize();
	double rms = 0.0 ;
	register double *s = sig;
	for (int step = 0; step < frameSize; ++step) {
		rms += (*s) * (*s);
		s++;
	}
	return (sqrt (rms / double(frameSize)));
}

double 
PitchTracker::getPitch(double *array) {
	float s[maxFrameSize/10], g[MAXMM], h[MAXMM];
	for(int j=0; j < JMAX; ++j)
		s[j] = array[10*(j+1)-1];
	double qsum = 0.0;
	for(int i=0; i < MM; ++i) {
		float gval = g[i] = .5 * ( s[MM+i-1] - s[MM-i-1] );
		float hval = h[i] = .5 * ( s[MM+i-1] + s[MM-i-1] );
		qsum += (gval*gval) + (hval*hval);
	}
	return search(qsum, g, h);
}

void
PitchTracker::trigpo(double omega, PA5MAX phi, PA6MAX psi,
		float *gampsi, float *gamphi, int n) {
	int j=0;
	int k,np;
	double alpha,beta,gamma;
	double wcos[MAXMM],wsin[MAXMM];
	double p,z,a,b,yy;

	np=n+1;
	for (k=0 ; k<MM ; ++k) {
		yy = omega* float(k);
		wcos[k]=cos(yy);
		wsin[k]=sin(yy);
	}
	beta=gamma= 0.0;
	for (k=0 ; k<MM ; ++k) {
		p=wsin[k];
		z=p*p;
		beta+=z*wcos[k];
		gamma+=z;
		phi[0][k]=p;
	}
	gamphi[0]=gamma;
	a=2.0 * beta/gamma;
	alpha=beta=gamma=0.0;
	for (k=0 ; k<MM ; ++k) {
		p= (2.0 * wcos[k]-a) * phi[0][k];
		alpha += wcos[k] * p * phi[0][k];
		beta += wcos[k]* ( p * p ) ;
		gamma += ( p * p );
		phi[1][k] = p ;
	}
	gamphi[1]=gamma;
	a = 2.0 * beta/gamma;
	b = 2.0 *alpha/gamphi[0];
	for (j=2; j<n; ++j) {
		alpha=beta=gamma = 0.0;
		for (k=0 ; k< MM ; ++k) {
			p = (2.0 * wcos[k] - a) * phi[j-1][k] - b * phi[j-2][k];
			alpha += wcos[k] * p * phi[j-1][k];
			beta += wcos[k] * (p * p);
			gamma += (p * p) ;
			phi[j][k] = p ;
		}
		gamphi[j] = gamma;
		a = 2.0 * beta/gamma;
		b = 2.0 *alpha/gamphi[j-1];
	}
	beta = 0.0 ;
	gamma = double(MM);
	for ( k=0; k < MM ; ++k) {
		beta += wcos[k];
		psi[0][k]= 1.0;
	}
	gampsi[0]=gamma;
	a=beta/gamma;
	alpha=beta=gamma= 0.0;
	for ( k=0 ; k < MM ; ++k) {
		p = wcos[k]-a;
		alpha += wcos[k] * p*psi[0][k];
		beta += wcos[k] * (p * p);
		gamma += (p * p);
		psi[1][k] = p ;
	}
	gampsi[1] = gamma ;
	a = 2.0 * beta / gamma ;
	b = 2.0 * alpha / gampsi[0];
	for (j=2 ; j<np ;++j) {
		alpha=beta=gamma= 0.0;
		for ( k=0; k < MM ; ++k) {
			p=(2.0 * wcos[k]-a)* psi[j-1][k]-b*psi[j-2][k];
			alpha += wcos[k]*p*psi[j-1][k];
			beta += wcos[k]* (p*p);
			gamma += (p*p);
			psi[j][k] = p;
		}
		gampsi[j]=gamma;
		a=2.0*beta/gamma;
		b=2.0*alpha/gampsi[j-1];
	}
}

double
PitchTracker::search(double qsum, float *g, float *h) {
	float funmin; 
	float fun[50];
	float f1,f2,f3,x1,x2,x3,a,b,c,ftemp;
	int istar=0,i;

	filfun(qsum, fun, g, h);
	funmin = fun[istar];
	for (i = 1; i < 50; ++i) {
		ftemp = fun[i];
		if(ftemp < funmin) {
			funmin = ftemp;
			istar = i;
		}
	}
	if(istar == 0 || istar == 49) {
		return freq[istar];
	}
	else {
		x1 = freq[istar-1];
		f1 = fun[istar-1];
		x2 = freq[istar];
		f2 = fun[istar];
		x3 = freq[istar+1];
		f3 = fun[istar+1];
		a = f3/((x3-x1)*(x3-x2));
		b = f2/((x2-x1)*(x2-x3));
		c = f1/((x1-x2)*(x1-x3));
		return .5*(a*(x1+x2)+b*(x1+x3)+c*(x2+x3))/(a+b+c);
	}
}

void 
PitchTracker::filfun(double qsum, float *fun, float *g, float *h) {
	float c,sum;
	int i,n,np,j,k;

	for(i=0 ; i < 50 ;++i) {
		n = int((nyquist()/10.)/freq[i]);
		n = (n < 5) ? n : 5;
		np = n+1;
		sum = 0.0;
		for(j=0 ; j < n ; ++j) {
			c = 0.0;
			for(k=0 ; k< MM ; ++k)  
				c += g[k] * tphi[i][j][k];
			sum += (c*c)/tgamph[i][j];
		}
		for (j=0; j < np; ++j) {
			c = 0.0;
			for (k=0; k < MM; ++k) 
				c += h[k] * tpsi[i][j][k];
			sum += (c*c)/tgamps[i][j];
		}
		fun[i] = qsum -sum;
	}
}

PitchTrackFilter::PitchTrackFilter(Data *d) : SimpleFunction(d) {
	initialize();
}

void
PitchTrackFilter::initialize() {
	w1 = w11 = w12 = w2 = w21 = w22 = w3 = w31 = w32 = w4 = w41
		= w42 = 0.0;
}

double 
PitchTrackFilter::operator () (double x) {
	static const double c= .00048175311, a1= -1.89919924, c1= -1.923248041,
			d1= .985720370, a2= -1.86607670, c2= -1.90075003,
			d2= .948444690, a3= -1.66423461, c3= -1.87516686,
			d3= .896241023, c4= -.930449120;
	double temp,y;
	
	w1=c*x-c1*w11-d1*w12;
	temp=w1+a1*w11+w12;
	w12=w11;
	w11=w1;
	w2=temp-c2*w21-d2*w22;
	temp=w2+a2*w21+w22;
	w22=w21;
	w21=w2;
	w3=temp-c3*w31-d3*w32;
	temp=w3+a3*w31+w32;
	w32=w31;
	w31=w3;
	w4=temp-c4*w41;
	y=w4+w41;
	/* w42 set but not used in lowpass */
	w42=w41;
	w41=w4;
	return y;
}

