// sgi_dac.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "sgi_dac.h"
#include "application.h"
#include "statusaction.h"
#include "typeconvert.h"
#include <math.h>
#include <signal.h>
#include <errno.h>
#include <unistd.h>

#undef DEBUG_SGI

char *
SGIConverter::errorStrings[] = {
	"not implemented yet",
	"trying to use an invalid port",
	"trying to use an invalid configuration",
	"trying to use an invalid device",
	"unable to access the device",
	"illegal direction given for port",
	"not able to allocate a port",
	"invalid sample width",
	"an illegal state has occurred",
	"invalid queue size",
	"invalid fillpoint",
	"null buffer pointer",
	"negative count",
	"param/val buffer doesn't make sense",
	"negative buffer length",
	"odd length parameter/value buffer",
	"invalid channel specifier",
	"invalid parameter",
	"attempt to set invalid sample format",
	"invalid sample rate token",
	"invalid size for sample read/write",
	"invalid size for floatmax",
	NULL
};

SGIConverter::SGIConverter() : audio_port(nil) {
	// need to determine whether audio is in use.
	pvbuf[0] = AL_OUTPUT_COUNT;
	pvbuf[2] = AL_MONITOR_CTL;
	if(ALgetparams(AL_DEFAULT_DEVICE, pvbuf, 4) < 0) {
		if(oserror() == AL_BAD_DEVICE_ACCESS) {
			Application::alert("SGIConverter(): could not access audio hardware.");
			fail();
		}
	}
	else if(pvbuf[1] != 0) {
		Application::alert("SGIConverter(): audio hardware currently in use.");
		fail();
	}
	else if(initialize())
		catchSignals(true);
	else
		fail();
}

SGIConverter::~SGIConverter() {
	resetDefaults();
	if(audio_port)
		ALcloseport(audio_port);
	catchSignals(false);
}

int
SGIConverter::displayError() {
	int er = oserror();
#ifdef DEBUG_SGI
	fprintf(stderr, "Error # is %d error string is %s\n", er, errorStrings[er]);
#endif
	Application::alert(errorStrings[oserror()]);
	return false;
}

int
SGIConverter::initialize() {
	BUG("SGIConverter::initialize()");
	ALseterrorhandler(nil);				// SGIConverter will handle errors
	audio_port_config = ALnewconfig();
	if(!audio_port_config)
		return displayError();
	else
		return getDefaults();
}

long
SGIConverter::getParameter(long param) {
	pvbuf[0] = param;
	ALgetparams(AL_DEFAULT_DEVICE, pvbuf, 2);
	return pvbuf[1];
}

int
SGIConverter::setParameter(long param, long value) {
	pvbuf[0] = param;
	pvbuf[1] = value;
	return (ALsetparams(AL_DEFAULT_DEVICE, pvbuf, 2) == 0) ? true : displayError();
}

int
SGIConverter::getDefaults() {
	return true;
}

void
SGIConverter::resetDefaults() {
}

boolean
SGIConverter::isPlayableFormat(DataType type) {
	return (type != MuLawData);
}

int
SGIConverter::checkDataType(DataType type) {
	long format = (type == FloatData) ? AL_SAMPFMT_FLOAT
		: (type == DoubleData) ? AL_SAMPFMT_DOUBLE : AL_SAMPFMT_TWOSCOMP;
	if(ALsetsampfmt(audio_port_config, format) < 0 ||
			ALsetwidth(audio_port_config, type_to_sampsize(type)) < 0)
		return displayError();
	else if(type >= FloatData)
		ALsetfloatmax(audio_port_config, peakAmplitude());
	return true;
}

int
SGIConverter::checkSampleRate(int sampleRate) {
	if(willRecord()) {
		// if we are using the digital inputs, the rate cannot be set.
		// we can only check for a match
		pvbuf[0] = AL_INPUT_RATE;
		pvbuf[2] = AL_INPUT_SOURCE;
		pvbuf[4] = AL_DIGITAL_INPUT_RATE;
		ALgetparams(AL_DEFAULT_DEVICE, pvbuf, 6);
		if(pvbuf[1] == AL_RATE_AES_1 || pvbuf[3] == AL_INPUT_DIGITAL) {
			// we are using digital input -- just check for match
			if(ALgetdefault(AL_DEFAULT_DEVICE, AL_DIGITAL_INPUT_RATE) >= 0) {
				if(pvbuf[5] != sampleRate) {
					char msg[128];
					sprintf(msg, "Digital input sample rate (%ld) != sound sample rate (%d)", pvbuf[5], sampleRate);
					Application::alert(msg);
					return false;
				}
			}
			else {
				Application::alert("Unable to determine digital input sample rate.");
				return false;
			}
		}
	}
	// if non-digital input, or playing sound, set the sample rate
	return setParameter(
		willRecord() ? AL_INPUT_RATE : AL_OUTPUT_RATE, long(sampleRate)
	);
}

int
SGIConverter::checkChannels(int chans) {
	if(chans == 3 || chans > 4) {
		Application::alert("Only mono, stereo, or quad files may be played.");
		return false;
	}
	ALsetchannels(audio_port_config, chans);
	return true;
}

int
SGIConverter::doConfigure() {
	BUG("SGIConverter::doConfigure()");
#ifdef DEBUG_SGI
	fprintf(stderr, "Queue size set to %ld.\n", bestQueueSize());
#endif
	if(ALsetqueuesize(audio_port_config, bestQueueSize()) < 0)
		return displayError();
	if(audio_port) ALcloseport(audio_port);
	if((audio_port = ALopenport("mxv", willRecord() ? "r" : "w",
			audio_port_config)) == 0) {
		int err = oserror();
		if(err == AL_BAD_NO_PORTS)
			Application::alert("No audio ports currently available.");
		else if(err == AL_BAD_OUT_OF_MEM)
			Application::alert("Not enough memory to open audio port.");
		else
			Application::alert("ALopenport() failed.");
		return false;
	}
	return true;
}

// read and write 1/10 second of sound for all formats, so queue is twice
// that size, or 1/5 second

long
SGIConverter::bestQueueSize() {
	return round(float(channels()) * sampleRate() / 5.0);
}

long
SGIConverter::writeSamps() { return bestQueueSize()/2; }

long
SGIConverter::readSamps() { return bestQueueSize()/2; }

inline long min(long x, long y) { return (x < y) ? x : y; }

int
SGIConverter::doConversion(StatusAction* askedToStop) {
	BUG("SGIConverter::doConversion()");
	lockMemory();
	addr databuffer = addr(pointerToData());
	int sampsize = sampleSize();
	long nsamps = dataSize()/sampsize;		// total samples to write
	long bufsamps = writeSamps();			// size of each write in samples
	bufsamps = min(bufsamps, nsamps);
	long bufsize = bufsamps * sampsize;		// size of each write in bytes
#ifdef DEBUG_SGI
	fprintf(stderr, "Playing %ld samps = %ld bytes\n", nsamps, nsamps * sampsize);
	fprintf(stderr, "Buffer is %ld samps = %ld bytes\n", bufsamps, bufsize);
	fprintf(stderr, "Starting conversion...\n");
#endif
	Application::inform("Playing...");
	while(nsamps > 0) {
		if((*askedToStop)())
			return stop();
		if(ALgetfillable(audio_port) < bufsamps)
			continue;
		if(ALwritesamps(audio_port, databuffer, bufsamps) < 0)
			return error("Error in ALwritesamps().");
		databuffer += bufsize;
		nsamps -= bufsamps;
		bufsamps = min(bufsamps, nsamps);
#ifdef DEBUG_SGI
		fprintf(stderr, "\t%ld samps written, nsamps= %ld\n", bufsamps, nsamps);
#endif
	}
	waitForStop(askedToStop);
	return stop();
}

int
SGIConverter::doRecording(StatusAction* askedToStop) {
	int sampsize = sampleSize();
	long nsamps = (dataSize()/sampsize);		// total samples to read
	long bufsamps = readSamps();				// size of each read in samples
	bufsamps = min(bufsamps, nsamps);
	long bufsize = bufsamps * sampsize;		// size of each read in bytes
#ifdef DEBUG_SGI
	fprintf(stderr, "Recording %ld samps = %ld bytes\n", nsamps, nsamps * sampsize);
	fprintf(stderr, "Buffer is %ld samps = %ld bytes\n", bufsamps, bufsize);
#endif
	lockMemory();
	char *databuffer = (char *) pointerToData();
	Application::inform("Recording...");
	while(nsamps > 0) {
		if((*askedToStop)())
			return stop();
		if(ALreadsamps(audio_port, databuffer, bufsamps) < 0)
			return error("Error in ALreadsamps().");
		databuffer += bufsize;
		nsamps -= bufsamps;
#ifdef DEBUG_SGI
		fprintf(stderr, "       %ld samps read, nsamps= %ld\n", bufsamps, nsamps);
#endif
		bufsamps = min(bufsamps, nsamps);
	}
#ifdef DEBUG_SGI
		fprintf(stderr, "At bottom, calling stop()\n");
#endif
	stop();
	return true;
}

SGIConverter::InputSource
SGIConverter::currentInputSource() {
	long source = getParameter(AL_INPUT_SOURCE);
	return (source == AL_INPUT_LINE) ? Line
		: (source == AL_INPUT_MIC) ? Mike :
		Digital;
}

int
SGIConverter::setInputSource(InputSource source) {
	return setParameter(AL_INPUT_SOURCE,
		(source == Mike) ? AL_INPUT_MIC
		: (source == Line) ? AL_INPUT_LINE
		: AL_INPUT_DIGITAL
	);
}

static double log2(double val) {
	static double l_10_two = log10(2.0);
	return log10(val) / l_10_two;
}

int
SGIConverter::currentPlayLevel() {
	long param = getParameter(AL_LEFT_SPEAKER_GAIN);
	return round(log2((double)param + 1) / 8.0 * 100);
}

int
SGIConverter::currentRecordLevel() {
	long param = getParameter(AL_LEFT_INPUT_ATTEN);
#ifdef DEBUG_SGI
	fprintf(stderr, "getParameter returns %ld for record\n", param);
#endif
	return max(0, round(100 - (10 * param)/16.0));
}

int
SGIConverter::setPlayLevel(int volume){
	long gain = round(pow(256.0, volume/100.0) - 1);
	int status = setParameter(AL_LEFT_SPEAKER_GAIN, gain);
	return status && setParameter(AL_RIGHT_SPEAKER_GAIN, gain);
}

int
SGIConverter::setRecordLevel(int volume){
	long gain = 16.0 * ((100 - volume)/10.0);
#ifdef DEBUG_SGI
	fprintf(stderr, "setting record parameter to %ld from volume of %d\n", gain, volume);
#endif
	int status = setParameter(AL_LEFT_INPUT_ATTEN, gain);
	return status && setParameter(AL_RIGHT_INPUT_ATTEN, gain);
}

int
SGIConverter::setSpeakerOutput(boolean isOn) {
	return setParameter(AL_SPEAKER_MUTE_CTL,
		isOn ? AL_SPEAKER_MUTE_OFF : AL_SPEAKER_MUTE_ON);
}

int
SGIConverter::pause() {
	return false;
}

int
SGIConverter::resume() {
	return false;
}

int
SGIConverter::stop() {
	BUG("SGIConverter::stop()");
	int status = true;
	if(running()) {
		unlockMemory();
		resetDefaults();
		status = Super::stop();
	}
	return status;
}

int
SGIConverter::lockMemory() {
	// fix memory in RAM to avoid swapping
	return (mpin((void *) pointerToData(), unsigned(dataSize())) == 0);	
}

int
SGIConverter::unlockMemory() {
	// let it swap
	return (munpin((void *) pointerToData(), unsigned(dataSize())) == 0);	
}

int
SGIConverter::waitForStop(StatusAction* askedToStop) {
	BUG("SGIConverter::waitForStop()");
	while(ALgetfilled(audio_port) > 0) {	
		if((*askedToStop)())
			break;
		sginap(1);		// allow audio buffer to drain
	}
	return true;
}


int
SGIConverter::sampleSize() {
	return type_to_sampsize(dataType());
}

typedef void (*SignalFun)(...);

void
SGIConverter::catchSignals(boolean flag) {
	if(flag) {
		signal(SIGHUP, SignalFun(signalExit));
		signal(SIGINT, SignalFun(signalExit));
		signal(SIGQUIT, SignalFun(signalQuit));
		signal(SIGBUS, SignalFun(signalQuit));
		signal(SIGSEGV, SignalFun(signalQuit));
		signal(SIGTERM, SignalFun(signalExit));
	}
	else {
		signal(SIGHUP, SIG_DFL);
		signal(SIGINT, SIG_DFL);
		signal(SIGQUIT, SIG_DFL);
		signal(SIGBUS, SIG_DFL);
		signal(SIGSEGV, SIG_DFL);
		signal(SIGTERM, SIG_DFL);
	}
}

void
SGIConverter::ignoreSignals(boolean flag) {
	if(flag) {
#ifdef SIGTSTP
		signal(SIGTSTP, SignalFun(signalIgnore));
#endif
	}
	else {
#ifdef SIGTSTP
		signal(SIGTSTP, SIG_DFL);
#endif
	}
}

// these are declared static to allow passing to signal()

void
SGIConverter::signalExit(...) {
	fprintf(stderr, "\nCaught signal.  Exiting...\n");
	fflush(stderr);
	Converter::destroyInstance();
	exit(0);
}

void
SGIConverter::signalIgnore(...) {
	fprintf(stderr, "\nCaught signal during play or record...ignoring.\n");
}

void
SGIConverter::signalQuit(...) {
	fprintf(stderr, "\nCaught deadly signal.  Aborting...\n");
	fflush(stderr);
 	abort();
}
