// sun_dac.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#undef debug

#include <sys/stat.h>
#include <fcntl.h>
#include <stropts.h>
#include "application.h"
#include "typeconvert.h"
#include "sun_audio.h"
#include "sun_dac.h"
#include "statusaction.h"

extern "C" int poll(struct pollfd*, unsigned long, int);

double SunConverter::recordVolume = -1.0;
double SunConverter::playVolume = -1.0;
double SunConverter::savedRecordVolume = -1.0;
double SunConverter::savedPlayVolume = -1.0;
unsigned SunConverter::savedPort = AUDIO_SPEAKER;
boolean SunConverter::is_sparc_10 = false;
struct pollfd SunConverter::pollStruct;

SunConverter::SunConverter() : ConverterDevice("/dev/audio") {
	BUG("SunConverter::SunConverter()");
	if(!confirmDeviceType() || !initialize())
		fail();
}

SunConverter::~SunConverter() {
	BUG("SunConverter::~SunConverter()");
	stop();
}

boolean
SunConverter::isPlayableFormat(DataType type) {
	if(isSparc10())
		return (type == ShortData || type == MuLawData);
	else
		return (type == MuLawData);
}

DataType
SunConverter::bestPlayableType() {
	return isSparc10() ? ShortData : MuLawData;
}

int
SunConverter::pause() {
	BUG("SunConverter::pause()");
	int ret = playing() ? audio_pause_play(fdesc())
		: recording() ? audio_pause_record(fdesc()) : 0;
	return (ret == AUDIO_SUCCESS && Super::pause());
}

int
SunConverter::resume() {
	BUG("SunConverter::resume()");
	int ret = playing() ? audio_resume_play(fdesc())
		: recording() ? audio_resume_record(fdesc()) : 0;
	return (ret == AUDIO_SUCCESS && Super::resume());
}

int
SunConverter::stop() {
	BUG("SunConverter::stop()");
	int status = true;
	if(running()) {
		if(audio_flush(fdesc()) != AUDIO_SUCCESS) {
			error("Can't flush audio device.");
			status = false;
		}
		resetDefaults();
		status = Super::stop() && status;
	}
	return status;
}

int
SunConverter::doConfigure() {
	BUG("SunConverter::doConfigure()");
	Audio_hdr	Dev_hdr;		// audio header for device
	Audio_hdr	File_hdr;		// audio header for file
	
	// fill in values from buffer to be played
	File_hdr.sample_rate = sampleRate();
	File_hdr.samples_per_unit = 1;
	File_hdr.bytes_per_unit = type_to_sampsize(dataType());
	File_hdr.channels = channels();
	File_hdr.encoding = type_to_format(dataType());
	File_hdr.data_size = dataSize();
	
	boolean status = true;
	if(!reOpen() || !getConfiguration(&Dev_hdr))
		status = false;
	else if(audio_cmp_hdr(&Dev_hdr, &File_hdr) != 0
			&& !reconfigure(&Dev_hdr, &File_hdr))
		status = false;
	else if(!setLevels())
		status = false;
	int afd = fdesc();
    if(status && willRecord()) {	// Set the device up for non-blocking reads
		if(fcntl(afd, F_SETFL, fcntl(afd, F_GETFL, 0) | O_NDELAY) < 0) {
			error("Converter::configure: fcntl() failed.");
			status = false;
		}
	}
	pollStruct.fd = afd;	// initialize polling struct
	pollStruct.events = willRecord() ? POLLIN : POLLOUT;
	return status;
}

boolean
SunConverter::getConfiguration(Audio_hdr *devhdr) {
	// Get the device output encoding configuration
	boolean status = true;
	if(willPlay()) {
		if(audio_get_play_config(fdesc(), devhdr) != AUDIO_SUCCESS) {
			error("The device is not an audio device.");
			status = false;
		}
	}
	else if(willRecord()) {
		if(audio_get_record_config(fdesc(), devhdr) != AUDIO_SUCCESS) {
			error("The device is not an audio device.");
			status = false;
		}
	}
	// Get the device record/play levels and port settings
	return status && getDefaults();
}

boolean
SunConverter::reconfigure(Audio_hdr *devhdr, Audio_hdr *filehdr) {
	BUG("SunConverter::reconfigure()");
	char msg[AUDIO_MAX_ENCODE_INFO+64];
	char encoding[AUDIO_MAX_ENCODE_INFO];
	char* action = willRecord() ? "record" : "play";
	// wait for old output to drain, if any
	if(willPlay() && audio_drain(fdesc(), 0) != AUDIO_SUCCESS)
		return false;
	*devhdr = *filehdr;
	int retval = willPlay() ?
		audio_set_play_config(fdesc(), devhdr)
		: audio_set_record_config(fdesc(), devhdr);
	switch (retval) {
	case AUDIO_SUCCESS:
		break;
	case AUDIO_ERR_NOEFFECT:
		// Couldn't change the device.
		// Check to see if we're nearly compatible.
		// audio_cmp_hdr() returns >0 if only sample rate difference.
		static const double SampRateThreshold = 0.05;
		if(audio_cmp_hdr(devhdr, filehdr) == 1) {
			double ratio = double(abs(
				int(devhdr->sample_rate - filehdr->sample_rate))) /
				double(filehdr->sample_rate);
			if (ratio <= SampRateThreshold) {
				sprintf(msg, "Samp rate was %d, %sing at %d",
					filehdr->sample_rate, action, devhdr->sample_rate);
				Application::inform(msg, true);
			}
			else {
				sprintf(msg, "Cannot %s at sample rate %d.", action,
					filehdr->sample_rate);
				Application::alert(msg);
				return false;
			}
		}
		else {
			(void) audio_enc_to_str(filehdr, encoding);
			sprintf(msg,
				"SunConverter::reconfigure: Cannot %s sound in format: %s",
					action, encoding);
			Application::alert(msg);
			return false;
		}
		break;	/*NOTREACHED*/
	case AUDIO_ERR_ENCODING:
		(void) audio_enc_to_str(filehdr, encoding);
		sprintf(msg, "Cannot %s sound in format: %s", action, encoding);
		Application::alert(msg);
		return false;
		break;	/*NOTREACHED*/
	case AUDIO_UNIXERROR:
		if(isSparc10()) {	// Sparc 10 generates this instead of previous
			(void) audio_enc_to_str(filehdr, encoding);
			sprintf(msg, "Cannot %s sound in format: %s", action, encoding);
			Application::alert(msg);
		}
		else
			error("audio configuration failed");
		return false;
		break;	/*NOTREACHED*/
	default:
		(void) audio_enc_to_str(filehdr, encoding);
		sprintf(msg, "audio configuration failed for format: %s", encoding);
		Application::alert(msg);
		return false;
		break;	/*NOTREACHED*/
	}
	return true;
}

// Write EOF to device, then loop to check for play completion

int
SunConverter::waitForStop(StatusAction* askedToStop) {
	BUG("SunConverter::waitForStop()");
	if(audio_play_eof(fdesc()) != AUDIO_SUCCESS)
		return error("Unable to write EOF to device");
	unsigned EOF_count = 0;
	do {
		usleep(50000);	// sleep .05 secs
		if((*askedToStop)())
			break;
		if(audio_get_play_eof(fdesc(), &EOF_count) != AUDIO_SUCCESS)
			return error("Converter::waitForStop: audio_get_play_eof failed");
	} while (EOF_count == 0);
	return true;
}

SunConverter::OutputPort
SunConverter::currentOutputPort() {
	unsigned audio_port;
	if(reOpen() && audio_get_play_port(fdesc(), &audio_port) != AUDIO_SUCCESS) {
//		error("Can't get audio port!");
		fprintf(stderr, "SunConverter::currentOutputPort:  Can't get audio port!\n");
		return Speaker;
	}
	return (audio_port == AUDIO_SPEAKER) ? Speaker : Headphone;
}
	
int
SunConverter::setOutputPort(OutputPort port) {
	unsigned audio_port = (port == Speaker) ? AUDIO_SPEAKER : AUDIO_HEADPHONE;
	if(reOpen() && audio_set_play_port(fdesc(), &audio_port) != AUDIO_SUCCESS) {
		error("Can't set audio port!");
		return false;
	}
	close();
	return true;
}

double
SunConverter::playGain() {
	return (playVolume < 0.0) ? savedPlayVolume : playVolume;
}

double
SunConverter::recordGain() {
	return (recordVolume < 0.0) ? savedRecordVolume : recordVolume;
}

// public access to levels

int
SunConverter::currentPlayLevel() {
	return round(playGain() * 100);
}

int
SunConverter::currentRecordLevel() {
	return round(recordGain() * 100);
}

int
SunConverter::setPlayLevel(int volume) {
	playVolume = volume / 100.0;
	return true;
}

int
SunConverter::setRecordLevel(int volume) {
	recordVolume = volume / 100.0;
	return true;
}

boolean
SunConverter::setLevels() {
	BUG("SunConverter::setLevels()");
	double gain = 0.0;
	if(willRecord()) {
		gain = recordGain();
		if(audio_set_record_gain(fdesc(), &gain) != AUDIO_SUCCESS)
			return error("Unable to set converter record level.");
	}
	else if(willPlay()) {
		gain = playGain();
		if(audio_set_play_gain(fdesc(), &gain) != AUDIO_SUCCESS)
			return error("Unable to set converter play level.");
	}
	return true;
}

int
SunConverter::confirmDeviceType() {
	struct stat st;
	if(stat(deviceName(), &st) < 0) {
		error("Cannot stat converter device.");
		return false;
	}
	if (!S_ISCHR(st.st_mode)) {
		char msg[128];
		sprintf(msg, "%s %s", deviceName(), "is not an audio device.");
		Application::alert(msg);
		return false;
	}
	return true;
}

extern "C" char* getenv(const char*);

int
SunConverter::initialize() {
	BUG("SunConverter::initialize()");
	// to avoid two binaries, we use an environmental var to distinguish
	const char* enVar = getenv("MXV_SPARC_10");
	is_sparc_10 = (enVar && (!strcmp(enVar, "YES") || !strcmp(enVar, "yes")));
	// get current values from the device and store
	int status = reOpen() && getDefaults();
	close();
	return status;
}

int
SunConverter::getDefaults() {
	BUG("SunConverter::getDefaults()");
	int err;
	err = audio_get_play_gain(fdesc(), &savedPlayVolume);
	err = audio_get_play_port(fdesc(), &savedPort);
	err = audio_get_record_gain(fdesc(), &savedRecordVolume);
	if(err != AUDIO_SUCCESS) {
		error("Can't retrieve audio defaults.");
		return false;
	}
	return true;
}

void
SunConverter::resetDefaults() {
	BUG("SunConverter::resetDefaults()");
	audio_set_play_gain(fdesc(), &savedPlayVolume);
	audio_set_play_port(fdesc(), &savedPort);
	audio_set_record_gain(fdesc(), &savedRecordVolume);
}

void
SunConverter::waitForDevice() {
	(void) poll(&pollStruct, 1L, -1);
}

// always write 0.1 seconds of sound per buffer

int
SunConverter::writeSize() {
	return int(0.1 * sampleRate() * channels() * type_to_sampsize(dataType()));
}

// later this may be changed

int
SunConverter::readSize() {
	return writeSize();
}
