// arrayfunction.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/

#ifdef __GNUG__
#pragma implementation
#endif

#include "application.h"
#include "arrayfunction.h"
#include "data.h"

int
ArrayFunction::doApply(Data *data) {
	int frameSize = framesize();
	if(frameSize > data->length()) {
		Application::alert("Framesize is too large for selected region.");
		return false;
	}	
	int nframes = analysisLength();
	double *sig = new double[frameSize];
	bzero((char*) sig, frameSize * sizeof(double));
	int currentOffset = 0;
	int frameNumber = 0;
	int frameOffset = initialOffset();				// initial offset
	int overlap = max(0, frameSize - frameOffset);	// initial overlap
	Range rangeToClone(0, 0);
	int status = true;
	while(status == true) {
		int newsamps = frameSize - overlap;
		if(data->getArray(sig+overlap, newsamps, 0, currentOffset) <= -overlap)
			break;									// left frame edge at EOF 
		Data* frame = getAnalysis()->clone(rangeToClone);
		status = (*this)(sig, frame);				// do analysis of sig
		delete frame;
		currentOffset += frameOffset;				// incr. by old offset
		frameOffset = offset();						// get new offset
		overlap = max(0, frameSize - frameOffset);	// get new overlap
		double *to = sig;
		double *from = sig + frameOffset;
		for (int i = 0; i < overlap; i++)			// shift data to left
			*to++ = *from++;					
		rangeToClone += 1;
		frameNumber++;
		char report[64];
		sprintf (report, "%s frame %5d of %5d", 
			message(), frameNumber, nframes);
		if(!(frameNumber % 5)) Application::inform(report);
	}
	if(nframes > frameNumber)
		getAnalysis()->changeLength(frameNumber);	// truncate to actual #
	getAnalysis()->Notify();
	delete [] sig;
	status = (frameNumber > 0);	// success if any frames analyzed
	return status;
}

// returns the number of samps to read for the first frame -- often less
// than a full frame's worth

int
ArrayFunction::initialOffset() {
	return min(offset(), framesize());
}

// returns the number of analysis frames that will be generated from a given
// length file and a given frame size and offset

int
ArrayFunction::analysisLength() {
	return 1 + (target()->length() + framesize() - initialOffset()) / offset();
}

boolean 
ArrayFunction::setBaseValues(int flength, double frate, int foffset) {
	if(flength >= maxFrameLength()) {
		char msg[80];
		sprintf(msg, "Framesize must be less than %d.", maxFrameLength());
		Application::alert(msg);
		return false;
	}
	else if(flength <= 0) {
		Application::alert("Framesize must be positive, nonzero value.");
		return false;
	}
	setFrameLength(flength);
	int offst = (frate == 0.0) ? foffset : int(sampRate() / frate);
	if(offst <= 0) {
		Application::alert("Offset must be a positive, nonzero value.");
		return false;
	}
	setOffset(offst);
	return true;
}
