// data.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include <InterViews/subject.h>
#include "application.h"
#include "localdefs.h"
#include "comment.h"
#include "data.h"
#include "datafile.h"
#include "envelope.h"
#include "header.h"
#include "outputfunction.h"
#include "framemodifier.h"

//*******************

// DataState is a subject that contains the views of a Data instance, and it
// is shared between a Data instance and any clones of that instance

class DataState : public Subject {
	friend class Data;
protected:
	DataState(double maxval=0.0)
		: isModified(true), updateDeferred(false), nviews(0),
		  maximumValue(maxval), viewFrames(false) {}
	virtual ~DataState() {}
	virtual void Attach(Interactor *i) { Subject::Attach(i); nviews++; }
	virtual void Detach(Interactor *i);
	virtual void Notify() { if(!updateDeferred) Subject::Notify(); }
	boolean modified() const { return isModified; }
	void modified(boolean b) { isModified = b; }
	void deferUpdate(boolean flag);
	boolean updateIsDeferred() const { return updateDeferred; }
	int currentViews() const { return nviews; }
	boolean viewAsFrames() const { return viewFrames; }
	void viewAsFrames(boolean flag) { viewFrames = flag; }
	double maxValue() const { return maximumValue; }
	void setMaxValue(double m) { maximumValue = m; }
private:
	boolean isModified;
	boolean updateDeferred;
	int nviews;
	double maximumValue;
	boolean viewFrames;
};

inline void
DataState::Detach(Interactor *i) {
	if(IsView(i)) {
		Subject::Detach(i);
		nviews--;
	}
}

inline void
DataState::deferUpdate(boolean flag) {
	if(flag != updateDeferred) {
		updateDeferred = flag;
		Notify();
	}
}

//*******************

boolean Data::scanDeferred = false;

Data::Data() : rep(DataRep::create(CharData, 0, 1)),
		myState(nil), myComment(nil) {
	Init();
}

Data::Data(DataType type, int length, int chans)
		: rep(DataRep::create(type, length, chans)),
		  myState(nil), myComment(nil) {
	Init();
}

// these constructors are used only when creating clones

Data::Data(const Data *d, const Range &selection)
		: rep(d->rep->clone(selection)), myState(d->myState), myComment(nil) {
	Init();
}

Data::Data(const Data *d, const Range &selection, const Range &chans)
		: rep(d->rep->clone(selection, chans)),
		  myState(d->myState), myComment(nil) {
	Init();
}

// these constructors are for use by the "virtual constructors"

Data::Data(const Data *d) : rep(d->rep->newRep()), myState(nil), myComment(nil) {
	Init();
}

Data::Data(const Data *d, int newlen) : rep(d->rep->newLengthRep(newlen)),
		myState(nil), myComment(nil) {
	Init();
}

Data::~Data() {
	Resource::unref(myComment);
	Resource::unref(myState);
	Resource::unref(rep);
}

void
Data::Init() {
	rep->ref();
	if(myState == nil)
		myState = new DataState(1.0);	// default maxvalue of 1.0
	myState->ref();
	if(myComment != nil)
		myComment->ref();
}

Data *
Data::newData() {
	return new Data(this);
}

Data *
Data::newData(int length) {
	return new Data(this, length);
}

Data *
Data::clone(const Range& selection) {
	return new Data(this, selection);
}

Data *
Data::clone(const Range& selection, const Range& chans) {
	return new Data(this, selection, chans);
}

Data *
Data::cloneFrame(int frameno) {
	return cloneFrame(frameno, Range(0, frameLength() - 1));
}

// a clone of chans=nchans, length=1 is converted via changeNChans() into a
// clone of chans=frameChannels(), length=nchans/frameChannels()

Data *
Data::cloneFrame(int frameno, const Range& chans) {
	Range range = chans;
	range *= frameChannels();	// if == 2, 3 - 7 becomes 6 - 14, etc.
	Data* newclone = clone(Range(frameno, frameno), range);
	newclone->changeNChans(frameChannels());
	return newclone;
}
	
Data *
Data::copyOf() {
	Data *d = newData();
	d->copyFrom(this);
	return d;
}

int
Data::read(DataFile *f, Header* header) {
	if(header == nil) header = createHeader(f, true);
	header->ref();
	int status = false;
	if((status = header->read(f)) == true) {
		if(header->dataSize() == 0)
			Application::alert("Warning:  file contains 0 frames.");
		Resource::unref(rep);
		rep = DataRep::create(f, header->dataType(), header->nChans());
		rep->ref();
		if(rep->length() > 0) {		// until Exceptions enabled
			setMaxValue(0.0);
			readFromHeader(header);
			// postpone file scan if peak amp loaded by readFromHeader()
			boolean was = deferRescan(state()->maxValue() != 0.0);
			Notify();
			deferRescan(was);
			modified(false);		// because Notify() set to true
		}
		else status = false;		// implies memory failure
	}
	Resource::unref(header);
	return status;
}

const char *
Data::comment() {
	if(myComment == nil)
		setComment("");		// creates new comment of correct size
	return (char *) *myComment;
}

void
Data::setComment(const char* comm) {
	Resource::unref(myComment);
	myComment = new Comment(comm);
	myComment->ref();
	modified(true);
}

void
Data::readFromHeader(Header *h) { // this is called by derived classes as well
	Comment* comment = h->getComment();
	setComment(comment != nil ? (char *) *comment : "");
}

Header *
Data::createHeader(DataFile *, boolean) {
	return nil;
}

Header *
Data::queryForHeader(Controller* c) {
	Header* header = createHeader(nil, true);
	header->ref();
	if (!header->configure(c)) {
		Resource::unref(header);
		header = nil;
	}
	return header;
}

// public method

int
Data::write(DataFile *f) {
	BUG("Data::write(f)");
	return write(f, createHeader(f));
}

// protected method

int
Data::write(DataFile *f, Header* header) {
	BUG("Data::write(f, h)");
	header->ref();
	int status = (header->write(f) && rep->write(f));
	Resource::unref(header);
	modified(modified() && status != true);
	return status;
}

void
Data::writeToHeader(Header *h) {
	h->setDataSize(sizeInBytes());
	h->setComment(myComment);
}

void
Data::deferUpdate(boolean defer) {
	if(defer != state()->updateIsDeferred()) {
		if(defer == false && scanDeferred != true)
			checkValues();	// rescan if update no longer deferred
		state()->deferUpdate(defer);
	}
}

void
Data::Attach(Interactor *i) {
	state()->Attach(i);
	ref();
}

void
Data::Detach(Interactor *i) {
	state()->Detach(i);
	unref();
}

void 
Data::modified(boolean b) {
	state()->modified(b);
}

boolean
Data::modified() const {
	return state()->modified();
}

int
Data::currentViews() const {
	return state()->currentViews();
}

boolean
Data::displayAsFrames() const {
	return(channels() > 1 && state()->viewAsFrames());
}

boolean
Data::displayAsFrames(boolean flag) {
	if(flag && channels() < 4) {
		Application::alert("Data must have at least 4 channels to be displayed as frames.");
		return false;
	}
	else {
		 state()->viewAsFrames(flag);
		 return true;
	}
}

void
Data::Notify() {
	modified(true);
	if(!scanDeferred && !state()->updateIsDeferred())
		checkValues();	// no need to check if update def'rd
	state()->Notify();
}

// static function that can be called anywhere to postpone data scanning

boolean
Data::deferRescan(boolean b) {
	boolean old = scanDeferred;
	scanDeferred = b;
	return old;
}

void
Data::checkValues() {
	// base class does nothing
}

double
Data::maxValue() const {
	if(state()->maxValue() == 0.0) {
		double newmax = scanForMaxValue();
		// cheating here by casting away const to allow scan
		((Data *) this)->setMaxValue(newmax != 0.0 ? newmax : 1.0);
	}
	return state()->maxValue();
}

// retrieves the location and channel of maximum point in data

void
Data::maxValue(int *chan, int *loc) const {
	int len = length();
	double curmax = 0.0;
	Application::inform("Searching for max...");
	for(int ch=0; ch < channels(); ch++) {
		for(int lc = 0; lc < len; lc++) {
			double val = abs(get(lc, ch));
			if(val >= curmax) {
				curmax = val;
				*chan = ch;
				*loc = lc;
			}
		}
	}
}

// returns offset in frames to next zero crossing, or -1 if none

int
Data::zeroCrossing() const {
	int end = length();
	double previous = get(0, 0);
	Application::inform("Searching for zero crossing...");
	for(int loc = 1; loc < end; loc++) {
		double current = get(loc, 0);
		if(current == 0.0 || sign(current) != sign(previous))
			return loc;
		previous = current;
		current = get(loc, 0);
	}
	return -1;		// if none found
}

void
Data::setMaxValue(double val) { state()->setMaxValue(val); }

// the next four are information methods for displaying data

const char *
Data::channelName(int chan) const {
	static char string[8];
	sprintf(string, "Chan %d", chan);
	return string;
}

Range
Data::frameRange(RangeUnit units) const {
	return Range(0, (units==ChannelUnit || units==ChannelSpecialUnit) ?
		frameLength() - 1 : length() - 1);
}

Range
Data::channelRange(boolean) const { 
	return Range(0, channels() - 1);
}

const char *
Data::frameRangeLabel(RangeUnit units) const {
	return (units == ChannelUnit || units == ChannelSpecialUnit) ?
		"Channel" : "Frame Number";
}

// 2nd arg is flag to scan master rep, bypassing clones

Range
Data::limits(int chan, boolean real) const {
	Range range;
	int endpoint = real ? rep->realLength() - 1 : rep->length() - 1;
	rep->valueRange(range, 0, endpoint, chan, chan, real);
	return range.check();
}

double 
Data::scanForMaxValue() const {
	Range range;
	Application::inform("Scanning for peak...");
	rep->valueRange(range);
	return range.absoluteMax();
}

void
Data::copyFrom(const Data *src) {
	rep->copyFrom(src->rep);
	Notify();
}

void
Data::copyRescaledFrom(const Data *src, double scalingFactor, boolean dither) {
	rep->copyRescaledFrom(src->rep, scalingFactor, dither);
	Notify();
}

void
Data::getFrame(Data *frame, int frameno, int chanOffset) const {
	// channel offset is mult. by chans/frame
	rep->getFrame(frame->rep, frameno, chanOffset*frameChannels(), frameChannels());
}

void
Data::getFrame(float *array, int arraySize, int frameno) const {
	rep->getFrame(array, arraySize, frameno);
}

void
Data::getFrame(double *array, int arraySize, int frameno) const {
	rep->getFrame(array, arraySize, frameno);
}

void
Data::setFrame(const Data *frame, int frameno) {
	rep->setFrame(frame->rep, frameno, frameChannels());
}

void
Data::setFrame(float *array, int arraySize, int frameno) {
	rep->setFrame(array, arraySize, frameno);
}

void
Data::setFrame(double *array, int arraySize, int frameno) {
	rep->setFrame(array, arraySize, frameno);
}

void
Data::add(const Data *src) {
	rep->add(src->rep);
	Notify();
}

void
Data::crossfade(const Data* src, Envelope* evp) {
	int framesize = min(channels(), src->channels());
	int len = min(length(), src->length());
	double* target = new double[framesize + 1];
	double* source = new double[framesize + 1];
	evp->setMappingLength(len);
	for(int i = 0; i < len; i++) {
		double sourceAmp = evp->next();
		double targetAmp = 1.0 - sourceAmp;
		rep->getFrame(target, framesize, i);
		src->rep->getFrame(source, framesize, i);
		double* t = target;
		double* s = source;
		for(int j = 0; j < framesize; j++)
			*t++ = (*t * targetAmp) + (*s++ * sourceAmp);
		rep->setFrame(target, framesize, i);
	}
	delete [] source;
	delete [] target;
	Notify();
}

int
Data::replaceWith(const Data *source) {
	int newEndPoint = source->length();
	int status = false;
	if(rep->checkLength(newEndPoint)) {
		copyFrom(source);
		status = true;
	}
	else
		Application::alert("replaceWith:  unable to extend file length.");
	return status;
}

int
Data::spliceIn(const Range& selection) { 
	int status = rep->spliceIn(selection);
	if(status) Notify();
	return status;
}

int
Data::spliceOut(const Range& selection) {
	int status = rep->spliceOut(selection);
	if(status) Notify();
	return status;
}

void
Data::reverse() {
	rep->reverse();
	Notify();
}

void
Data::erase() {
	scaleSelf(0.0);
}

int
Data::apply(SimpleFunction& fun) {
	int status = true;
	if(fun.ok()) {
		int chan, chans = channels();
		for (chan = 0; chan < chans && status == true; ++chan) {
			fun.restoreState();
			status = rep->applyFunction(fun, 0, chan);
		}
		Notify();
	}
	return status;
}

int
Data::apply(OutputFunction& fun) {
	int status = true;
	if(fun.ok()) {
		int chan, chans = channels();
		for (chan = 0; chan < chans; ++chan) {
			fun.restoreState();
			if(rep->getSamplesFrom(fun, 0, chan) == Error) {
				status = false;
				break;
			}
		}
		Notify();
	}
	return status;
}

int
Data::apply(FrameModifier& modifier) {
	int status = true;
	for(int frameno = 0; frameno < length(); frameno++) {
		Data* frame = cloneFrame(frameno);
		frame->ref();
		if(!(status = modifier.modifyFrame(frame)))
			break;
		Resource::unref(frame);
	}
	Notify();
	return status;
}

int
Data::giveSamplesTo(InputOutputFunction& fun) const {
	return rep->giveSamplesTo(fun);
}

void
Data::scaleSelf(double value) {
	Scaler s(this, value);
	s.apply();
}

void
Data::normalize() {
	scaleSelf(1.0/maxValue());
}

int
Data::changeLength(int newlen) {
	return rep->changeLength(newlen);
}

int
Data::getArray(int *array, int size, int channel) const {
	return rep->getArray(array, size, channel);
}

int
Data::getArray(float *array, int size, int channel) const {
	return rep->getArray(array, size, channel);
}

int
Data::getArray(double *array, int size, int channel) const {
	return rep->getArray(array, size, channel);
}

// these next two are protected internal versions

int
Data::getArray(double *array, int size, int channel, int offset) const {
	return rep->getArray(array, size, channel, offset);
}

int
Data::setFromArray(float *array, int size, int channel, int offset) {
	return rep->setFromArray(array, size, channel, offset);
}

int
Data::setFromArray(double *array, int size, int channel, int offset) {
	return rep->setFromArray(array, size, channel, offset);
}

// return value is number of channels loaded into the envelope

int
Data::getEnvelope(Data *evp, int channel, EnvelopeType etype) const {
	double referenceAmp = 0.0;
	// dont extract two traces if envelope has only one channel
	if(etype == PeakToPeakAmplitude && evp->channels() < 2)
		etype = AbsoluteMagnitude;
	else if(etype == Decibels)
		// integral samp types use their max value as reference point
		referenceAmp = (dataType() >= FloatData) ? maxValue() : valueLimit();
	return rep->getEnvelope(evp->rep, channel, etype, referenceAmp);
}

// this is protected, and used only by the Sound subclass

int
Data::changeDataType(DataType newType) {
	int status = false;
	if(newType != dataType()) {
		boolean was = deferRescan(true);
		deferUpdate(true);
		int len = length();
		int chans = channels();
		Data* oldData = copyOf();
		if(oldData->length() > 0) {	// until Exceptions enabled
			Resource::unref(rep);
			rep = DataRep::create(newType, len, chans);
			if(rep->length() > 0) {	// until Exceptions enabled
				rep->ref();
				copyFrom(oldData);
				status = true;
			}
			else Resource::unref(rep);
		}
		deferRescan(was);
		Resource::unref(oldData);
		deferUpdate(false);		// calls Notify()
	}
	return status;
}

// this is protected, and used only by the PvocData subclass

void
Data::changeNChans(int newchans) {
	if(newchans > 0 && newchans != channels())
		rep->changeNChans(newchans);
}
