// datamodifier.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "application.h"
#include "localdefs.h"
#include "controller.h"
#include "datamodifier.h"
#include "data.h"
#include "editor.h"
#include "lpcdata.h"
#include "processfun.h"
#include "range.h"
#include "sound.h"
#include "query.h"
#include "request.h"

DataModifier::DataModifier(Data* data) : myData(data), myEditor(nil) {
	myData->ref();
}

DataModifier::DataModifier(Data* data, DataEditor* editor)
		: myData(data), myEditor(editor) {
	myData->ref();
}

DataModifier::~DataModifier() {
	Resource::unref(myData);
}

int
DataModifier::apply() {
	int status = false;
	if(ok()) {
		Application::inform(message());
		status = doApply(myData);
	}
	return status;
}

void
DataModifier::doPostModifyCommands() {
	if(myEditor != nil)
		doEditorCommands(myEditor);
}

int
DataModifier::sampRate() { return myData->sRate(); }

//********

Modifier *
Reverser::create(DataEditor* ed) {
	return new Reverser(ed->currentSelection());
}

int
Reverser::doApply(Data* data) {
	data->reverse();
	return true;
}

Modifier *
Reverser::createUnDo() {
	return new Reverser(target());
}

//********

Modifier *
Normalizer::create(DataEditor* ed) {
	return new Normalizer(ed->currentSelection());
}

const QueryInfo *
Normalizer::requestInfo() {
    static QueryLabelInfo labels[] = {
		{ "Normalize Current Selection:" },
        { "This will rescale the current selection's amplitudes" },
		{ "to fit between -1 and 1." },
		{ nil }
    };
    static QueryInfo info[] = {
        { labels, "" },
        { nil }
    };
	return info;
}

boolean
Normalizer::confirmValues(Controller *c) {
	return (
		(target()->dataType() > ShortData)
		||
		c->confirm(
			"Warning:  Performing this operation on non-floating-point samples",
			"will irreversibly destroy the data.",
			"Continue anyway?",
			Cancel
		)
	);
}

int
Normalizer::doApply(Data* data) {
	reverseFactor = data->maxValue();		// store inverse for undo
	data->normalize();
	return true;
}

// undo is only possible when data type is float or double

Modifier *
Normalizer::createUnDo() {
	return (target()->dataType() > ShortData) ?
		new Scaler(target(), reverseFactor) : nil;
}

//********

Modifier *
SpaceInsert::create(DataEditor* ed) {
	return new SpaceInsert(ed->currentSelection());
}

const QueryInfo *
SpaceInsert::requestInfo() {
    static QueryLabelInfo labels[] = {
        { "Insert Blank Space: " }, { nil }
    };
    static QueryValueInfo values[] = {
        { "Amount To Insert (frames): ", "0", CharCheck::posIntsOnly },
        { nil }
    };
    static QueryInfo info[] = {
        { labels, "", values },
        { nil }
    };
	return info;
}

boolean
SpaceInsert::setValues(Request& request) {
	QueryValue v;
	request.retrieveValues(v);
	nFrames = v;
	return true;
}

int
SpaceInsert::doApply(Data *data) {
	Range edit(0, nFrames);
	boolean was = Data::deferRescan(true);	// inserting zeros so no rescan
	int status = data->spliceIn(edit);
	Data::deferRescan(was);
	return status;
}

//********

TimeInsert::TimeInsert(Data* d) : SpaceInsert(d) {}

Modifier *
TimeInsert::create(DataEditor* ed) {
	return new TimeInsert(ed->currentSelection());
}

const QueryInfo *
TimeInsert::requestInfo() {
	static QueryLabelInfo labels[] = {
		{ "Insert Blank Space:" }, { nil }
	};
	static QueryValueInfo values[] = {
		{ "Amount To Insert (seconds): ", "0.00",
			CharCheck::posNumsOnly },
		{ nil }
	};
	static QueryInfo info[] = {
		{ labels, "", values },
		{ nil }
	};
	return info;
}

boolean
TimeInsert::setValues(Request& request) {
	QueryValue v;
	request.retrieveValues(v);
	nFrames = round(double(v) * sampRate());
	return true;
}

//********

Modifier *
OutSplicer::create(DataEditor* de) {
	return nil;
}

int
OutSplicer::doApply(Data* data) {
	if(region == data->frameRange()) {
		Application::alert("Entire file may not be spliced out.",
			"Select only a portion of the file.");
		return false;
	}
	else
		return data->spliceOut(region);
}

Modifier *
OutSplicer::createUnDo() {
	return nil;
}

//********

LengthChanger::LengthChanger(Data* d)
	: DataModifier(d), oldlength(d->length()) {}

LengthChanger::LengthChanger(Data* d, int newlen)
		: DataModifier(d), newlength(newlen), oldlength(d->length()) {
	initialize();
}

Modifier *
LengthChanger::create(DataEditor* ed) {
	return new LengthChanger(ed->model());
}

const QueryInfo *
LengthChanger::requestInfo() {
	static QueryLabelInfo labels[] = {
		{ "Change File Length:" }, { nil }
	};
	static QueryInfo info[] = {
		{ labels, "" },
		{ nil }
	};
	return info;
}

void
LengthChanger::configureRequest(Request& request) {
	request.appendValue("New Length (in frames): ", oldlength,
		CharCheck::posIntsOnly);
}

boolean
LengthChanger::confirmValues(Controller *) {
	boolean status = false;
	status = (newlength >= oldlength ||
		Application::confirm("This operation will destroy any data past",
			"the new file endpoint, and is not reversible.", "Continue?",
			Cancel)
	);
	return status;
}

boolean
LengthChanger::setValues(Request& request) {
	QueryValue v;
	request.retrieveValues(v);
	newlength = v;
	if(newlength <= 0) {
		Application::alert("New length must be > 0 frames.");
		return false;
	}
	return true;
}

int
LengthChanger::doApply(Data *data) {
	int status = data->changeLength(newlength);
	if(status == true) {
		// defer rescan if increasing length since extending with zeros
		boolean was = Data::deferRescan(newlength >= oldlength);
		data->Notify();		// data does not automatically do this 
		Data::deferRescan(was);
	}
	return status;
}

Modifier *
LengthChanger::createUnDo() {
	return (newlength > oldlength) ?
		new LengthChanger(target(), oldlength) : nil;
}

//********

DurChanger::DurChanger(Data* d) : LengthChanger(d) {}

Modifier *
DurChanger::create(DataEditor* ed) {
	return new DurChanger(ed->model());
}

const QueryInfo *
DurChanger::requestInfo() {
	static QueryLabelInfo labels[] = {
		{ "Change File Duration:" }, { nil }
	};
	static QueryInfo info[] = {
		{ labels, "" },
		{ nil }
	};
	return info;
}

void
DurChanger::configureRequest(Request& request) {
    request.appendValue("New Length (in seconds): ",
        double(oldlength)/sampRate(), CharCheck::posNumsOnly);
}

boolean
DurChanger::setValues(Request& request) {
	QueryValue v;
	request.retrieveValues(v);
	newlength = round(double(v) * sampRate());
	if(newlength <= 0) {
		Application::alert("Duration must be > 0 seconds.");
		return false;
	}
	return true;
}

//********

SampRateChanger::SampRateChanger(Data* d) : DataModifier(d), doInterp(false) {}
		
SampRateChanger::SampRateChanger(Data* d, int newsrate, boolean interp)
		: DataModifier(d), newSrate(newsrate), doInterp(interp) {
	initialize();
}

Modifier *
SampRateChanger::create(DataEditor* ed) {
	return new SampRateChanger(ed->model());
}

const QueryInfo *
SampRateChanger::requestInfo() {
	static QueryLabelInfo labels[] = {
		{ "Change Sampling Rate:" }, { nil }
	};
	static QueryInfo info[] = {
		{ labels, "" },
		{ nil }
	};
	return info;
}

void
SampRateChanger::configureRequest(Request& request) {
	request.appendValue("New Sampling Rate: ", sampRate(),
		CharCheck::posIntsOnly);
	if(target()->isA(Sound_Data))
		request.appendChoice("Interpolate sound to preserve pitch level:",
			"|Yes|No|", 0x1, true);
}

boolean
SampRateChanger::setValues(Request& request) {
	QueryValue v;
	request.retrieveValues(v);
	newSrate = v;
	QueryChoice c;
	request.retrieveChoices(c);
	doInterp = (c == 0x1);
	return true;
}

boolean
SampRateChanger::confirmValues(Controller *) {
	if(doInterp)
		return Application::confirm("This operation is not reversible.",
			"Continue?");
	else
		return true;
}

int
SampRateChanger::doApply(Data *data) {
	return data->changeSRate(newSrate, doInterp);
}

//********

FormatChanger::FormatChanger(Data* d)
		: DataModifier(d), oldType(d->dataType()) {}
		
FormatChanger::FormatChanger(Data* d, DataType newtype)
		: DataModifier(d), newType(newtype), oldType(d->dataType()) {
	initialize();
}

Modifier *
FormatChanger::create(DataEditor* ed) {
	return new FormatChanger(ed->model());
}

const QueryInfo *
FormatChanger::requestInfo() {
	static QueryLabelInfo labels[] = {
		{ "Change Sound Sample Format:" }, { nil }
	};
	static QueryInfo info[] = {
		{ labels,
		"Reversible only when moving from right to left in format list." },
		{ nil }
	};
	return info;
}

void
FormatChanger::configureRequest(Request& request) {
	request.appendChoice("Convert format to:",
		"|8-bit linear|8-bit mu law|16-bit linear|floating point|",
		(oldType == ShortData) ? ToFloat
		: (oldType == FloatData) ? ToShort
		: ToShort,
		true
	);
}

boolean
FormatChanger::setValues(Request& request) {
	QueryChoice c;
	request.retrieveChoices(c);
	int state = c;
	newType = (state == ToChar) ? CharData : (state == ToMuLaw) ? MuLawData
		: (state == ToShort) ? ShortData : FloatData;
	boolean status = true;
	if(newType == oldType) {
		Application::alert("Samples are already in this format.");
		status = false;
	}
	return status;
}

int
FormatChanger::doApply(Data *data) {
	return ((Sound *) data)->changeDataType(newType);
}

Modifier *
FormatChanger::createUnDo() {
	return (newType > oldType) ? new FormatChanger(target(), oldType) : nil;
}

//********

Modifier *
FrameStabilizer::create(DataEditor* ed) {
	return new FrameStabilizer(ed->model());
}

int
FrameStabilizer::applyToLPCData(LPCData* lpc) {
	lpc->stabilizeFrames();
	return true;
}

//********

FrameWarper::FrameWarper(Data* data, Data* warpdata)
		: LPCModifier(data), warpData(warpdata) {
	warpData->ref();
}

FrameWarper::~FrameWarper() {
	Resource::unref(warpData);
}

const QueryInfo *
FrameWarper::requestInfo() {
	static QueryLabelInfo labels[] = {
		{ "Warp LPC Data Frames: " }, { nil }
	};
	static QueryValueInfo values[] = {
		{ "Amount to warp (between -1.0 and 1.0): ", "0.01",
			CharCheck::numsOnly },
		{ nil }
	};
	static QueryChoiceInfo choices[] = {
		{ "", "|Use envelope|", 0x0, false },
		{ nil }
	};
	static QueryInfo info[] = {
		{ labels, "", values, choices },
		{ nil }
	};
	return info;
}

boolean
FrameWarper::setValues(Request& request) {
	QueryValue v;
	request.retrieveValues(v);
	double warpFactor = v;
	if(warpFactor > 1.0 || warpFactor < -1.0) {
		Application::alert("Warp factor must be between -1.0 and 1.0");
		return false;
	}
	QueryChoice choice;
	request.retrieveChoices(choice);
	if(choice == 1) {
		warpData->normalize();
		warpData->scaleSelf(warpFactor);	// map factor to curve
	}
	else {
		warpData->erase();
		Offset o(warpData, warpFactor);	// set to single value
		o.apply();
	}
	return true;
}

int
FrameWarper::applyToLPCData(LPCData* lpc) {
	return lpc->warpFrames(warpData);
}
