// pvocdata.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/

#ifdef __GNUG__
#pragma implementation
#endif

#include "localdefs.h"
#include "controller.h"
#include "envelope.h"
#include "cs_pvocheader.h"
#include "pvocheader.h"
#include "pvocdata.h"
#include "request.h"

// class members and methods

int PvocData::default_NumberOfBands = 512;
double PvocData::default_FrameRate = 200.0;

void
PvocData::setDefaultNumberOfBands(int bands) { default_NumberOfBands = bands; }

void
PvocData::setDefaultFrameRate(double rate) { default_FrameRate = rate; }

// object methods

Data *
PvocData::newData() { return new PvocData(this); }

Data *
PvocData::newData(int length) { return new PvocData(this, length); }

Data *
PvocData::clone(const Range &r) { return new PvocData(this, r); }

Data *
PvocData::clone(const Range &r, const Range &c) {
	return new PvocData(this, r, c);
}

void
PvocData::print(FILE *out) {
	int npoints = channels();
	fprintf(out, "Phase Vocoder Data Dump\n");
	for(int frame=0; frame < length(); frame++) {
		fprintf(out, "Frame %d:\n", frame);
		for(int i = 0; i < npoints; i=i+2) {
			fprintf(out, "\tFreq:\t%0.2f\tAmp:\t%0.4f\n",
				get(frame, i+1), get(frame, i));
		}
	}
	fflush(out);
}

// returns specific range for freq bands, and max range for all amp bands

Range
PvocData::limits(int chan, boolean real) const {
	Range limit;
	if(chan % 2 != 0)
		return Data::limits(chan);
	else {
		double peak = maxValue();	
		return limit.set(0.0, peak);
	}
}

void
PvocData::information(Controller *controller) {
	AlertRequest request("Phase Vocoder Data Information:");
	request.setBell(false);
	char str[128];
	request.appendLabel("------------");
	request.appendLabel("Filename: ", controller->fileName());
	request.appendLabel("Length (frames): ", pstring(str, length()));
	request.appendLabel("Frequency Bands: ", pstring(str, nBands()));
	request.appendLabel("Frame Rate: ", pstring(str, frameRate()));
	request.appendLabel("Frame Offset (samples): ",
		pstring(str, frameOffset()));
	request.appendLabel("File Size (Mb): ", 
		pstring(str, sizeInBytes()/1000000.0));
	request.appendLabel("Source Sample Rate: ", pstring(str, sRate()));
	controller->handleRequest(request);
}

const char *
PvocData::channelName(int chan) const {
    static char string[64];
	int band = chan / 2;
	double freq = double(band)/nBands() * sRate()/2.0;
	double pfreq = (freq < 1000.0) ? freq : freq/1000.0;
	char freqstring[28];
	sprintf(freqstring, freq < 1000.0 ? "%0.1f hz" : "%0.2fk hz", pfreq);
    sprintf(string, (chan % 2 == 0) ? "Amps@ %s" : "Frqs@ %s", freqstring);
    return string;
}

// This global scope function exists because the two Pvoc header classes are
// currently unrelated, so this cannot be a static member of the common class.

Header *
createPvocHeader(PvocData* pv, DataFile* file) {
    Header* header = nil;
    int magic = file ? Header::readMagicNumber(file) : 0;
    switch(magic) {
    case CS_PvocHeader::PV_MAGIC:
        header = new CS_PvocHeader(
			pv->channels(), pv->frameRate(), pv->sRate()
		);
        break;
    case _SND:
    default:
        header = new PvocHeader(pv->channels(), int(pv->frameRate()));
        break;
    }
	return header;
}

// protected

Header *
PvocData::createHeader(DataFile *file, boolean isNew) {
	Header* h = ::createPvocHeader(this, file);
	if(!isNew) writeToHeader(h);
	return h;
}

void
PvocData::readFromHeader(Header *h) {
	Data::readFromHeader(h);
	PvocHeader *hd = (PvocHeader *) h;
	framerate = hd->sampleRate();	// stored SR is framerate
	// determine srate of anal file from highest freq band
	double maxFreqBand = Data::limits(channels() - 1).max();
	sr = (maxFreqBand > 20000.0) ? 44100
		: (maxFreqBand > 10000.0) ? 22050 : 8000;
}

int
PvocData::getBandNumber(double freq) const {
	double nyquist = sRate() / 2.0;
	return round((min(freq, nyquist) * (nBands() - 1)) / nyquist);
}
