// pvocmodifier.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "application.h"
#include "controller.h"
#include "envelope.h"
#include "pvocdata.h"
#include "pvocmodifier.h"
#include "query.h"
#include "request.h"

SingleFactorModifier::SingleFactorModifier(Data* data)
	: PvocModifier(data), maxBand(pvTarget()->nBands() - 1), envelope(nil) {}

SingleFactorModifier::~SingleFactorModifier() {
	Resource::unref(envelope);
}

boolean
SingleFactorModifier::configure(Controller *c) {
	envelope = (Envelope *) c->findSelection();
	return PvocModifier::configure(c);
}

void
SingleFactorModifier::configureRequest(Request& request) {
	request.appendValue("Modify Frequencies From: ", 0.0,
		CharCheck::posNumsOnly);
	request.appendValue("To: ", double(sampRate()/2.0),
		CharCheck::posNumsOnly);
	request.appendChoice("", "|Use envelope for factor|", 0x0, false);
}

boolean
SingleFactorModifier::setValues(Request& request) {
	static const int nVals = 3;
	QueryValue v[nVals];
	request.retrieveValues(v, nVals);
	factor = v[0];
	double lower = v[1];
	double upper = v[2];
	if(upper > sampRate() / 2.0) {
		Application::alert("Upper boundry must be <= nyquist frequency.");
		return false;
	}
	else if(lower > upper) {
		Application::alert("Invalid range:  upper must be > lower.");
		return false;
	}
	else {
		lowerBand = pvTarget()->getBandNumber(lower);
		upperBand = pvTarget()->getBandNumber(upper);
	}
	QueryChoice choice;
	request.retrieveChoices(choice);
	if(choice == 1) {				// using envelope
		if(envelope == nil) {
			Application::alert("No envelope is currently selected.");
			return false;
		}
	}
	else {
		Resource::unref(envelope);
		envelope = nil;				// to make sure it is not used
	}
	return checkFactor();
}

void
SingleFactorModifier::initialize() {
	if(envelope != nil) {
		envelope->ref();
		if(!envelope->isA(EVP_Data)) {
			Envelope* evp = new Envelope(envelope->length());
			evp->copyFrom(envelope);
			Resource::unref(envelope);
			envelope = evp;
			envelope->ref();
		}
		envelope->setMappingLength(target()->length());
	}
	Super::initialize();
}

double
SingleFactorModifier::currentFactor() {
	return (envelope != nil) ?
		1.0 + envelope->next() * (factor - 1.0) :
		factor;
}

////////

const QueryInfo *
SpectrumTransposer::requestInfo() {
    static QueryLabelInfo labels[] = {
        { "Harmonically Shift (Transpose) Portion of Spectrum: " }, { nil }
    };
    static QueryValueInfo values[] = {
        { "Transposition factor: ", "2.0", CharCheck::posNumsOnly },
        { nil }
    };
    static QueryInfo info[] = {
        { labels, "", values },
        { nil }
    };
	return info;
}

int
SpectrumTransposer::modifyFrame(Data* frame) {
	int fromBand = 0, eraseBand = 0;
	double factor = currentFactor();
	if(factor > 1.0) {			// moving spectrum towards nyquist
		int upperTargetBand = min(maxBand, round(upperBand * factor));
		for(int intoBand = upperTargetBand; intoBand >= 0 ; intoBand--) {
			if((fromBand = round(intoBand / factor)) < lowerBand)
				break;
			frame->set(frame->get(fromBand, 0), intoBand, 0);
			frame->set(factor * frame->get(fromBand, 1), intoBand, 1);
		}
		// zero out portion of old range that doesnt overlap new
		for(eraseBand = min(upperBand, intoBand); eraseBand >= lowerBand;
				eraseBand--) {
			frame->set(0.0, eraseBand, 0);
			frame->set(0.0, eraseBand, 1);
		}
	}
	else if(factor < 1.0) {		// moving spectrum towards zero
		for(int intoBand = lowerBand; intoBand <= upperBand; intoBand++) {
			if((fromBand = round(intoBand / factor)) > maxBand)
				break;
			frame->set(frame->get(fromBand, 0), intoBand, 0);
			frame->set(factor * frame->get(fromBand, 1), intoBand, 1);
		}
		// zero out portion of old range that doesnt overlap new
		for(eraseBand = max(lowerBand, intoBand); eraseBand <= upperBand;
				eraseBand++) {
			frame->set(0.0, eraseBand, 0);
			frame->set(0.0, eraseBand, 1);
		}
	}
	return true;
}

boolean
SpectrumTransposer::checkFactor() {
	return true;
}

////////

const QueryInfo *
SpectrumStretcher::requestInfo() {
    static QueryLabelInfo labels[] = {
        { "Stretch/Shrink Portion of Spectrum: " }, { nil }
    };
    static QueryValueInfo values[] = {
        { "Stretch factor: ", "2.0", CharCheck::posNumsOnly },
        { nil }
    };
    static QueryInfo info[] = {
        { labels, "", values },
        { nil }
    };
	return info;
}

int
SpectrumStretcher::modifyFrame(Data* frame) {
	int fromBand = 0, eraseBand = 0;
	double factor = currentFactor();
	if(factor >= 1.0) {		// stretching spectrum upwards
		int upperTargetBand = min(maxBand, round(upperBand * factor));
		int totalBands = upperTargetBand - lowerBand;
		for(int intoBand = upperTargetBand;  intoBand >= 0; intoBand--) {
			double fac = ((factor-1.)*(intoBand-lowerBand)/totalBands) + 1;
			if((fromBand = round(intoBand / fac)) < lowerBand)
				break;
			frame->set(frame->get(fromBand, 0), intoBand, 0);
			frame->set(fac * frame->get(fromBand, 1), intoBand, 1);
		}
		// zero out portion of old range that doesnt overlap new
		for(eraseBand = min(upperBand, intoBand); eraseBand >= lowerBand;
				eraseBand--) {
			frame->set(0.0, eraseBand, 0);
			frame->set(0.0, eraseBand, 1);
		}
	}
	else {					// compressing spectrum downwards
		int upperTargetBand = round(upperBand * factor);
		int totalBands = upperTargetBand - lowerBand;
		for(int intoBand = lowerBand; intoBand <= upperBand; intoBand++) {
			double fac = 1. - ((1.-factor)*(intoBand-lowerBand)/totalBands);
			if((fromBand = round(intoBand / fac)) > maxBand)
				break;
			frame->set(frame->get(fromBand, 0), intoBand, 0);
			frame->set(fac * frame->get(fromBand, 1), intoBand, 1);
		}
		// zero out portion of old range that doesnt overlap new
		for(eraseBand = max(lowerBand, intoBand); eraseBand <= upperBand;
				eraseBand++) {
			frame->set(0.0, eraseBand, 0);
			frame->set(0.0, eraseBand, 1);
		}
	}
	return true;
}

boolean
SpectrumStretcher::checkFactor() {
	if(round(upperBand * factor) <= lowerBand) {
		Application::alert("Factor incompatible with given frequency range.");
		return false;
	}
	return true;
}
