/*******************************************************************************
+
+  LEDA  2.2.0                                                 03-05-1992
+
+
+  basic.h
+
+
+  Copyright (c) 1992  by  Max-Planck-Institut fuer Informatik
+  Im Stadtwald, 6600 Saarbruecken, FRG     
+  All rights reserved.
+ 
*******************************************************************************/


#ifndef BASICH
#define BASICH


//------------------------------------------------------------------------------
// Compiler/System dependent definitions and includes
//------------------------------------------------------------------------------


#if defined(__ZTC__) || defined(GNUDOS)
#define __MSDOS__
#endif


#include <stdio.h>
#include <stdlib.h>


#if defined(__GNUG__)
#include <stream.h>

#else
#if defined(__ZTC__)
#include <iostream.hpp>

#else
#include <iostream.h>

#endif
#endif



#if !defined(__MSDOS__) || defined(__GNUG__)

#include <values.h>
#include <generic.h>

#else

// values.h:

#define MAXINT          (int(0x7FFFFFFF))
#define MAXFLOAT	(float(3.37E+38))
#define MAXDOUBLE       (double(1.797693E+308))

// generic.h:

#define name2(z, y)		__name2(z, y)
#define __name2(z, y)		z##y
#define name3(z, y, x)	        __name3(z, y, x)
#define __name3(z, y, x)        z##y##x
#define name4(z, y, x, w)	__name4(z, y, x, w)
#define __name4(z, y, x, w)	z##y##x##w

#define declare(z, y) name2(z, declare)(y)
#define implement(z, y) name2(z, implement)(y)
#define declare2(z, y, x) name2(z, declare2)(y, x)
#define implement2(z, y, x) name2(z, implement2)(y, x)


#endif




//------------------------------------------------------------------------------
// Global Type Definitions
//------------------------------------------------------------------------------


typedef void* GenPtr;    // generic pointer type 

typedef int     (*CMP_PTR)    (GenPtr&,GenPtr&);
typedef GenPtr  (*PTR_TO_PTR) (GenPtr&);
typedef int     (*PTR_TO_INT) (GenPtr&);
typedef void    (*PTR_TO_VOID)(GenPtr&);


//------------------------------------------------------------------------------
// LEDA_SIMPLE: base class for LEDA simple types  string, real, point, ...
//------------------------------------------------------------------------------

struct LEDA_SIMPLE { 

GenPtr PTR;

friend int  compare(const LEDA_SIMPLE& x, const LEDA_SIMPLE& y) 
                                      { return int(x.PTR) - int(y.PTR);}

friend void Print(const LEDA_SIMPLE& x, ostream& out = cout) 
                                      { out << "L" << int(x.PTR); }

friend void Read(LEDA_SIMPLE&, istream& = cin) {}
friend void Clear(LEDA_SIMPLE&)             {}
friend void Init(LEDA_SIMPLE&)              {}
friend GenPtr Copy(const LEDA_SIMPLE& y)    { return y.PTR; }
friend GenPtr Convert(const LEDA_SIMPLE& y) { return y.PTR; }

};


//------------------------------------------------------------------------------
// Macros
//------------------------------------------------------------------------------

#define nil      NULL 

#define ACCESS(type,x)  ((type&)Access(name2(type,X),x))
#define ACCESS1(type,x) ((type&)Access(name2(type,X1),x))
#define ACCESS2(type,x) ((type&)Access(name2(type,X2),x))
#define ACCESS3(type,x) ((type&)Access(name2(type,X3),x))


#if defined(__STDC__)
#define STRINGIZE(x) #x
#else
#define STRINGIZE(x) "x"
#endif


#define name5(a,b,c,d,e)	name2(name2(a,b),name3(c,d,e))
#define declare1(a,t)           name2(a,declare1)(t)
#define declare3(a,t1,t2,t3)    name2(a,declare3)(t1,t2,t3)
#define declare4(a,t1,t2,t3,t4) name2(a,declare4)(t1,t2,t3,t4)

#define ITEM(T)      name2(T,_item)

#define COMPARE(t,T) name3(t,T,_cmp) 
#define APPLY(t,T)   name3(t,T,_apply) 
#define ORD(t,T)     name3(t,T,_ord) 


#define Main            main(int argc, char** argv)
#define newline         (cout << "\n").flush()
#define forever         for(;;)
#define loop(a,b,c)     for (a=b;a<=c;a++)
#define Max(a,b)        ( (a>b) ? a : b )
#define Min(a,b)        ( (a>b) ? b : a )

#define SWAP(a,b)       if (a!=b) { *(int*)&a ^= int(b); \
                                    *(int*)&b ^= int(a); \
                                    *(int*)&a ^= int(b); }



// iteration macros for lists and sets:

#define forall(x,S)       for((S).init_iterator(); (S).next_element(x); )
#define Forall(x,S)       for((S).init_iterator(); (S).prev_element(x); )
#define forall_items(x,S) for(x = (S).first_item(); x; x = (S).next_item(x) )





//------------------------------------------------------------------------------
// Memory Management
//------------------------------------------------------------------------------

struct  memory_elem_type { memory_elem_type* next; };
typedef memory_elem_type* memory_elem_ptr;

extern memory_elem_ptr memory_free_list[];

extern void            memory_clear();
extern void            memory_kill();
extern memory_elem_ptr memory_allocate_block(int);
extern memory_elem_ptr allocate_words(int size);
extern memory_elem_ptr Allocate_words(int size);
extern void            deallocate_words(void* p, int size);
extern void            Deallocate_words(void* p, int size);
extern void            deallocate_list(void* head,void* tail, int size);
extern void            print_statistics();


inline memory_elem_ptr allocate_bytes(int n)
{ return allocate_words(n/4); }

inline void deallocate_bytes(void* p, int n)
{ deallocate_words(p,n/4); }

#if defined(__TURBOC__) || defined(GNUDOS)
/* problems with malloc in TURBOC and operator delete in g++-1.39
   --> turn off memory management <--
 */
#define OPERATOR_NEW(words)
#define OPERATOR_DEL(words)

#else
#define OPERATOR_NEW(words)\
void* operator new(size_t)\
{ memory_elem_ptr* q = &(memory_free_list[words]);\
  if (*q==0) *q = memory_allocate_block(words);\
  memory_elem_ptr p = *q;\
  *q = p->next;\
  return p;\
 }

#define OPERATOR_DEL(words)\
void  operator delete(void* p)\
{ memory_elem_ptr* q = &(memory_free_list[words]);\
  memory_elem_ptr(p)->next = *q;\
  *q = memory_elem_ptr(p);\
 }

#endif

#define LEDA_MEMORY(type)\
OPERATOR_NEW(sizeof(type)/4)\
OPERATOR_DEL(sizeof(type)/4)



//------------------------------------------------------------------------------
// compare, Print, Read, Init, Copy, Clear, Convert 
// for basic builtin types
//------------------------------------------------------------------------------


//compare: defines a linear order

inline int compare(GenPtr x, GenPtr y) { return int(x)-int(y); }
inline int compare(char x, char y)     { return x-y; }
inline int compare(int  x, int  y)     { return x-y; }
inline int compare(long x, long y)     { return x-y; }


inline int compare(double x, double y) 
{ if (x < y)  return -1;  
  else if (x > y)  return  1;  
  else return 0; 
}


inline int compare(float x, float y) 
{ if (x < y)  return -1;  
  else if (x > y)  return  1;  
  else return 0; 
}


//Print:  output function

inline void Print(const GenPtr x, ostream& out = cout) { out << int(x); }
inline void Print(char x, ostream& out = cout)         { out << x; }
inline void Print(int  x, ostream& out = cout)         { out << x; }
inline void Print(long x, ostream& out = cout)         { out << x; }
inline void Print(float x, ostream& out = cout)        { out << x; }
inline void Print(double x, ostream& out = cout)       { out << x; }



//Read:  input function

inline void Read(GenPtr,   istream&)          {}
inline void Read(char& x,  istream& in = cin) { in >> x; }
inline void Read(int&  x,  istream& in = cin) { in >> x; }
inline void Read(long& x,  istream& in = cin) { in >> x; }
inline void Read(float& x, istream& in = cin) { in >> x;}
inline void Read(double& x,istream& in = cin) { in >> x;}



//Init: initialize an object

inline void Init(GenPtr)    {}
inline void Init(char&   x) { x=0; }
inline void Init(int&    x) { x=0; }
inline void Init(long&   x) { x=0; }
inline void Init(float&  x) { x=0; }
inline void Init(double& x) { x=0; }


//Clear: destroy the object  (must have been created by Copy)

inline void Clear(GenPtr)   {}
inline void Clear(char  )   {}
inline void Clear(int   )   {}
inline void Clear(long  )   {}
inline void Clear(float )   {}
inline void Clear(double& x) { deallocate_words(&x,2); }


//Copy: makes a copy of the objecty on the heap 
//      and returns a generic pointer to it

inline GenPtr Copy(GenPtr x)  { return x; }
inline GenPtr Copy(char x)    { return GenPtr(x); }
inline GenPtr Copy(int  x)    { return GenPtr(x); }
inline GenPtr Copy(long x)    { return GenPtr(x); }
inline GenPtr Copy(float x)   { return *(GenPtr*)&x; }
inline GenPtr Copy(const double& x) { double* p = (double*) allocate_words(2); 
                                      *p = x; return p; }


//Convert:  returns the address of an object as a GenPtr

inline GenPtr Convert(GenPtr x)        { return x; }
inline GenPtr Convert(int  x)          { return GenPtr(x); }
inline GenPtr Convert(long x)          { return GenPtr(x); }
inline GenPtr Convert(float x)         { return *(GenPtr*)&x; }
inline GenPtr Convert(const double& x) { return GenPtr(&x); }


//Access:  accessing an object through a GenPtr

inline GenPtr&  Access(GenPtr, const GenPtr& p) { return *(GenPtr*)&p; }
inline int&     Access(int,    const GenPtr& p) { return *(int*)&p;  }
inline long&    Access(long,   const GenPtr& p) { return *(long*)&p; }
inline float&   Access(float,  const GenPtr& p) { return *(float*)&p;}
inline double&  Access(double, const GenPtr& p) { return *(double*)p;} 


/* 
   parameter types must define:

   constructor taking no arguments
   input operator:   istream& operator>>(istream&,type&)
   output operator:  ostream& operator<<(ostream&,const type&)
   compare function: int compare(const type&, const type&)

   used if defined:
   destructor       type::~type
   copy constructor type::type(const type&)
   assignment:      type& type::operator=(const type&)


*/


#define LEDA_TYPE_PARAMETER(type)\
inline void   Init(type&) {}\
inline void   Clear(type& x) { delete (type*)&x; }\
inline GenPtr Copy(type& x) { type* p = new type; *p = x; return p; }\
inline GenPtr Convert(type& x) { return GenPtr(&x);}\
inline type&  Access(const type&, GenPtr p) { return *(type*)p; } \
inline void   Read(type& x, istream& in = cin) { in >> x; }\
inline void   Print(const type& x, ostream& out = cout) { out << x; }




//------------------------------------------------------------------------------
// Basic Data Types
//------------------------------------------------------------------------------

enum rel_pos { before = 1, after = 0 };

enum direction { forward = 0, backward = 1 };


//------------------------------------------------------------------------------
// boolean
//------------------------------------------------------------------------------

typedef int bool;       // bad "implementation", should be a class type

#if defined(__MSDOS__)

const bool true  = 1;
const bool false = 0;

#else

extern const bool true;
extern const bool false;

#endif


//------------------------------------------------------------------------------
// strings
//------------------------------------------------------------------------------


class srep { 

public:

      char*   s;      // pointer to c-string
      int count;      // reference counter

 srep(const char*);

~srep() { delete s; }

  LEDA_MEMORY(srep)

};


class string  : public LEDA_SIMPLE 
{

friend class srep;

static char* str_dup(const char*);
static char* str_cat(const char*,const char*);
static char* str_cat(int, char**);

srep*  ptr() const { return (srep*)PTR; }

public:

 string()                { PTR = new srep(""); }
 string(const string& x) { PTR = x.ptr();  ptr()->count++; }
 string(int argc, char** argv) 
                         { PTR = new srep(str_cat(argc,argv)); }


 string(const char*, ...);         // form-like constructor



#if defined(__GNUG__)

// cfront: ambiguity  string(const char* ...)  <--->  string(const char*)

 string(const char* s)   { PTR = new srep(s);}

#endif

void clear()  { if (--ptr()->count==0) delete ptr(); }

~string()     { clear(); }


char*  operator~()  const   { return str_dup(ptr()->s); }
char*  cstring()    const   { return ptr()->s; }

int    length()          const;
string sub(int,int)      const;
int    pos(string,int=0) const;

string operator()(int i, int j)  const { return sub(i,j); }
string head(int i)               const { return sub(0,i-1); }
string tail(int i)               const { return sub(length()-i,length()-1); }

string insert(string, int=0)             const;
string insert(int, string)               const;

string replace(string, string, int=1)    const; 
string replace(int, int, string)    const; 

string del(string, int=1)    const; 
string del(int, int)         const; 

void   read(istream&, char=' ');
void   read(char delim=' ')     { read(cin,delim); }

string replace_all(string s1, string s2) const  { return replace(s1,s2,0); } 
string replace(int i, string s)          const  { return replace(i,i,s);  } 

string del_all(string s)                 const  { return del(s,0); }
string del(int i)                        const  { return del(i,i); }  

void   read_line(istream& I = cin)              { read(I,'\n'); }

string format(string) const;

int    refs() const { return ptr()->count; }

string& operator=(const char*);
string& operator=(const string&);

char  operator[](int) const;
char& operator[](int);

string operator+(const string& x)  const;
string operator+(const char* s1)   const;
string& operator+=(const char* s1);
string& operator+=(const string& x);


friend int operator==(const string& x, const char* s);
friend int operator==(const string& x, const string& y);
friend int operator!=(const string& x, const char* s);
friend int operator!=(const string& x, const string& y);
friend int operator< (const string& x, const string& y);
friend int operator> (const string& x, const string& y);
friend int operator<=(const string& x, const string& y);
friend int operator>=(const string& x, const string& y);


friend istream& operator>>(istream&, string&); 
friend ostream& operator<<(ostream&, const string&) ;

friend void Print(const string& x, ostream& out = cout) { out << x; }
friend void Read(string&  x, istream& in = cin)   { in  >> x; }


friend GenPtr  Copy(const string& x) { x.ptr()->count++;  return x.PTR; }
friend void    Clear(string& x)      { x.clear(); }
friend string& Access(const string&, const GenPtr& p){ return *(string*)&p; }

friend int  compare(const string& x, const string& y);


};

//------------------------------------------------------------------------------
// STRING(cmp): string with user defined linear order cmp
//------------------------------------------------------------------------------

#define STRING(cmp) name2(string_,cmp)

#define STRINGdeclare(cmp)\
struct STRING(cmp) : public string \
{  STRING(cmp)(string s)       : string(s) {}\
   STRING(cmp)(const STRING(cmp)& s) : string(s) {}\
   STRING(cmp)() {}\
 ~ STRING(cmp)() {}\
};\
\
int compare(const STRING(cmp)& x, const STRING(cmp)& y) { return cmp(x,y); }


//------------------------------------------------------------------------------
// INT(cmp): int with user defined linear order cmp
//------------------------------------------------------------------------------

#define INT(cmp) name2(int_,cmp)

#define INTdeclare(cmp)\
struct INT(cmp)\
{  int p;\
 INT(cmp)()                  { p = 0; }\
 INT(cmp)(const int i)       { p = i;}\
 operator int() { return p; }\
\
friend void Print(const INT(cmp)& x, ostream& out = cout) { out << x.p; }\
friend void Read(INT(cmp)&  x, istream& in = cin)  { in  >> x.p; }\
friend void Init(INT(cmp)&  x)            { x.p=0; }\
friend GenPtr Copy(const INT(cmp)& x)     { return GenPtr(x); }\
friend void Clear(INT(cmp)& x)            { x.p = 0; }\
friend GenPtr Convert(const INT(cmp)& x)  { return GenPtr(x.p); }\
inline INT(cmp)&  Access(const INT(cmp)&, const GenPtr& p) { return *(INT(cmp)*)&p; }\
\
};\
\
int compare(const INT(cmp)& x, const INT(cmp)& y) { return cmp(x,y); }




//------------------------------------------------------------------------------
// some useful functions
//------------------------------------------------------------------------------

typedef int (*SIG_PF) (...);

extern SIG_PF signal(int, SIG_PF);

extern int interrupt_handler(...);

inline SIG_PF catch_interrupts(SIG_PF handler = interrupt_handler)
{ return signal(2,handler); }



extern void     init_random(int=0);
extern double   rrandom();                   // random number in [0,1]
extern int      random(int a, int b);        // integer random number in [a..b]
extern unsigned random(unsigned, unsigned);

extern float used_time();
extern float used_time(float&);
extern void  print_time(string s);

inline void print_time() { print_time(""); }

extern void wait(unsigned int seconds);


//------------------------------------------------------------------------------
// input/output
//------------------------------------------------------------------------------

// Problems with cfront's "stdarg" on SPARCs


#if defined(sparc) && !defined(__GNUG__) 

extern  "C" void  save_sparc_registers();  // defined in "_fix.c"

#define Va_Start  save_sparc_registers();va_start
#define form      ~string

#else

#define Va_Start  va_start

#endif


extern int     Yes(string s);
extern int     Yes();

extern int     read_int(string s);
extern int     read_int();

extern char    read_char(string s);
extern char    read_char();

extern double  read_real(string s);
extern double  read_real();

extern char*  read_string(string s);
extern char*  read_string();

extern char*  read_line(istream& s);
extern char*  read_line();


//------------------------------------------------------------------------------
// Error Handling
//------------------------------------------------------------------------------

typedef void (*PEH)(int,char*);   // Pointer to Error Handler

extern PEH p_error_handler;
extern PEH set_error_handler(PEH);
extern void default_error_handler(int,char*);


inline void error_handler(int i, char* s)  { p_error_handler(i,s); }
inline void error_handler(int i, string s) { p_error_handler(i,~s); }


//------------------------------------------------------------------------------
// LEDA INIT
//------------------------------------------------------------------------------

extern const char* leda_version_string;

class LEDA_INIT {

char* init_list;

public:

 LEDA_INIT();
~LEDA_INIT();

};

extern LEDA_INIT L_E_D_A;

#endif
