/*******************************************************************************
+
+  LEDA  2.2.0                                                 03-05-1992
+
+
+  _pers_tree.c
+
+
+  Copyright (c) 1992  by  Max-Planck-Institut fuer Informatik
+  Im Stadtwald, 6600 Saarbruecken, FRG     
+  All rights reserved.
+ 
*******************************************************************************/



#include <LEDA/pers_tree.h>


#define LEFT 0
#define RIGHT 1
#define f_isred(p)   ((p)->f_color)
#define c_isred(p)   ((p)->c_color)
#define f_mred(p)    ((p)->f_color=1)
#define c_mred(p)    ((p)->c_color=1)
#define f_mblack(p)  ((p)->f_color=0)
#define c_mblack(p)  ((p)->c_color=0)
#define f_lchild(p)  ((p)->f_link[0])
#define c_lchild(p)  ((p)->c_link[0])
#define f_rchild(p)  ((p)->f_link[1])
#define c_rchild(p)  ((p)->c_link[1])
#define f_parGenPtr(p)  ((p)->f_link[2])
#define c_parGenPtr(p)  ((p)->c_link[2])
#define f_isleaf(p)  (f_lchild(p)==(p))
#define c_isleaf(p)  (c_lchild(p)==(p))


void pers_rb_tree::f_rbclear(F_NODE* p)
{ if (!(f_isleaf(p)))
    { f_rbclear(f_lchild(p));
      f_rbclear(f_rchild(p));
     }
  delete p;
 }

void pers_rb_tree::c_rbclear(C_NODE* p)
{ if (!(c_isleaf(p)))
    { c_rbclear(c_lchild(p));
      c_rbclear(c_rchild(p));
     }
  delete p;
}


F_NODE* pers_rb_tree::f_rbsearch(F_NODE* p, version i)
{
  F_NODE *q;
  
  q = f_rchild(p);
  while (!(f_isleaf(q)))
  {  
    if (i == q->ver_stamp || vless(i, q->ver_stamp))
      q = f_lchild(q);
    else
      q = f_rchild(q);
  }
  return(q);
}

C_NODE* pers_rb_tree::c_rbsearch(C_NODE* p, version i)
{
  C_NODE *q;
  
  q = c_rchild(p);
  while (!(c_isleaf(q)))
  {
    if (i == q->vers_stamp || vless(i, q->vers_stamp))
      q = c_lchild(q);
    else
      q = c_rchild(q);
  }
  return(q);
}

F_NODE * pers_rb_tree::f_nleaf(NODE* newvalue, version i)
{
  F_NODE* result = new F_NODE;

  f_mblack(result);
  result->ver_stamp = i;
  f_lchild(result) = f_rchild(result) = result;
  result->poin_value = newvalue;
  return(result);
}

C_NODE * pers_rb_tree::c_nleaf(int newvalue, version i)
{
  C_NODE* result = new C_NODE;

  c_mblack(result);
  result->vers_stamp = i;
  c_lchild(result) = c_rchild(result) = result;
  result->col_value = newvalue;
  return(result);
}

F_NODE * pers_rb_tree::f_nnode(F_NODE* c1, F_NODE* c2)
{
  F_NODE* result = new F_NODE;

  f_mred(result);
  if (vless(c1->ver_stamp, c2->ver_stamp))
  {
    f_lchild(result) = c1;
    c1->f_right = 0;
    f_rchild(result) = c2;
    c2->f_right = 1;
  }
  else
  {
    f_lchild(result) = c2;
    c2->f_right = 0;
    f_rchild(result) = c1;
    c1->f_right = 1;
  }
  f_parGenPtr(c1) = f_parGenPtr(c2) = result;
  result->ver_stamp = (f_lchild(result))->ver_stamp;
  result->poin_value = NULL;
  return(result);
}

C_NODE* pers_rb_tree::c_nnode(C_NODE* c1, C_NODE* c2)
{
  C_NODE* result = new C_NODE;

  c_mred(result);
  if (vless(c1->vers_stamp, c2->vers_stamp))
  {
    c_lchild(result) = c1;
    c1->c_right = 0;
    c_rchild(result) = c2;
    c2->c_right = 1;
  }
  else
  {
    c_lchild(result) = c2;
    c2->c_right = 0;
    c_rchild(result) = c1;
    c1->c_right = 1;
  }
  c_parGenPtr(c1) = c_parGenPtr(c2) = result;
  result->vers_stamp = (c_lchild(result))->vers_stamp;
  result->col_value = 0;  
  return(result);
}

void  pers_rb_tree::f_single_rot(F_NODE* top_node, int dir)
{
  F_NODE *temp, *q;

  q = top_node->f_link[1-dir];
  top_node->f_link[1-dir] = temp = q->f_link[dir];
  temp->f_right = 1 - dir;
  f_parGenPtr(temp) = top_node;
  q->f_link[dir] = top_node;
  temp = f_parGenPtr(top_node);
  f_parGenPtr(top_node) = q;
  q->f_right = top_node->f_right;
  temp->f_link[top_node->f_right] = q;
  f_parGenPtr(q) = temp;
  top_node->f_right = dir;
  return;
}

void  pers_rb_tree::c_single_rot(C_NODE* top_node, int dir)
{
  C_NODE *temp, *q;

  q = top_node->c_link[1-dir];
  top_node->c_link[1-dir] = temp = q->c_link[dir];
  temp->c_right = 1 - dir;
  c_parGenPtr(temp) = top_node;
  q->c_link[dir] = top_node;
  temp = c_parGenPtr(top_node);
  c_parGenPtr(top_node) = q;
  q->c_right = top_node->c_right;
  temp->c_link[top_node->c_right] = q;
  c_parGenPtr(q) = temp;
  top_node->c_right = dir;
  return;
}


void  pers_rb_tree::f_double_rot(F_NODE* top_node, int dir)
{
  F_NODE *q, *r, *temp;

  q = top_node->f_link[1-dir];
  r = q->f_link[dir];
  top_node->f_link[1-dir] = temp = r->f_link[dir];
  temp->f_right = 1 - dir;
  f_parGenPtr(temp) = top_node;
  q->f_link[dir] = (temp = r->f_link[1-dir]);
  temp->f_right = dir;
  f_parGenPtr(temp) = q;
  temp = f_parGenPtr(top_node);
  r->f_right = top_node->f_right;
  r->f_link[dir] = top_node;
  f_parGenPtr(top_node) = r;
  top_node->f_right = dir;
  r->f_link[1-dir] = q;
  f_parGenPtr(q) = r;
  temp->f_link[r->f_right] = r;
  f_parGenPtr(r) = temp;
  return;
}

void  pers_rb_tree::c_double_rot(C_NODE* top_node, int dir)
{
  C_NODE *q, *r, *temp;

  q = top_node->c_link[1-dir];
  r = q->c_link[dir];
  top_node->c_link[1-dir] = temp = r->c_link[dir];
  temp->c_right = 1 - dir;
  c_parGenPtr(temp) = top_node;
  q->c_link[dir] = (temp = r->c_link[1-dir]);
  temp->c_right = dir;
  c_parGenPtr(temp) = q;
  temp = c_parGenPtr(top_node);
  r->c_right = top_node->c_right;
  r->c_link[dir] = top_node;
  c_parGenPtr(top_node) = r;
  top_node->c_right = dir;
  r->c_link[1-dir] = q;
  c_parGenPtr(q) = r;
  temp->c_link[r->c_right] = r;
  c_parGenPtr(r) = temp;
  return;
}


int pers_rb_tree::f_insert(F_NODE* head, NODE* newvalue, version i)
{
  int aux_bool;
  F_NODE *p, *q, *r, *aux_node;

  if (f_rchild(head) == NULL)
  {
    p = f_nleaf(newvalue, i);
    p->f_right = 1;
    f_parGenPtr(p) = head;
    f_rchild(head) = p;
    return(0);
  }
  p = f_rbsearch(head, i);
  if (p->ver_stamp == i)
  {
    p->poin_value = newvalue;
    return(1);
  }
  q = f_nleaf(newvalue, i);
  aux_bool = p->f_right;
  aux_node = f_parGenPtr(p);
  r = f_nnode(p, q);
  f_parGenPtr(r) = aux_node;
  r->f_right = aux_bool;
  aux_node->f_link[aux_bool] = r;
  if (!f_isred(aux_node))
  {
    f_mblack(f_rchild(head));
    return(0);
  }
  q = aux_node;
  p = f_parGenPtr(q);
  while ((f_isred(f_lchild(p))) && (f_isred(f_rchild(p))))
  {
    f_mblack(f_lchild(p));
    f_mblack(f_rchild(p));
    f_mred(p);
    r = p;
    q = f_parGenPtr(r);
    if (!f_isred(q))
    {
      f_mblack(f_rchild(head));
      return(0);
    }
    p = f_parGenPtr(q);
  }
  if (q->f_right == r->f_right)
  {
    f_mred(p);
    f_mblack(q);
    f_single_rot(p, 1-r->f_right);
  }
  else
  {
    f_mblack(r);
    f_mred(p);
    f_double_rot(p, r->f_right);
  }
  f_mblack(f_rchild(head));
  return(0);
}

int pers_rb_tree::c_insert(C_NODE* head, int newvalue, version i)
{
  int aux_bool;
  C_NODE *p, *q, *r, *aux_node;

  if (c_rchild(head) == NULL)
  {
    p = c_nleaf(newvalue, i);
    p->c_right = 1;
    c_parGenPtr(p) = head;
    c_rchild(head) = p;
    return(0);
  }
  p = c_rbsearch(head, i);
  if (p->vers_stamp == i)
  {
    p->col_value = newvalue;
    return(1);
  }
  q = c_nleaf(newvalue, i);
  aux_bool = p->c_right;
  aux_node = c_parGenPtr(p);
  r = c_nnode(p, q);
  c_parGenPtr(r) = aux_node;
  r->c_right = aux_bool;
  aux_node->c_link[aux_bool] = r;
  if (!c_isred(aux_node))
  {
    c_mblack(c_rchild(head));
    return(0);
  }
  q = aux_node;
  p = c_parGenPtr(q);
  while ((c_isred(c_lchild(p))) && (c_isred(c_rchild(p))))
  {
    c_mblack(c_lchild(p));
    c_mblack(c_rchild(p));
    c_mred(p);
    r = p;
    q = c_parGenPtr(r);
    if (!c_isred(q))
    {
      c_mblack(c_rchild(head));
      return(0);
    }
    p = c_parGenPtr(q);
  }
  if (q->c_right == r->c_right)
  {
    c_mred(p);
    c_mblack(q);
    c_single_rot(p, 1-r->c_right);
  }
  else
  {
    c_mblack(r);
    c_mred(p);
    c_double_rot(p, r->c_right);
  }
  c_mblack(c_rchild(head));
  return(0);
}



/* insert the new version in the appropriate position and update the 
 * version list
 */

version pers_rb_tree::new_version(version i)
{

  version succ = v_list->vl.succ(i);
  
  ver_node p = new VER_NODE;

  if (succ != nil)
     p->ser_num = (v_list->vl[i]->ser_num + v_list->vl[succ]->ser_num) / 2.0;
  else 
     p->ser_num = v_list->vl[i]->ser_num + 1000;


  p->popul = v_list->vl[i]->popul;
  p->acc_pointer = v_list->vl[i]->acc_pointer;
  v_list->count++;
  return v_list->vl.insert(p,i);
}

/* implementation of the update step (change of left or right child)
 * for a specific persistent node and update operation 
 */
 
void pers_rb_tree::update(F_NODE* p, NODE* newvalue, version i)
{ 
  F_NODE *p1, *i1, *i2;

  version ip = v_list->vl.succ(i);

  if (f_insert(p, newvalue, i) == 1) return;

  if (ip == nil || vless(ip,p->ver_stamp)) return;

  p1 = f_rbsearch(p, i);
  if (p1->f_right)
  {
    i1 = f_lchild(f_parGenPtr(p1));
    if (f_isleaf(i1))
      goto la;
    else
      i1 = f_rchild(i1);
la: while (p1 != f_rchild(p) && p1->f_right)
      p1 = f_parGenPtr(p1);
    if (p1 == f_rchild(p))
      i2 = NULL;
    else
    {
      i2 = f_rchild(f_parGenPtr(p1));
      while (!f_isleaf(i2))
        i2 = f_lchild(i2);
    }
  }
  else
  {
    i2 = f_rchild(f_parGenPtr(p1));
    if (f_isleaf(i2))
      goto m;
    else
      i2 = f_lchild(i2);
 m: while (p1->f_right == 0)
      p1 = f_parGenPtr(p1);
    i1 = f_lchild(f_parGenPtr(p1));
    while (!f_isleaf(i1))
      i1 = f_rchild(i1);
  }
  if (i2 == NULL || vless(ip, i2->ver_stamp))
     f_insert(p, i1->poin_value, ip);
}
   

/* implementation of the update step (change of color) for a specific
 * persistent node and update operation 
 */
 
void pers_rb_tree::up_col(C_NODE* p, int newvalue, version i)
{ 
  C_NODE *p1, *i1, *i2;
  
  version ip = v_list->vl.succ(i);

  if (c_insert(p, newvalue, i) == 1) return;

  if (ip ==nil  || vless(ip,p->vers_stamp)) return;

  p1 = c_rbsearch(p, i);
  if (p1->c_right)
  {
    i1 = c_lchild(c_parGenPtr(p1));
    if (c_isleaf(i1))
      goto lb;
    else
      i1 = c_rchild(i1);
lb: while (p1 != c_rchild(p) && p1->c_right)
      p1 = c_parGenPtr(p1);
    if (p1 == c_rchild(p))
      i2 = NULL;
    else
    {
      i2 = c_rchild(c_parGenPtr(p1));
      while (!c_isleaf(i2))
        i2 = c_lchild(i2);
    }
  }
  else
  {
    i2 = c_rchild(c_parGenPtr(p1));
    if (c_isleaf(i2))
      goto ma;
    else
      i2 = c_lchild(i2);
ma: while (p1->c_right == 0)
      p1 = c_parGenPtr(p1);
    i1 = c_lchild(c_parGenPtr(p1));
    while (!c_isleaf(i1))
      i1 = c_rchild(i1);
  }
  if (i2 == NULL || vless(ip, i2->vers_stamp))
     c_insert(p, i1->col_value, ip);
}
   

/* implementation of the access step for a specific persistent node 
 * and version 
 */

NODE * pers_rb_tree::acc_step(F_NODE *head, version i)
{
  F_NODE *q, *i1;
 
  q = f_rbsearch(head, i);
  if (q->ver_stamp == i)
    return(q->poin_value);
  if (vless(q->ver_stamp, i))
    return(q->poin_value);
  if (q->f_right)
  {
     i1 = f_lchild(f_parGenPtr(q));
     if (f_isleaf(i1))
       goto t;
     else
       i1 = f_rchild(i1);
  }
  else
  {
    while (q->f_right == 0)
      q = f_parGenPtr(q);
    i1 = f_lchild(f_parGenPtr(q));
    while (!f_isleaf(i1))
      i1 = f_rchild(i1);
  }
t:return(i1->poin_value);
}
  


/* find out whether a given persistent node is red or not in a 
 * specific version 
 */

int pers_rb_tree::isred(NODE* p, version i)
{
  C_NODE *head, *q, *i1;
  
  if (isleaf(p))
    return(0);
  head = p->red;
  q = c_rbsearch(head, i);
  if (q->vers_stamp == i)
    return(q->col_value);
  if (vless(q->vers_stamp, i))
    return(q->col_value);
  if (q->c_right)
  {  
     i1 = c_lchild(c_parGenPtr(q));
     if (c_isleaf(i1))
       goto s;
     else
       i1 = c_rchild(i1);
  }
  else
  {
    while (q->c_right == 0)
      q = c_parGenPtr(q);
    i1 = c_lchild(c_parGenPtr(q));
    while (!c_isleaf(i1))
      i1 = c_rchild(i1);
  }
s:return(i1->col_value);
}

 
/* create a new leaf and initialize the fields with the appropriate values */
 
NODE* pers_rb_tree::newleaf(void* val, void* inf, NODE* pred, NODE* succ, version i)
{
  NODE *result;
  F_NODE *res1, *res2;
  C_NODE *res3;
  
  result = new NODE;
  res1   = new F_NODE;
  res2   = new F_NODE;
  res3   = new C_NODE;
   
    result->key = val;
    result->info = inf;
    result->parent = NULL;
    result->right = 1;
    result->is_leaf = 1;
    result->copy = NULL;

    result->next = v_list->used;
    v_list->used = result;

    res1->ver_stamp = res2->ver_stamp = res3->vers_stamp = i;

    f_lchild(res1) = f_rchild(res1) = f_parGenPtr(res1) = NULL;
    f_lchild(res2) = f_rchild(res2) = f_parGenPtr(res2) = NULL;
    c_lchild(res3) = c_rchild(res3) = c_parGenPtr(res3) = NULL;

    res1->f_right = res2->f_right = res3->c_right = 1;
    res1->f_color = res2->f_color = res3->c_color = 0;

    res1->poin_value = res2->poin_value = NULL; 
    res3->col_value = 0;  

    f_insert(res1, pred, i);
    f_insert(res2, succ, i);
    c_insert(res3, 0, i);

    result->link[0] = res1;
    result->link[1] = res2;
    result->red     = res3;

    return result;
}

/* create a new persistent node and initialize its fields with the
 * appropriate values 
 */
 
NODE* pers_rb_tree::newnode(NODE* c1, NODE* c2, version i)
{
  // c1 and c2 are leaves

  NODE *result;
  F_NODE *res1, *res2,  *res4;   // s.n. : res4 pointer to leaf (copy)
  C_NODE *res3;
  
  result = new NODE;
  res1   = new F_NODE;
  res2   = new F_NODE;
  res3   = new C_NODE;
  res4   = new F_NODE;

    result->parent = NULL;

    result->next = v_list->used;
    v_list->used = result;

    res1->ver_stamp = res2->ver_stamp = res3->vers_stamp = i;

    f_lchild(res1) = f_rchild(res1) = f_parGenPtr(res1) = NULL;
    f_lchild(res2) = f_rchild(res2) = f_parGenPtr(res2) = NULL;
    f_lchild(res4) = f_rchild(res4) = f_parGenPtr(res4) = NULL;
    c_lchild(res3) = c_rchild(res3) = c_parGenPtr(res3) = NULL;

    res1->f_right = res2->f_right = res3->c_right = res4->f_right = 1;
    res1->f_color = res2->f_color = res3->c_color = res4->f_color = 0;

    res1->poin_value = res2->poin_value = res4->poin_value = NULL; 
    res3->col_value = 1;  

    c_insert(res3, 1, i);

    if (cmp_keys(c1->key, c2->key) < 1)
    {
      f_insert(res1, c1, i);
      f_insert(res2, c2, i);
      f_insert(res4, c1, i);
      result->key = c1->key;
    }
    else
    {
      f_insert(res1, c2, i);
      f_insert(res2, c1, i);
      f_insert(res4, c2, i);
      result->key = c2->key;
    }

    result->link[0] = res1;
    result->link[1] = res2;
    result->red     = res3;
    result->copy    = res4;
    result->is_leaf = 0;

    return(result);
}

  
/* implementation of the single rotation for the fully persistent
 * red-black tree
 */

NODE* pers_rb_tree::single_rot(NODE* top_node, int dir, version i)
{
  NODE *temp, *q, *newroot;
  
  newroot = NULL;
  q = child(1 - dir, top_node, i);
  temp = child(dir, q, i);
  update(top_node->link[1 - dir], temp, i);
  update(q->link[dir], top_node, i);
  if ((temp = top_node->parent) == NULL)
    newroot = q;
  top_node->parent = q;
  if (temp != NULL)
    update(temp->link[top_node->right], q, i);
  top_node->right = dir;
  return(newroot);
}

/* implementation of the double rotation for the fully persistent
 * red-black tree
 */

NODE* pers_rb_tree::double_rot(NODE* top_node, int dir, version i)
{
  NODE *q, *r, *temp, *newroot;
  
  newroot = NULL;
  q = child(1 - dir, top_node, i);
  r = child(dir, q, i);
  temp = child(dir, r, i);
  update(top_node->link[1 - dir], temp, i);
  temp = child(1 - dir, r, i);
  update(q->link[dir], temp, i);
  if ((temp = top_node->parent) == NULL)
    newroot = r;
  update(r->link[dir], top_node, i);
  update(r->link[1 - dir], q, i);
  if (temp != NULL)
    update(temp->link[top_node->right], r, i);
  return(newroot);
}

/* the root is colored black after each  update operation */

void pers_rb_tree::m_b_root(version i)
{
  NODE *p;

  p = v_list->vl[i]->acc_pointer;

  if (p != NULL && !isleaf(p) && isred(p, i))
     up_col(p->red, 0, i);
}




//------------------------------------------------------------------------------
// member functions
//------------------------------------------------------------------------------


void pers_rb_tree::init_tree()
{
   /* create dummy (empty) version 0 */
   ver_node v = new VER_NODE;
   v->ser_num =  0;
   v->popul   =  0;
   v->acc_pointer = NULL;
   v_list = new V_LIST;
   v_list->vl.append(v);
   v_list->count = 1;

   v_list->used = NULL;

}

NODE* pers_rb_tree::search(void *val, NODE*& copy, version i)
{
  NODE *p, *q;

  copy = NULL;
  
  if ((p = v_list->vl[i]->acc_pointer) == NULL)
    return(NULL);

  p->parent = NULL;
  p->right = 1;
  while (!isleaf(p))
  {
    int v = cmp_keys(val, get_key(p,i));

    if (v < 1)
    { if (v==0) copy = p;
      q = p;
      p = child(0, p, i);
      p->parent = q;
      p->right = 0;
     }
    else
    { q = p;
      p = child(1, p, i);
      p->parent = q;
      p->right = 1;
     }
   }

   return p;
}


 
version  pers_rb_tree::del(void *val, version i1)
{
  NODE *pos_del, *par1, *par2, *root, *newroot, *temp, *copy;
  
  if ((pos_del = search(val, copy, i1)) == NULL) return i1; //empty tree

  if (cmp_keys(pos_del->key, val) != 0) return i1;  // key not found

  version i  = new_version(i1);

  if ((--(v_list->vl[i]->popul)) == 0)
  {
    v_list->vl[i]->acc_pointer = NULL;
    return i;
  }


  //update links to neighbor leaves

  NODE* pred = child(0,pos_del,i);
  NODE* succ = child(1,pos_del,i);

  if (pred) update(pred->link[1],succ,i);
  if (succ) update(succ->link[0],pred,i);


  root = v_list->vl[i]->acc_pointer;

  if ((par1 = pos_del->parent) == root)
  {
    v_list->vl[i]->acc_pointer = sibling(pos_del, i);
    goto end;
   }

  par2 = par1->parent;
  pos_del = sibling(pos_del, i);
  update(par2->link[par1->right], pos_del, i);
  pos_del->parent = par2;
  pos_del->right = par1->right;

  if (copy != NULL && copy != par1)  // we have to overwrite copy  by pred
  {
    update(copy->copy, pred, i);

   }
  

  if (isred(par1, i))
    goto end;
  if (isred(pos_del, i))
  {
    up_col(pos_del->red, 0, i);
    goto end;
  }
  par1 = sibling(pos_del, i);
  while ((!isred(par1, i)) && (!isred(child(0, par1, i), i)) &&
         (!isred(child(1, par1, i), i)))
  {
    up_col(par1->red, 1, i);
    pos_del = pos_del->parent;
    if (isred(pos_del, i))
    {
      up_col(pos_del->red, 0, i);
      goto end;
    }
    if (pos_del == root)
      goto end;
    else
      par1 = sibling(pos_del, i);
  }
  par2 = pos_del->parent;
  if (isred(par1, i))
  {
    up_col(par2->red, 1, i);
    up_col(par1->red, 0,i);
    if ((newroot = single_rot(par2, pos_del->right, i)) != NULL)
      v_list->vl[i]->acc_pointer = newroot;
    par1 = sibling(pos_del,i);
    if ((!isred(child(0, par1, i), i)) && (!isred(child(1, par1, i), i)))
    {
      up_col(par1->red, 1, i);
      up_col((pos_del->parent)->red, 0, i);
      goto end;
    }
  }
  par2 = pos_del->parent;
  if (!pos_del->right)
    if (isred(child(0, par1, i), i))
    {
      temp = child(0, par1, i);
      up_col(temp->red, isred(par2, i), i);
      up_col(par2->red, 0, i);
      newroot = double_rot(par2, LEFT, i);
    }
    else
    {
      up_col(par1->red, isred(par2, i), i);
      up_col(par2->red, 0, i);
      temp = child(1, par1, i);
      up_col(temp->red, 0, i);
      newroot = single_rot(par2, LEFT, i);
    }
  else
    if (isred(child(0, par1, i), i))
    {
      up_col(par1->red, isred(par2, i), i);
      up_col(par2->red, 0, i);
      temp = child(0, par1, i);
      up_col(temp->red, 0, i);
      newroot = single_rot(par2, RIGHT, i);
    }
    else
    {
      temp = child(1, par1, i);
      up_col(temp->red, isred(par2, i), i);
      up_col(par2->red, 0, i);
      newroot = double_rot(par2, RIGHT, i);
    }
  if (newroot != NULL)
    v_list->vl[i]->acc_pointer = newroot;

end:
  m_b_root(i);
  return i;
}


version pers_rb_tree::insert(void *val, void* info, version i1)
{
  int aux_bool; 
  NODE *p, *q, *r, *aux_node, *root, *newroot, *temp;

  p = search(val, i1);

  if (p && cmp_keys(p->key, val) == 0) return i1;  // key already there

  version i = new_version(i1);

  copy_key(val);
  copy_inf(info);
  
  if (p == NULL)   // empty tree
  { copy_key(val);
    copy_inf(info);
    p = newleaf(val, info, nil, nil, i);
    v_list->vl[i]->acc_pointer = p;
    v_list->vl[i]->popul = 1;
    goto end;
  }

  (v_list->vl[i]->popul)++;

  if (cmp_keys(val, p->key) > 0)   // new rightmost leaf 
    { q = newleaf(val,info, p, nil, i);   
      update(p->link[1], q, i);
     }
  else // new  leaf before  p 
    { NODE * pred = child(0,p,i);
      q = newleaf(val,info, pred, p, i);   
      update(p->link[0], q, i);
      if (pred) update(pred->link[1], q, i);
     }

  aux_bool = p->right;
  r = newnode(p, q, i);

  if (p->parent == NULL)
  {
    v_list->vl[i]->acc_pointer = r;
    goto end;
  }
  aux_node = p->parent;
  r->right = aux_bool;
  update(aux_node->link[aux_bool], r, i);

  if (!isred(aux_node, i))
     goto end;

  q = aux_node;
  p = q->parent;
  root = v_list->vl[i]->acc_pointer;
  while ((isred(child(0, p, i), i)) && (isred(child(1, p, i), i)))
  {
    temp = child(0, p, i);
    up_col(temp->red, 0, i);
    temp = child(1, p, i);
    up_col(temp->red, 0, i);
    up_col(p->red, 1, i);
    if (p == root)
      goto end;
    r = p;
    q = r->parent;
    if (!isred(q, i))
      goto end;
    p = q->parent;
  }
  if (q->right == r->right)
  {
    up_col(p->red, 1, i);
    up_col(q->red, 0, i);
    newroot = single_rot(p, 1 - r->right, i);
  }
  else
  {
    up_col(r->red, 0, i);
    up_col(p->red, 1, i);
    newroot = double_rot(p, r->right, i);
  }

  if (newroot != NULL)
    v_list->vl[i]->acc_pointer = newroot;

end:
  m_b_root(i);
  return i;
}


version pers_rb_tree::change_inf(NODE* p, void* info, version i1)
{
  version i = new_version(i1);

  p = search(p->key, i1);  // setting parent pointers

  NODE* pred = child(0,p,i);
  NODE* succ = child(1,p,i);

  copy_inf(info);
  NODE* q = newleaf(p->key,info, pred, succ, i);   


  if (pred) update(pred->link[1],q,i);
  if (succ) update(succ->link[0],q,i);

  q->right = p->right;
  update(p->parent->link[q->right], q, i);

  return i;

}



NODE* pers_rb_tree::min(version v)
{ NODE* r = v_list->vl[v]->acc_pointer;
  if (r == nil) return nil;
  while ( ! isleaf(r)) r = child(0,r,v);
  return r;
}  

NODE* pers_rb_tree::max(version v)
{ NODE* r = v_list->vl[v]->acc_pointer;
  if (r == nil) return nil;
  while ( ! isleaf(r)) r = child(1,r,v);
  return r;
}  

void pers_rb_tree::print(NODE *p, version v)
{ if (p)
  { if (isleaf(p))  
    { print_key(p->key);
      cout << " ";
     }
    else
     { print(child(0, p, v), v);
       print(child(1, p, v), v);
      }
   }
}  

void pers_rb_tree::del_tree()
{ 
  while (v_list->used)
  { NODE* p = v_list->used;
    v_list->used = v_list->used->next;

    f_rbclear(f_rchild(p->link[0]));
    delete p->link[0];

    f_rbclear(f_rchild(p->link[1]));
    delete p->link[1];

    if (p->copy != nil)
    {
      f_rbclear(f_rchild(p->copy));
      delete p->copy;
     }

    c_rbclear(c_rchild(p->red));
    delete p->red;

    if (isleaf(p))
    { clear_key(p->key);
      clear_inf(p->info);
     }

    delete p;
   }

   ver_node q;
   forall(q, v_list->vl) delete q;

   delete v_list;
 }  


NODE*  pers_rb_tree::lookup(void *val,version v)
{ NODE* p = search(val,v);
  return (p && cmp_keys(key(p), val) == 0) ? p : nil;
 }

NODE* pers_rb_tree::locate(void *val,version v)
{ NODE* p = search(val,v);
  return ( p && cmp_keys(key(p), val) >= 0) ?  p : nil;
 }

NODE* pers_rb_tree::locate_pred(void *val,version v)
{ NODE* p = search(val,v);
  if (p==0) return nil;
  if (cmp_keys(key(p), val) <= 0)  return p;
  return child(0,p,v);

}




void pers_rb_tree::draw(DRAW_NODE_FCT draw_node,
                        DRAW_EDGE_FCT draw_edge, 
                        version v, NODE* r, 
                        double x1, double x2, double y, 
                        double ydist, double last_x)
{ 

  double x = (x1+x2)/2;

  if (r==NULL) return;

  if (last_x != 0) draw_edge(last_x,y+ydist,x,y);

  if (isleaf(r)) 
     draw_node(x,y,r->key);
  else
    { draw_node(x,y,get_key(r,v));
      draw(draw_node,draw_edge,v,child(0,r,v),x1,x,y-ydist,ydist,x);
      draw(draw_node,draw_edge,v,child(1,r,v),x,x2,y-ydist,ydist,x);
     }
}

void pers_rb_tree::draw(DRAW_NODE_FCT draw_node,
                        DRAW_EDGE_FCT draw_edge, 
                        version v, double x1, double x2, double y, double ydist)
{ draw(draw_node,draw_edge,v,v_list->vl[v]->acc_pointer,x1,x2,y,ydist,0); }
