#ifndef INTVEC_H
#define INTVEC_H
#pragma once

/*
 *	Declarations for Integer vectors
 *
 *	Copyright (C) 1988, 1989.
 *
 *	Dr. Thomas Keffer
 *	Rogue Wave Associates
 *	P.O. Box 85341
 *	Seattle WA 98145-1341
 *
 *	Permission to use, copy, modify, and distribute this
 *	software and its documentation for any purpose and
 *	without fee is hereby granted, provided that the
 *	above copyright notice appear in all copies and that
 *	both that copyright notice and this permission notice
 *	appear in supporting documentation.
 *	
 *	This software is provided "as is" without any
 *	expressed or implied warranty.
 *
 *
 *	@(#)IntVec.h	2.2	9/18/89
 */

/*	This code is designed to be as compatible as possible with the
 *	NIH Vector classes, while preserving efficiency.  These Vectors
 *	are NOT based on the NIH "Object" class, making them much
 *	smaller.  They also implement reference counting, making them
 *	faster. 
 */


#include "vdefs.h"
#include <math.h>

class istream;
class ostream;
class IntVec;
class DoubleVec;
class FloatVec;

class IntBlock {
  unsigned short  	refs;		// Number of references
  unsigned  		npts;		// Number of elements
  int    		array[1];	// The data
  friend		IntVec;
public:
  IntBlock(unsigned n);
  IntBlock(unsigned n, int val);
  IntBlock(unsigned n, int val, int by);
  ~IntBlock();

  void			add_reference()	{refs++;}
  unsigned		references()	{return refs;}
  int*			data()		{return array;}
};

class IntVec {
  IntBlock*		block;
  int*			begin;
  unsigned		npts;
  int			step;

  static int		numberPerLine; // For printing
  IntVec(IntVec&, int, unsigned, int); // For slices
protected:
  void			boundsErr(int);
  void			boundsCheck(int);
  void			lengthErr(int);
  void			lengthCheck(int i)	{if(npts!=i) lengthErr(i);}
  void			emptyErr(const char* fname);
  void			sliceErr(unsigned, int, unsigned, int);
public:
  IntVec();
  IntVec(unsigned n);
  IntVec(unsigned n, int val);
  IntVec(unsigned n, int val, int by);
  IntVec(const IntVec& a);
  IntVec(const int* dat, unsigned n);  // Copy of dat will be made
  ~IntVec();
  
  // Conversion:
  operator		DoubleVec();	// Convert to DoubleVec
  operator		FloatVec();	// Convert to FloatVec

  IntVec		slice(int start, unsigned lgt, int strider=1);
  
  int*			data()		{return begin;}
  unsigned		length()	{return npts;}
  int			stride()	{return step;}

  IntVec&		reference(IntVec& v);	// Reference self to v
  IntVec		deepCopy();	// copy of self with distinct instance variables 
  IntVec		copy()		{return deepCopy();} // Synonym for deepCopy()
  void			deepenShallowCopy();	// Insures only 1 reference to data
  void			resize(unsigned);	// Will pad with zeroes if necessary

  void			scanFrom(istream& s); // Read to eof or delimit with []
  void			printOn(ostream& s);  // Pretty print
  void			setFormatting(int);   // Change # items per line
  void			readFrom(istream&);   // Internal ASCII formatting
  void			storeOn(ostream&);
  void			readFrom(fileDescTy&);// Internal binary formatting
  void			storeOn(fileDescTy&);
  
  // Indexing:
  int&			operator[](int i);	// With bounds checking
  int&			operator()(int i);	// With optional bounds checking
  
  // Assignment:
  IntVec&		operator=(const IntVec& v); // Must be same length as v
  IntVec&		operator=(int);
  
  // Arithmetic operators:
  IntVec&		operator++();
  IntVec&		operator--();
  IntVec&		operator+=(const IntVec&);
  IntVec&		operator+=(int);
  IntVec&		operator-=(const IntVec&);
  IntVec&		operator-=(int);
  IntVec&		operator*=(const IntVec&);
  IntVec&		operator*=(int);
//IntVec&		operator/=(const IntVec&);
//IntVec&		operator/=(int);
  
  // Friendly arithmetic operators:
  friend IntVec	operator-(const IntVec&);
  friend IntVec	operator+(const IntVec&);
  friend IntVec	operator*(const IntVec&,const IntVec&);
//friend IntVec	operator/(const IntVec&,const IntVec&);
  friend IntVec	operator+(const IntVec&,const IntVec&);
  friend IntVec	operator-(const IntVec&,const IntVec&);
  friend IntVec	operator*(const IntVec&,int);
  friend IntVec	operator*(int,const IntVec&);
//friend IntVec	operator/(const IntVec&,int);
//friend IntVec	operator/(int,const IntVec&);
  friend IntVec	operator+(const IntVec&,int);
  friend IntVec	operator+(int,const IntVec&);
  friend IntVec	operator-(const IntVec&,int);
  friend IntVec	operator-(int,const IntVec&);
  
  
#ifndef NO_VECTOR_MATHFUN
  // Math functions:
  IntVec	apply(mathFunTy);
  friend	IntVec	abs(const IntVec&);
//friend	IntVec	acos(const IntVec&);
//friend	IntVec	asin(const IntVec&);
//friend	IntVec	atan(const IntVec&);
//friend	IntVec	atan2(const IntVec&,const IntVec&);
//friend	IntVec	ceil(const IntVec&);
//friend	IntVec	cos(const IntVec&);
//friend	IntVec	cosh(const IntVec&);
  friend	IntVec	cumsum(const IntVec&);
  friend	IntVec	delta(const IntVec&);
  friend	int	dot(const IntVec&,const IntVec&);
//friend	IntVec	exp(const IntVec&); 
//friend	IntVec	floor(const IntVec&);
//friend	IntVec	log(const IntVec&);
  friend	int	max(const IntVec&);
  friend	int	min(const IntVec&);
//friend	int	mean(const IntVec&);
  friend	int	prod(const IntVec&);
//friend	IntVec	pow(const IntVec&,const IntVec&);
  friend	IntVec	reverse(const IntVec&);
//friend	IntVec	rint(const IntVec&);
//friend	IntVec	sin(const IntVec&);
//friend	IntVec	sinh(const IntVec&);
//friend	IntVec	sqrt(const IntVec&);
  friend	int	sum(const IntVec&);
//friend	IntVec	tan(const IntVec&);
//friend	IntVec	tanh(const IntVec&);
//friend	int	variance(const IntVec&);

// Int specific functions:
  friend	IntVec	trunc(const DoubleVec&);// Truncate DoubleVec to IntVec
#endif
  
};

// Other (related) declarations:
ostream&	operator<<(ostream&, const IntVec&);
istream&	operator>>(istream&, IntVec&);

/******************* I N L I N E S **************************/

Inline void	IntVec::setFormatting(int i){numberPerLine = i;}

Inline void	IntVec::boundsCheck(int i){
  if(i<0 || i>npts) boundsErr(i);
}
Inline int&	IntVec::operator[](int i){
  boundsCheck(i); return begin[i*step];
}
Inline int&	IntVec::operator()(int i) {
#if BOUNDS_CHECK    
  boundsCheck(i);
#endif
  return begin[i*step];
}

Inline IntVec	operator+(const IntVec& a)		{return a;}
Inline IntVec	operator*(int a, const IntVec& b)	{return b*a;}
Inline IntVec	operator+(int a, const IntVec& b)	{return b+a;}

#endif
