/*
 *	Definitions for <A>GEMatrix I/O
 *
 *	Copyright (C) 1988, 1989.
 *
 *	Dr. Thomas Keffer
 *	Rogue Wave Associates
 *	P.O. Box 85341
 *	Seattle WA 98145-1341
 *
 *	Permission to use, copy, modify, and distribute this
 *	software and its documentation for any purpose and
 *	without fee is hereby granted, provided that the
 *	above copyright notice appear in all copies and that
 *	both that copyright notice and this permission notice
 *	appear in supporting documentation.
 *	
 *	This software is provided "as is" without any
 *	expressed or implied warranty.
 *
 *
 *	@(#)xgematio.cc	2.2	9/18/89
 */

#define NO_VECTOR_MATHFUN
#include "rw/<A>GEMatrix.h"
#include <stream.h>
#include <ctype.h>

static const char SCCSid[] = "@(#)xgematio.cc	2.2 9/18/89";

static char typeID[] = "<A>GEMat2.2";

void
<A>GEMatrix::assertRowRange(int i)
{
  if(i < 0 || i >= nrows){
    char msg[120];
    sprintf(msg, "Row index (%d) out of range [0->%d].", i, nrows-1);
    RWnote("<A>GEMatrix::assertRowRange()", msg);
    RWerror(DEFAULT);
  }
}

void
<A>GEMatrix::assertColRange(int j)
{
  if(j < 0 || j >= ncols){
    char msg[120];
    sprintf(msg, "Column index (%d) out of range [0->%d].", j, ncols-1);
    RWnote("<A>GEMatrix::assertColRange()", msg);
    RWerror(DEFAULT);
  }
}

void
<A>GEMatrix::assertRowCol(const <A>GEMatrix& m)
{
  if(m.nrows != nrows || m.ncols != ncols){
    char msg[120];
    sprintf(msg, "Rows/Columns do not match: (%d, %d) versus (%d, %d)",
	    nrows, ncols, m.nrows, m.ncols);

    RWnote("<A>GEMatrix::assertRowCol()", msg);
    RWerror(DEFAULT);
  }
}

void
<A>GEMatrix::assertLength(const <T>Vec& v)
{
  if(nrows*ncols != v.length()){
    char msg[120];
    sprintf(msg, "Vector length (%d) does not match rows and columns (%d, %d)",
	    v.length(), nrows, ncols);

    RWnote("<A>GEMatrix::assertLength()", msg);
    RWerror(DEFAULT);
  }
}

void
<A>GEMatrix::assertSquare()
{
  if(nrows != ncols){
    char msg[120];
    sprintf(msg, "Matrix (%d by %d) is not square", nrows, ncols);
    RWnote("<A>GEMatrix::assertSquare()", msg);
    RWerror(DEFAULT);
  }
}

void
<A>GEMatrix::assertProduct(const <A>GEMatrix& m)
{
  if(ncols!=m.nrows){
    char msg[120];
    sprintf(msg, "Inner product not possible between (%d by %d) and (%d by %d)",
	    nrows, ncols, m.nrows, m.ncols);
    RWnote("<A>GEMatrix::assertProduct()", msg);
    RWerror(DEFAULT);
  }
}

void
<A>GEMatrix::assertProduct(const <T>Vec& v)
{
  if(ncols!=v.length()){
    char msg[120];
    sprintf(msg, "Inner product not possible between matrix %d by %d and vector %d points long",
	    nrows, ncols, v.length());
    RWnote("<A>GEMatrix::assertProduct()", msg);
    RWerror(DEFAULT);
  }
}

// As written scanFrom makes 2 temporary copies of the data, one to read it
// in and one to transpose it.  This is the only way (I think) if we want
// to avoid unnecessary de-aliasing.  C'est la vie.  If the matrix is big
// enough that this matters use readFrom and storeTo.

void
<A>GEMatrix::scanFrom(istream& s)
{
  unsigned numRows, numCols;	    // Read number rows/cols from stream
  s >> numRows;
  char c;	// Read the (optional) x character
  s >> c;
  if (isdigit(c))
    s.putback(c);
  s >> numCols;

  <T>Vec v;		    // Read the data transposed (ie in row major order)
  v.scanFrom(s);
  <A>GEMatrix transposed(v,numCols,numRows);

  resize(numRows,numCols);  // The transpose of "transposed" is what we want
  (*this) = transpose(transposed);
}

void
<A>GEMatrix::printOn(ostream& s)
{
  for (register int i=0; i< rows(); i++){
    for (register int j = 0; j < cols(); j++) s << (*this)(i,j) << " ";
    s << NL;
  }
}

ostream&
operator<<(ostream& s, const <A>GEMatrix& m)
{
  m.printOn(s); return s;
}

istream&
operator>>(istream& s, <A>GEMatrix& m)
{
  m.scanFrom(s); return s;
}

void
<A>GEMatrix::readFrom(istream& s)
{
  RWreadID(s,typeID,"<A>GEMatrix::readFrom");
  s >> nrows >> ncols;
  <T>Vec::readFrom(s);
  assertLength(*this);	// Check dimensions
}

void
<A>GEMatrix::storeOn(ostream& s)
{
  RWstoreID(s,typeID,"<A>GEMatrix::storeOn");
  s << nrows << " " << ncols << "\n";
  <T>Vec::storeOn(s);
}

void 
<A>GEMatrix::readFrom(fileDescTy& fd)
{
  static const char routine[] = "<A>GEMatrix::readFrom";
  RWreadID(fd,typeID,routine);
  RWreadBin(fd,&nrows,sizeof(unsigned),routine);
  RWreadBin(fd,&ncols,sizeof(unsigned),routine);
  <T>Vec::readFrom(fd);
  assertLength(*this);	// Check dimensions
}

void
<A>GEMatrix::storeOn(fileDescTy& fd)
{
  static const char routine[] = "<A>GEMatrix::storeOn";
  RWstoreID(fd,typeID,routine);
  RWstoreBin(fd,&nrows,sizeof(unsigned),routine);
  RWstoreBin(fd,&ncols,sizeof(unsigned),routine);
  <T>Vec::storeOn(fd);
}
