(*  :Title:	Tree manipulation routines  *)

(*  :Authors:	Brian Evans, James McClellan  *)

(*
    :Summary:	Extend tree manipulation abilities in Mathematica
		(see also the standard package DiscreteMath`Tree`)
 *)

(*  :Context:	SignalProcessing`Support`Tree`  *)

(*  :PackageVersion:  2.4	*)

(*
    :Copyright:	Copyright 1989-1991 by Brian L. Evans
		Georgia Tech Research Corporation

	Permission to use, copy, modify, and distribute this software
	and its documentation for any purpose and without fee is
	hereby granted, provided that the above copyright notice
	appear in all copies and that both that copyright notice and
	this permission notice appear in supporting documentation,
	and that the name of the Georgia Tech Research Corporation,
	Georgia Tech, or Georgia Institute of Technology not be used
	in advertising or publicity pertaining to distribution of the
	software without specific, written prior permission.  Georgia
	Tech makes no representations about the suitability of this
	software for any purpose.  It is provided "as is" without
	express or implied warranty.
 *)

(*  :History:	*)

(*  :Keywords:	tree data structure	*)

(*  :Source:	*)

(*  :Warning:	*)

(*  :Mathematica Version:  1.2 or 2.0  *)

(*  :Limitation:  *)

(*
    :Discussion:	Trees are represented a list of lists.

			a0 ---> b1           -> g3
			    |               |
			     -> c1 ---> d2 ---> f3
				    |
				     -> e2

			would be represented as

			{ a0, b1, {c1, {d2, f3, g3}, e2} }

		See also the standard package DiscreteMath`Tree`.
 *)

(*  :Functions:  AddChildToTree DeleteFromTree InitTree SubTree  *)


If [ TrueQ[ $VersionNumber >= 2.0 ],
     Off[ General::spell ];
     Off[ General::spell1 ] ]


(*  B E G I N     P A C K A G E  *)

BeginPackage [ "SignalProcessing`Support`Tree`" ]


(*  U S A G E     I N F O R M A T I O N  *)

AddChildToTree::usage =
	"AddChildToTree[tree, parent, newchild] adds newchild under \
	every parent in tree."

DeleteFromTree::usage =
	"DeleteFromTree[tree, node] deletes all nodes with info/name of node. \
	If the node is a parent, then the entire subtree is pruned."

InitTree::usage =
	"InitTree[root] returns an empty tree with a root of root."

SubTree::usage =
	"SubTree[tree, head] returns the subtree with root head."

(*  E N D     U S A G E     I N F O R M A T I O N  *)


Begin [ "`Private`" ]


(*  M E S S A G E S  *)

AddChildToTree::empty = "Empty tree encountered."
DeleteFromTree::empty = "Empty tree encountered."


(*  B E G I N     P A C K A G E  *)

(*  AddChildToTree  *)
AddChildToTree[ tree_, parent_, newchild_ ] :=
	addchildtotree[tree, parent, newchild]

addchildtotree[ tree_, parent_, newchild_ ] :=
	Replace[ add[tree, parent, newchild], addchildrules ] 

addchildrules = {
	add[{}, parent_, newchild_] :> Message[ AddChildToTree::empty ],

	add[parent_, parent_, newchild_] :> { parent, newchild },

	add[List[parent_], parent_, newchild_] :> { parent, newchild },

	add[List[parent_, rest__], parent_, newchild_] :>
		{ parent, newchild } ~Join~
		Map[ addchildtotree[#, parent, newchild]&, {rest} ],

	add[List[other_, rest__], parent_, newchild_] :>
		{ other } ~Join~
		  Map[ addchildtotree[#, parent, newchild]&, {rest} ] /;
		! SameQ[other, parent],

	add[x_, parent_, newchild_] :> x
}

(*  DeleteFromTree *)
(*    replace all deleted sections by an empty list {} to get new.  *)
(*    use Complement to sort tree and remove all {}'s.		    *)
DeleteFromTree[ tree_, node_ ] :=
	If [ SameQ[node, First[tree]],
	     { First[tree] } ~Join~ deletefromtree[Rest[tree], node],
	     deletefromtree[tree, node] ]

deletefromtree[ tree_, node_ ] :=
	     Replace[ delete[tree, node], deletenoderules]

deletenoderules = {
	delete[{}, node_] :> Message[ DeleteFromTree::empty ],

	delete[node_, node_] :> {},
	delete[List[node_], node_] :> {},
	delete[List[node_, rest__], node_ ] :> {},

	delete[List[other_, rest__], node_] :>
		{ other } ~Join~
		  Select[ Map[ deletefromtree[#, node]&, {rest} ],
			  ! SameQ[#, {}] & ] /;
		! SameQ[other, node],

	delete[x_, node_] :> x
}

(*  InitTree  *)
InitTree[ root_ ] := { root }

(*  SubTree   *)
SubTree[ tree_, head_ ] :=
	Block [	{ returntree },

		subtree[ curtree_ ] :=
			Block [	{newflag},
				newflag = If [ SameQ[Head[curtree], List],
					       SameQ[head, First[curtree]],
					       SameQ[head, curtree] ];
				If [ newflag,
				     returntree = curtree ];
				newflag ];

		returntree = {};
		Scan [ ( If [ subtree[#], Return ] ) &, tree, Infinity ];
		returntree ]


(*  E N D     P A C K A G E  *)


End[]
EndPackage[]

If [ TrueQ[ $VersionNumber >= 2.0 ],
     On[ General::spell ];
     On[ General::spell1 ] ]

Null
