; environment.scm
;
; Defines the global environment of the substitution model.
;


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                        ;;;
;;; The Global Environment ;;;
;;;                        ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;(define (1+ n) (+ n 1))  ;for portability

;(define (-1+ n) (- n 1))  ;for portability

(define (init-the-global-environment)

;  math
  `((+ <<+>>)                  (- <<->>)                 (* <<*>>)
   (/ <</>>)                   (= <<=>>)                 (< <<_<_>>) 
   (> <<_>_>>)                 (>= <<_>=_>>)             (<= <<_<=_>>)
   (1+ <<1+>>)                 (-1+ <<-1+>>)

   (sqrt <<sqrt>>)             (expt <<expt>>)           (round <<round>>)
   (abs <<abs>>)               (gcd <<gcd>>)             (max <<max>>) 
   (min <<min>>)               (zero? <<zero?>>)         (positive? <<positive?>>)
   (negative? <<negative?>>)   (odd? <<odd?>>)           (even? <<even?>>) 
   (exp <<exp>>)               (log <<log>>)             (sin <<sin>>)
   (cos <<cos>>)               (tan <<tan>>)             (asin <<asin>>)
   (acos <<acos>>)             (atan <<atan>>)           (quotient <<quotient>>)

;  logic
   (not <<not>>)       

;  predicates
   (equal? <<equal?>>)         (boolean? <<boolean?>>)   (symbol? <<symbol?>>)
   (char? <<char?>>)           (pair? <<pair?>>)         (number? <<number?>>)
   (procedure? <<procedure?>>) (list? <<list?>>)
   
;  list-related
   (car <<car>>)               (cdr <<cdr>>)
   (caddr (lambda (exp) (car (cdr (cdr exp)))))
   (cadr (lambda (exp) (car (cdr exp))))
   (cons <<cons>>)
   (list <<list>>)             (null? <<null?>>)         (length <<length>>)   
   (append <<append>>)         (member <<member>>)       (assoc <<assoc>>)
   (map <<map>>)               (nil ,nil-builtin)
   (apply <<apply>>)
   (reverse (lambda (lst)
              (if
               (<<null?>> lst)
               ,nil-builtin
               (<<append>> (reverse (<<cdr>> lst)) (<<list>> (<<car>> lst))))))
   (delete (lambda (sym lst)
             (cond ((<<null?>> lst) ,nil-builtin)
                   ((<<equal?>> (<<car>> lst) sym)
                    (delete sym (<<cdr>> lst)))
                   (else
                    (<<cons>> (<<car>> lst) (delete sym (<<cdr>> lst)))))))
   (list-tail <<list-tail>>)
   (list-ref (lambda (lst index)
               (<<car>> (<<list-tail>> lst index))))

   ;  stringops

   (string? <<string?>>)       (make-string <<make-string>>)
   (string <<string>>)         (string-length <<string-length>>)
   (string-ref <<string-ref>>) (string=? <<string=?>>)   (string<? <<string<?>>)
   (string>? <<string>?>>)     (string<=? <<string<=?>>) (string>=? <<string>=?>>)
   (string-ci=? <<string-ci=?>>)    (string-ci<? <<string-ci<?>>)
   (string-ci>? <<string-ci>?>>)    (string-ci<=? <<string-ci<=?>>)
   (string-ci>=? <<string-ci>=?>>)
   (substring <<substring>>)   (string-append <<string-append>>)
   (string->list <<string->list>>)  (list->string <<list->string>>)
   (number->string <<number->string>>)  (symbol->string <<symbol->string>>)

   ;  side-fx
   (display <<display>>)       (newline <<newline>>)     (error <<error>>)))

(define the-global-environment (copy (init-the-global-environment)))

; The following definitions are used to define our
; primitive procedure objects (denoted by angle brackets)

(define <<list>> list)
(define <<list?>> list?)
(define <<cons>> cons)
(define <<car>> car)
(define <<cdr>> cdr)
(define <<apply>> apply)
(define (<<equal?>> x y)
	   (if (number? x)
               (= x y))
           (equal? x y))
(define <<map>> map)
(define <<member>> member)
(define <<append>> append)
(define <<delete>> delete)
(define <<reverse>> reverse)
(define <<list-tail>> list-tail)
(define <<list-ref>> list-ref)
(define <<assoc>> assoc)
(define <<error>> (lambda (message) (error message)))
(define <<display>> display)
(define <<newline>> newline)
(define <<load>> load)
(define <<import>> load)
(define <<+>> +)
(define <<->> -)
(define <<*>> *)
(define <</>> /)
(define <<=>> =)
(define <<_<_>> <)
(define <<_>_>> >)
(define <<_>=_>> >=)
(define <<_<=_>> <=)
(define <<1+>> 1+)
(define <<-1+>> -1+)
(define <<quotient>> quotient)
(define <<sqrt>> sqrt)
(define <<expt>> expt)
(define <<round>> round)
(define <<abs>> abs)
(define <<gcd>> gcd)
(define <<max>> max)
(define <<min>> min)
(define <<zero?>> zero?)
(define <<positive?>> positive?)
(define <<negative?>> negative?)
(define <<odd?>> odd?)
(define <<even?>> even?)
(define <<exp>> exp)
(define <<log>> log)
(define <<sin>> sin)
(define <<cos>> cos)
(define <<tan>> tan)
(define <<asin>> asin)
(define <<acos>> acos)
(define <<atan>> atan)
(define <<not>> not)
(define <<eqv?>> eqv?)
(define <<eq?>> eq?)
(define <<boolean?>> boolean?)
(define <<symbol?>> symbol?)
(define <<char?>> char?)
(define <<number?>> number?)
(define <<procedure?>> procedure?)      
(define <<null?>> null?)
(define <<pair?>> pair?)
(define <<length>> length)
(define <<string?>> string?)
(define <<make-string>> make-string)
(define <<string>> string)
(define <<string-length>> string-length)
(define <<string-ref>> string-ref)
(define <<string=?>> string=?)
(define <<string<?>> string<?)
(define <<string>?>> string>?)
(define <<string<=?>> string<=?)
(define <<string>=?>> string>=?)
(define <<string-ci=?>> string-ci=?)
(define <<string-ci<?>> string-ci<?)
(define <<string-ci>?>> string-ci>?)
(define <<string-ci<=?>> string-ci<=?)
(define <<string-ci>=?>> string-ci>=?)
(define <<substring>> substring)
(define <<string-append>> string-append)
(define <<string->list>> string->list)
(define <<list->string>> list->string)
(define <<number->string>> number->string)
(define <<symbol->string>> symbol->string)

;;; RULE-DEFINED LIST OPERATIONS
;;; These operators mimic certain primitive list
;;; operators, necessitated by our different
;;; representations of lists and cons pairs.

(define (rule-defined-listop? exp)
  (memq (car exp) rule-defined-listops))

(define rule-defined-listops
  '(<<cons>> <<list>> <<car>> <<cdr>> <<apply>>
    <<list?>> <<equal?>> <<append>> <<map>> <<member>> <<list-tail>>))
;;;REVISE add ASSOC, LIST-REF, DELETE, REVERSE

(define (apply-rule-defined-listop exp) ;REVISE: create an APPLY-RULE procedure for all these 
  (case (car exp)
    ((<<cons>>) (applycons exp))
    ((<<list>>) (applylist exp))
    ((<<car>>) (applycar exp))
    ((<<cdr>>) (applycdr exp))
    ((<<apply>>) (applyapply exp))
    ((<<list?>>) (applylist? exp))
    ((<<equal?>>) (applyequal? exp))
    ((<<append>>) (applyappend exp))
    ((<<map>>) (applymap exp))
    (<<member>> (applymember exp))
    (<<list-tail>> (applylist-tail exp))))

; CONS-APPLY and LIST-APPLY

(define (consexp? exp) 
  (tagged-list? exp '<<cons>>))

(define (listexp? exp)
  (tagged-list? exp '<<list>>))

(define (applycons cons-exp)
  (set! display-variable #f)
  (cond ((not (= 3 (length cons-exp)))
	 (sm-error "ERROR: <<cons>> takes 2 args"
		cons-exp))
	((listexp? (caddr cons-exp))
	 (set! display-variable #t)
	 (cons '<<list>> (cons (cadr cons-exp) (cdaddr cons-exp))))
	((nil-builtin? (caddr cons-exp))
	 (set! display-variable #t)
	 (list '<<list>> (cadr cons-exp)))
	(else cons-exp)))

(define (applylist list-exp)
  (set! display-variable #f)
  (if (equal? list-exp '(<<list>>))
      nil-builtin
      list-exp))

; CAR AND CDR

(define (applycar exp)
  (set! display-variable #f)
  (let ((arg (cadr exp)))
    (if (or (consexp? arg) (listexp? arg))
        (cadr arg)
        (sm-error "ERROR: <<car>> arg not a pair" arg))))

(define (applycdr exp)
  (set! display-variable #f)
  (let ((arg (cadr exp)))
    (cond ((consexp? arg)
           (caddr arg))
          ((listexp? arg)
           (if (null? (cddr arg))
               nil-builtin
               (cons '<<list>> (cddr arg))))
          (else (sm-error "ERROR: <<cdr>> arg not a pair" arg)))))

(define (applyequal? exp)
  (set! display-variable #f)
  (if (not (= (length exp) 3))
      (sm-error "ERROR: <<equal?>> takes 2 args" exp)
      (let ((x (cadr exp))
            (y (caddr exp)))
        (cond ((and (number? x)
                    (number? y))
               (= x y))
              (else (equal? x y))))))


; APPLY

(define (applyapply exp)
  (if (= (length exp) 3)
      (let ((arglist (arglist-apply exp)))
        (if (listexp? arglist)
            (cons (procedure-apply exp) (cdr arglist))
            (sm-error "ERROR: <<apply>> nonlist 2nd arg" arglist)))
      (sm-error "ERROR: <<apply>> takes 2 args" exp)))

;LIST?

(define (applylist? exp)
  (set! display-variable #f)
  (if (= (length exp) 2)
      (or (nil-builtin? (cadr exp))
          (listexp? (cadr exp)))
      (sm-error "ERROR: <<list?>> takes 1 arg" exp)))


;APPEND

(define (applyappend exp)
  (let ((arg-list (arglist-append exp)))
    (if (for-all? arg-list nil-builtin?)
        nil-builtin
        (if (for-all? arg-list
                      (lambda (arg)
                        (or (nil-builtin? arg) (listexp? arg))))
            (cons '<<list>>
                  (apply
                   append
                   (map cdr (delete '() arg-list))))
            (sm-error "ERROR: <<append>> arg not a list :" arg-list)))))

;MAP

(define (applymap exp)
  (if (= (length exp) 3)
      (let ((arglist (caddr exp)))
        (cond ((nil-builtin? arglist) nil-builtin)
              ((listexp? arglist)
               (let ((proc (cadr exp)))
                 (cons '<<list>> (map (lambda (arg) (list proc arg)) (cdr arglist)))))
              (else (sm-error "ERROR: <<map>> nonlist 2nd arg" arglist))))
      (sm-error "ERROR: <<map>> takes 2 args" exp)))

;LIST-TAIL

(define (applylist-tail exp)
  (if (not (= (length exp) 3))
      (sm-error "ERROR: <<list-tail>> takes 2 args" exp)
      (let ((lst (cadr exp))
            (k (caddr exp)))
        (cond ((zero? k) (set! display-variable #f) lst)
              ((< k 0)
               (sm-error "ERROR: <<list-tail>> illegal negative 2nd arg" exp))
              ((nil-builtin? lst)
               (sm-error "ERROR: <<list-tail>> 2nd arg exceeds length of 1st arg" exp))
              ((listexp? lst)
               (set! display-variable #t)
               (letrec ((aux (lambda (l n)
;                               (display "l: ")(display l)(newline)
;                               (display "n: ")(display n)(newline)
                               (if (zero? n)
                                   (if (null? l)
                                       nil-builtin
                                       (cons '<<list>> l))
                                   (if
                                    (null? l)
                                    (sm-error
                                     "ERROR: <<list-tail>> 2nd arg exceeds length of 1st arg"
                                     exp)
                                    (aux (cdr l) (-1+ n)))))))
                 (aux (cdr lst) k)))
              (else
               (sm-error "ERROR: <<list-tail>> 1st arg not a list" exp))))))

;MEMBER

(define (applymember exp)
  (if (= (length exp) 3)
      (let ((arglist (caddr exp)))
        (if (nil-builtin? arglist)
            nil-builtin
            (if (listexp? arglist)
                (let* ((obj (cadr exp))
                       (resultlist (member obj (cdr arglist))))
                  (if resultlist
                      (cons '<<list>> resultlist)
                      nil-builtin))
                (sm-error "ERROR: <<member>> nonlist 2nd arg" arglist))))
      (sm-error "ERROR: <<member>> takes 2 args" exp)))

; ANGLED-PRIMITIVE? checks an expression to see if it has the syntax
; of our model's system constants
;
(define (angled-primitive? exp)
  (and (symbol? exp)
       (let* ((str (symbol->string exp))
	      (len (string-length str)))
	 (and (string=? (string-tail str (- len 2)) ">>")
	      (string=? (substring str 0 2) "<<")))))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;                                                                      ;
; lookup a variable in the global environment                          ;
;                                                                      ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define (lookup var)
  (let ((binding (assq var the-global-environment)))
    (if binding
        (cadr binding)
        (sm-error "Variable unbound in SM-global-env" var))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;                                                                      ;
; Change or add a binding to the global environments.                  ;
;                                                                      ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define (do-bindings def-variable def-value)
  (let ((assn1 (assq def-variable the-global-environment)))
    (if assn1
        (set-car! (cdr assn1) def-value)
        (let ((binding (list def-variable def-value)))
          (set! the-global-environment
                (cons binding the-global-environment))))))
