/****************************************************************************
**
**    consistency.c                   NQ                       Werner Nickel
**
**    Copyright 1992                            Mathematics Research Section
**                                           School of Mathematical Sciences 
**                                            Australian National University
*/

#include "presentation.h"
#include "pc.h"
#include "pcarith.h"
#include "collect.h"
#include "macro.h"

/*
**    Define macros for the different types of consistency checks.
**    b' denotes the inverse of b. If the polycyclic presentation
**    is weighted, then consistency checks have to be performed only
**    for those tuples that satisfy a weight condition.
*/
#define	CBA 1  /* c (b a)  = (c b)  a    for c > b > a                   */
#define	BAA 2  /* b (a' a) = (b a') a    for b > a, Exponent[a] == 0     */
#define	BBA 3  /* b' (b a) = (b' b) a    for b > a, Exponent[b] == 0     */
               /* b'(b a') = (b' b) a'   for b > a, Exponent[b] == 0     */
	       /*				    Exponent[a] == 0     */
#define	BAN 4  /* b a^n = (b a) a^(n-1)  for b > a, Exponent[a] == n > 0 */
#define	BNA 5  /* b^n a = b^(n-1) (b a)  for b > a, Exponent[b] == n > 0 */
#define	ANA 6  /* a^n a = a a^n                     Exponent[a] == n > 0 */

static	word	*Generator;
static	int	Confluent;
static	int	AbelianInconsistency;
static	void	(*ProcessResult)();

static	void	PrintExpVec( ev )
expvec	ev;

{ 	gen	g;

	InitPrint( stdout );
	for( g = 1; g <= NrPcGens+NrCenGens; g++ )
		if( ev[ g ] != 0 ) break;

	if( g > NrPcGens+NrCenGens ) { printf( "1" ); return; }

	PrintGen( g );
	if( ev[g] != 1 ) printf( "^%d", ev[g] );
	g++;
	
	for( ; g <= NrPcGens+NrCenGens; g++ )
	    if( ev[g] != 0 ) {
		putchar( '*' );
		PrintGen( g );
		if( ev[g] != 1 ) printf( "^%d", ev[g] );
	    }
}

void	PrintConCheck( which, c, b, a )
int	which;
gen	c, b, a;

{	switch( which ) {
	    case CBA: if( a > (gen)0 )
			  printf( "%s (%s %s) = (%s %s) %s",
			           GenName(c), GenName(b), GenName(a),
			           GenName(c), GenName(b), GenName(a) );
		      else
			  printf( "%s (%s %s') = (%s %s) %s'",
			           GenName(c), GenName(b), GenName(-a),
			           GenName(c), GenName(b), GenName(-a) );
		      break;
	    case BAA: printf( "%s (%s' %s) = (%s %s') %s",
			       GenName(b), GenName(a), GenName(a),
			       GenName(b), GenName(a), GenName(a) );
		      break;
	    case BBA: if( a > (gen)0 ) 
	                  printf( "%s' (%s %s) = (%s' %s) %s",
			           GenName(b), GenName(b), GenName(a),
			           GenName(b), GenName(b), GenName(a) );
		      else
	                  printf( "%s' (%s %s') = (%s' %s) %s'",
			           GenName(b), GenName(b), GenName(-a),
			           GenName(b), GenName(b), GenName(-a) );
		      break;
	    case BAN: printf( "%s %s^%d = (%s %s) %s",
			       GenName(b), GenName(a), Exponent[a],
			       GenName(b), GenName(a), GenName(a) );
		      if( Exponent[a] > 2 ) printf( "^%d", Exponent[a]-1 );
		      break;
	    case BNA: printf( "%s^%d %s = %s",
			       GenName(b), Exponent[b], GenName(a),
			       GenName(b) );
		      if( Exponent[b] > 2 ) printf( "^%d", Exponent[b]-1 );
		      printf( " (%s %s)", GenName(b), GenName(a) );
		      break;
	    case ANA: printf( "%s^%d %s = %s %s^%d",
			      GenName(a), Exponent[a], GenName(a),
			      GenName(a), GenName(a), Exponent[a] );
		      break;
	    default:  printf( "Screwed up first argument.\n" );
		      break;
	}
}

void	PrintInconsistency( result, which, c, b, a )
int	which;
gen	c, b, a;
void	*result;

{	if( Confluent )
	    printf( "Presentation is not confluent.\n" );

	PrintConCheck( which, c, b, a );
	printf( " : " );

	if( AbelianInconsistency ) PrintExpVec( (expvec)result );
	else			   PrintPcWord( (word)result );
	printf( "\n" );
}

static	void	DoCBA( c, b, a )
gen	c, b, a;

{	int	i, nonZero = 0;
	expvec	ev1, ev2;
	word	w1, w2, w3;
	void	*tmp;

	/* the left hand side first : c (b a) = c a b^a */
	ev1 = ExpVecWord( Generator[c] );
	Collect( ev1, Generator[a], 1 );
	if( b <= Commute[abs(a)] ) Collect( ev1, Conjugate[b][a], 1 );
	else                       Collect( ev1, Generator[b], 1 );

	/* then the right hand side : (c b) a */
	ev2 = ExpVecWord( Generator[c] );
	Collect( ev2, Generator[b], 1 );
	Collect( ev2, Generator[a], 1 );

	if( AbelianInconsistency ) {
	    for( i = 1; i <= NrPcGens+NrCenGens; i++ ) 
		if( (ev2[i] -= ev1[i]) != (exp)0 ) nonZero = 1;
	    Free( ev1 );
	    tmp = (void *)ev2;
	}
	else {
	    w1 = WordExpVec( ev1 ); Free( ev1 );
	    w2 = WordExpVec( ev2 ); Free( ev2 );
	    w3 = Solve( w1, w2 );   Free( w1 ); Free( w2 );
	    nonZero = (w3->g != EOW);
	    tmp = (void *)w3;
	}
	if( nonZero ) {
	    (*ProcessResult)( tmp, CBA, c, b, a );
	    Confluent = 0;
	}
	Free( tmp );
}

static	void	DoBAA( b, a )
gen	b, a;

{	int	i, nonZero = 0;
	expvec	ev;
	word	w, w2;
	void	*tmp;

	/* (b a^-1) a */
	ev = ExpVecWord( Generator[b] );
	Collect( ev, Generator[-a], 1 );
	Collect( ev, Generator[ a], 1 );

	/* Divide by b. */
	if( AbelianInconsistency ) {
	    ev[ b ] -= 1;
	    for( i = 1; i <= NrPcGens+NrCenGens; i++ ) 
		if( ev[i] != (exp)0 ) nonZero = 1;
	    tmp = (void *)ev;
	}
	else {
	    w  = WordExpVec( ev );         Free( ev );
	    w2 = Solve( Generator[b], w ); Free( w  );
	    nonZero = (w2->g != EOW);
	    tmp = (void *)w2;
	}
	if( nonZero ) {
	    (*ProcessResult)( tmp, BAA, (gen)0, b, a );
	    Confluent = 0;
	}

	Free( tmp );
}

static	void	DoBBA( b, a )
gen	b, a;

{	int	i, nonZero = 0;
	expvec	ev;
	word	w, w2;
	void	*tmp;

	/* a can be negative! */

	/* b^-1 (b a) = b^-1 a b^a */
	ev = ExpVecWord( Generator[-b] );
	Collect( ev, Generator[a], 1 );
	if( b <= Commute[abs(a)] ) Collect( ev, Conjugate[b][a], 1 );
	else                       Collect( ev, Generator[b], 1 );

	/* Divide by a. */
	if( AbelianInconsistency ) {
	    ev[abs(a)] -= sgn(a);
	    for( i = 1; i <= NrPcGens+NrCenGens; i++ ) 
		if( ev[i] != (exp)0 ) nonZero = 1;
	    tmp = (void *)ev;
	}
	else {
	    w = WordExpVec( ev );          Free( ev );
	    w2 = Solve( Generator[a], w ); Free( w  );
	    nonZero = (w2->g != EOW);
	    tmp = (void *)w2;
	}
	if( nonZero ) {
	    (*ProcessResult)( tmp, BBA, (gen)0, b, a );
	    Confluent = 0;
	}

	Free( tmp );
}

static	void	DoBAN( b, a )
gen	b, a;

{	int	i, nonZero = 0;
	expvec	ev1, ev2;
	word	w1, w2, w3;
	void	*tmp;

	/* b (a^n) */
	ev1 = ExpVecWord( Generator[b] );
	if( Power[a] != (word)0 ) Collect( ev1, Power[a], 1 );

	/* (b a) a^(n-1) */
	ev2 = ExpVecWord( Generator[b] );
	Collect( ev2, Generator[a], Exponent[a] );

	if( AbelianInconsistency ) {
	    for( i = 1; i <= NrPcGens+NrCenGens; i++ ) 
		if( (ev2[i] -= ev1[i]) != (exp)0 ) nonZero = 1;
	    Free( ev1 );
	    tmp = (void *)ev2;
	}
	else {
	    w1 = WordExpVec( ev1 ); Free( ev1 );
	    w2 = WordExpVec( ev2 ); Free( ev2 );
	    w3 = Solve( w1, w2 );   Free( w1 ); Free( w2 );
	    nonZero = (w3->g != EOW);
	    tmp = (void *)w3;
	}
	if( nonZero ) {
	    (*ProcessResult)( tmp, BAN, (gen)0, b, a );
	    Confluent = 0;
	}
	Free( tmp );
}

static	void	DoBNA( b, a )
gen	b, a;

{	int	i, nonZero = 0;
	expvec	ev1, ev2;
	word	w1, w2, w3;
	void	*tmp;

	/* (b^n) a */
	ev1 = ExpVecWord( Power[b] );
	Collect( ev1, Generator[a], 1 );

	/* b^(n-1) (b a) = b^(n-1) a b^a */
	ev2 = ExpVecWord( Generator[b] ); ev2[b] = Exponent[b]-1;
	Collect( ev2, Generator[a], 1 );
	if( b <= Commute[a] ) Collect( ev2, Conjugate[b][a], 1 );
	else                  Collect( ev2, Generator[b], 1 );

	if( AbelianInconsistency ) {
	    for( i = 1; i <= NrPcGens+NrCenGens; i++ ) 
		if( (ev2[i] -= ev1[i]) != (exp)0 ) nonZero = 1;
	    Free( ev1 );
	    tmp = (void *)ev2;
	}
	else {
	    w1 = WordExpVec( ev1 ); Free( ev1 );
	    w2 = WordExpVec( ev2 ); Free( ev2 );
	    w3 = Solve( w1, w2 );   Free( w1 ); Free( w2 );
	    nonZero = (w3->g != EOW);
	    tmp = (void *)w3;
	}
	if( nonZero ) {
	    (*ProcessResult)( tmp, BNA, (gen)0, b, a );
	    Confluent = 0;
	}
	Free( tmp );
}

static	void	DoANA( a )
gen	a;

{	int	i, nonZero = 0;
	expvec	ev1, ev2;
	word	w1, w2, w3;
	void	*tmp;

	/* a^n a */
	ev1 = ExpVecWord( Power[a] );
	Collect( ev1, Generator[a], 1 );

	/* a a^n */
	ev2 = ExpVecWord( Generator[a] );
	if( Power[a] != (word)0 ) Collect( ev2, Power[a], 1 );

	if( AbelianInconsistency ) {
	    for( i = 1; i <= NrPcGens+NrCenGens; i++ ) 
		if( (ev2[i] -= ev1[i]) != (exp)0 ) nonZero = 1;
	    Free( ev1 );
	    tmp = (void *)ev2;
	}
	else {
	    w1 = WordExpVec( ev1 ); Free( ev1 );
	    w2 = WordExpVec( ev2 ); Free( ev2 );
	    w3 = Solve( w1, w2 );   Free( w1 ); Free( w2 );
	    nonZero = (w3->g != EOW);
	    tmp = (void *)w3;
	}
	if( nonZero ) {
	    (*ProcessResult)( tmp, ANA, (gen)0, (gen)0, a );
	    Confluent = 0;
	}
	Free( tmp );
}

static	void	ConsistencyA( a )
gen	a;

{	gen	b, c;

	/*
	**    c * ( b * a )  = ( c * b ) * a   for all c > b > a.
	**    c * ( b * a' ) = ( c * b ) * a'  for all c > b > a
	**                                        and Exponent[a] == 0.
	*/
	for( b = a+1; b <= NrPcGens; b++ )
	    for( c = b+1; c <= NrPcGens; c++ ) {
		if( IsWeighted && Wt(c)+Wt(b)+Wt(abs(a)) > Class+1 ) break;
		DoCBA( c, b, a );
		if( Exponent[a] == 0 ) DoCBA( c, b, -a );
	    }

	/*
	**    b * ( a' * a ) = ( b * a') * a  for all b > a, Exponent[a] == 0
	*/
	if( Exponent[a] == 0 )
	    for( b = a+1; b <= NrPcGens; b++ ) {
		if( IsWeighted && Wt(b)+Wt(a) > Class+1 ) break;
		DoBAA( b, a );
	    }

	/*
	**    b' * ( b * a ) = ( b' * b ) * a  for all b > a, Exponent[b] == 0.
	**    b' * ( b * a') = ( b' * b ) * a' for all b > a, Exponent[b] == 0,
	**                                                 if Exponent[a] == 0.
	*/
	for( b = a+1; b <= NrPcGens; b++ ) {
	    if( IsWeighted && Wt(b)+Wt(a) > Class+1 ) break;
	    if( Exponent[b] == 0 ) {
		    DoBBA( b, a );
		    if( Exponent[a] == 0 ) DoBBA( b, -a );
	    }
	}

	/*
	**    b * a^n = ( b * a ) * a^(n-1) for all b > a, Exponent[a] == n > 0.
	*/
	if( Exponent[a] > 0 )
	    for( b = a+1; b <= NrPcGens; b++ ) {
		if( IsWeighted && Wt(b)+Wt(a) > Class+1 ) break;
		DoBAN( b, a );
	    }

	/*
	**    b^n * a = b^(n-1) * ( b * a ) for all b > a, Exponent[b] == n > 0.
	*/
	for( b = a+1; b <= NrPcGens; b++ ) {
	    if( IsWeighted && Wt(b)+Wt(a) > Class+1 ) break;
	    if( Exponent[b] > 0 )
		    DoBNA( b, a );
	}

	/*
	**    a^n * a = a * a^n if Exponent[a] == n > 0.
	*/
	if( IsWeighted && 2*Wt(c) > Class+1 ) return;
	if( Exponent[a] > 0 ) DoANA( a );
}

int	Consistency( action, isAbelian )
void	(*action)();
int	isAbelian;

{	gen	a;
	int	reset = 0;

	Confluent = 1;
	AbelianInconsistency = isAbelian;
	if( action == (void (*)()) 0 )
	    ProcessResult = PrintInconsistency;
	else
	    ProcessResult = action;

	Generator = (word *)Allocate( (2*NrPcGens+1)*sizeof(word) );
	Generator += NrPcGens;
	for( a = -NrPcGens; a <= NrPcGens; a++ ) {
	    if( a == 0 ) continue;
	    Generator[a] = (word)Allocate( 2*sizeof(gpower) );
	    Generator[a][0].g = a;
	    Generator[a][0].e = 1;
	    Generator[a][1].g = EOW;
	    Generator[a][1].e = 0;
	}

	for( a = NrPcGens; a >= 1; a-- ) ConsistencyA( a );

	for( a = -NrPcGens; a <= NrPcGens; a++ ) {
	    if( a == 0 ) continue;
	    Free( Generator[a] );
	}
	Free( Generator-NrPcGens );

	if( Confluent ) {
	    if( action == (void (*)()) 0 )
		printf( "Presentation is confluent.\n" );
	    return 0;
	}
	else return 1;
}
