/*****************************************************************************
**
**    nq.c                            NQ                       Werner Nickel
**                                                    werner@pell.anu.edu.au
**
**    Copyright Dec 1992                        Mathematics Research Section
**                                           School of Mathematical Sciences 
**                                            Australian National University
*/


#include "nq.h"
#include "engel.h"

int	Debug = 0;
int	Gap = 0;
int	Verbose = 0;

static char	*ProgramName;
static char	*InputFile;
static int	Cl;

static
void	usage( error )
char	*error;

{	int	i;

	if( error != (char *)0 ) fprintf( stderr, "%s\n", error );
	fprintf( stderr, "usage: %s", ProgramName );
	fprintf( stderr, " [-d] [-g] [-v] [-l <n>] [-r <n>] [-e <n>]\n" );
	for( i = strlen(ProgramName)+7; i > 0; i-- )
	    fputc( ' ', stderr );
	fprintf( stderr, " <presentation>  [<class>]\n" );
    	exit( 1 );
}

static int leftEngel  = 0,
	   rightEngel = 0,
		engel = 0;

static
void	printHeader() {

	char	*s, hostname[128];

	gethostname( hostname, 128 );

	printf( "#\n" );
	printf( "#    A Nilpotent Quotient Program (Version %s)\n", VERSION );
	printf( "#    Calculating a nilpotent quotient\n" );
	printf( "#    Input: %s", InputFile );
	if( leftEngel ) {
	    s = "th";
	    if( leftEngel == 1 ) s = "st";
	    if( leftEngel == 2 ) s = "nd";
	    if( leftEngel == 3 ) s = "rd";
	    printf( " & %d%s left Engel", leftEngel, s );
	}
	if( rightEngel ) {
	    s = "th";
	    if( rightEngel == 1 ) s = "st";
	    if( rightEngel == 2 ) s = "nd";
	    if( rightEngel == 3 ) s = "rd";
	    printf( " & %d%s right Engel", rightEngel, s );
	}
	if( engel ) {
	    s = "th";
	    if( engel == 1 ) s = "st";
	    if( engel == 2 ) s = "nd";
	    if( engel == 3 ) s == "rd";
	    printf( " & %d%s Engel", engel, s );
	}
	printf( "\n" );
	if( Cl != 666 ) printf( "#    Nilpotency class: %d\n", Cl );
	printf( "#    Program: %s", ProgramName );
	printf( "     Machine: %s\n#\n", hostname );
}

main( argc, argv )
int	argc;
char	*argv[];

{	FILE	*fp;
	long	time, start, begin;
	gen	g;
	extern	int	NrGens;
	extern	word	Epimorphism();

	CatchSignals();
	start = sbrk(0);
	begin = RunTime();

	ProgramName = argv[0]; argc--; argv++;

	setbuf( stdout, NULL );

	while( argc > 0 && argv[0][0] == '-' ) {
	    if( argv[0][2] != '\0' ) {
		fprintf( stderr, "unknown option: %s\n", argv[0] );
		usage( NULL );
	    }
	    switch( argv[0][1] ) {
		case 'r': if( --argc <= 1 ) usage("-r requires an argument");
			  argv++;
			  if( (rightEngel = atoi(argv[0])) <= 0 ) {
			      fprintf( stderr, "%s\n", argv[0] );
			      usage( "<n> must be positive." );
			  }
			break;
		case 'l': if( --argc <= 1 ) usage("-l requires an argument.");
			  argv++;
			  if( (leftEngel = atoi(argv[0])) <= 0 ) {
			      fprintf( stderr, "%s\n", argv[0] );
			      usage( "<n> must be positive." );
			  }
			break;
		case 'e': if( --argc <= 1 ) usage("-e requires an argument" );
			  argv++;
			  if( (engel = atoi(argv[0])) <= 0 ) {
			      fprintf( stderr, "%s\n", argv[0] );
			      usage( "<n> must be positive." );
			  }
			break;
		case 'g': Gap = !Gap;
			break;
		case 'v': Verbose = !Verbose;
			break;
		case 'd': Debug = !Debug;
			break;
		default : fprintf( stderr, "unknown option: %s\n", argv[0] );
			  usage( NULL );
			break;
	    }
	    argc--; argv++;
	}

	switch( argc ) {
	    case 1: InputFile = argv[0];
		     Cl = 666;
		 break;
	    case 2: InputFile = argv[0];
		     Cl = atoi( argv[1] );
		     if( Cl <= 0 ) usage( "<class> must be positive." );
		 break;
	    default: usage( NULL );
		 break;
	}
	
	if( (fp = fopen( InputFile, "r" )) == NULL ) {
	    perror( InputFile );
	    exit( 1 );
	}

	/* Read in the finite presentation. */
	Presentation( fp, InputFile );
	/* Set the number of generators. */
	WordInit( Epimorphism );
	InitEngel( leftEngel, rightEngel, engel );
	InitPrint( stdout );

	printHeader();

	time = RunTime();

	if( Gap ) printf( "NqLowerCentralSeries := [\n" );

	printf( "#    Calculating the abelian quotient ...\n" );
	InitEpim();

	NqEvalRelations();
	EvalEngel();

	ElimEpim();

	if( NrCenGens == 0 ) {
	    printf( "#    trivial abelian quotient\n" );
	    goto end;
	}

	if( Cl == 1 ) goto end;

	InitPcPres();

	printf( "#    The abelian quotient" );
	printf( " has %d generators\n", Dimension[Class] );
	printf( "#        with the following exponents:" );
	for( g = NrPcGens-Dimension[Class]+1; g <= NrPcGens; g++ )
	    printf( " %d", Exponent[g] );
	printf( "\n" );
	if(Verbose) {
	    printf("#    runtime       : %d msec\n",RunTime()-time);
	    printf("#    total runtime : %d msec\n",RunTime()-begin);
	    printf("#    total size    : %d byte\n",sbrk(0)-start);
	}
	printf( "#\n" );

	while( Class < Cl ) {
	    time = RunTime();
	    printf( "#    Calculating the class %d quotient ...\n", Class+1 );

	    AddGenerators();
	    Tails();
	    Consistency();
	    NqEvalRelations();
	    EvalEngel();
	    ElimGenerators();
	    if( NrCenGens == 0 ) goto end;
	    ExtPcPres();

	    printf( "#    Layer %d of the lower central series", Class );
	    printf( " has %d generators\n", Dimension[Class] );
	    printf( "#          with the following exponents:" );
	    for( g = NrPcGens-Dimension[Class]+1; g <= NrPcGens; g++ )
		printf( " %d", Exponent[g] );
	    printf( "\n" );
	    if(Verbose) {
		printf("#    runtime       : %d msec\n",RunTime()-time);
	        printf("#    total runtime : %d msec\n",RunTime()-begin);
	        printf("#    total size    : %d byte\n",sbrk(0)-start);
	    }
	    printf( "#\n" );

	}

end:
	printf( "\n\n#    The epimorphism :\n");
	PrintEpim();
	printf( "\n\n#    The nilpotent quotient :\n");
	PrintPcPres();
	printf( "\n" );
	printf( "#    total runtime : %d msec\n", RunTime() - time );
	printf( "#    total size    : %d byte\n", sbrk(0) - start );

	if( Gap ) printf( "];\n" );
	return 0;
}
