#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <ctype.h>
#include <pwd.h>
#include <time.h>
#include "defines.h"
#include "main.h"
#include "hash.h"
#include "mem_alloc.h"
#include "scale.h"

export Hash_table_p nodetable;
export Node *firstnode;
export int n_nodes;
export Line *firstline;
export float bbox[4];

/**** forward declarations ***/
static Node *get_node( char *name);
static int word_back(char *first, char **pc);
static char *copy_header( char *infile, char **box_file, int *linecnt,
		Boolean *got_prolog);
static FILE *open_prolog( char *name);
static char *pass_prolog( char *text, int *linecnt);
static void choose_bbox( char *box_file);
static void make_header( int cnt, char *infile);
static void copy_prolog( FILE *pro);
static void read_data( char *lasttext, int linecnt);
static void read_node( Line *line, char *keyword, int linecnt);
static void read_edge( Line *line, char *keyword, int linecnt, int hide);
static Node *find_supernode( char *name);
static void print_edge_coords( Edge *edge);


/**** exported functions ****/
export void read_files( char *infile)
{
	char *box_file = NULL;
	FILE *profile;
	int linecnt = 0; /* number of lines read */
	char *lasttext;
	Boolean seen_prolog = False;

	lasttext = copy_header( infile, &box_file, &linecnt, &seen_prolog);

	if ( prologname) /* check for postscript in requested include */
		profile = open_prolog( prologname);
	else	profile = NULL;

	if (boxarg || postscript)
		choose_bbox( box_file);

	if (postscript)
		make_header( linecnt, infile);

	if (seen_prolog) lasttext = pass_prolog( lasttext, &linecnt);
	if (prologname) copy_prolog( profile);

	if (seen_prolog && prologname)
		warning(" Requested include done behind resident prolog. "
			" O.K. ???\n");

	read_data( lasttext, linecnt);
}

export void write_data()
{
	Line *line;
	float x1, y1, x2, y2;

	if (postscript && writeunits)
	{	do_scale( 0, 0, &x1, &y1);
		do_scale( DIST, 1, &x2, &y2);
		x1 -= x2;
		if (x1 < 0) x1 = -x1;
		y1 -= y2;
		if (y1 < 0) y1 = -y1;
		printf( "%.4g %.4g nodedist\n", x1, y1);
	}
	if (postscript)
	{	printf( "/makecurves %s def\n", curvedges?"true":"false");
		printf( "/makearrows %s def\n", arrowedges?"true":"false");
	}

	for (line = firstline; line; line = line->next)
	{
		if (line->node)
		{	do_scale( line->node->x, line->node->y, &x1, &y1);
			printf( "%s %.4g %.4g %s\n",
				line->text, x1, y1, nodeword);
		}
		else if (line->edge)
		{	printf( "%s", line->text);
			print_edge_coords( line->edge);
			printf( "%s\n", edgeword);
		}
		else	printf( "%s", line->text);
	}
	if (postscript && showpage)
		printf( "showpage\n");

	fclose (stdout);
}

/******************** local functions ***************************/

/* print start and end coordinates of edge */
/* if 'routedge' was on, an edge is printed as a longer list of coordinates */
/* if an edge is 'reversed' the point print order is reversed */
static void print_edge_coords( Edge *edge)
{	
	Node *to, *from;
	float x, y;

	if (edge->reversed)
	{	from = edge->to;
		to = edge->from;
	} else
	{	from = edge->from;
		to = edge->to;
	}

	/* print the edge-list by recursion */
	if (!from->name)
		print_edge_coords( edge->reversed ?
				  from->outedges : from->inedges);

	do_scale( from->x, from->y, &x, &y);
	printf( "%.4g %.4g ", x, y);

	/* Only once (never inside the recursion!) as last: */
	if (to->name)
	{	do_scale( to->x, to->y, &x, &y);
		printf( "%.4g %.4g ", x, y);
	}
}

static char *copy_header( char *infile, char **box_file, int *linecnt,
		Boolean *seen_prolog)
{
	static char buf[ Bufsize];
	int n = 0;

	if (infile && strcmp( infile, "-"))
	{	if (debuglevel >= 2)
			verbose( "opening inputfile '%s'\n", infile);
		if (!freopen( infile, "r", stdin))
			error( "Cannot open inputfile '%s'!\n", infile);
	}

	if (fgets( buf, Bufsize, stdin))
	{	n++;
		if (!strncmp( buf, "%!PS", 4) ||
		    !strncmp( buf, "%%BeginProlog", 13))
			postscript = True;
		if (!strncmp( buf, "%%BeginProlog", 13))
			*seen_prolog = True;
	}
	
	if (debuglevel >= 4)
		verbose( "copy_header: postscript = %d\n", postscript);

	if (postscript && !*seen_prolog)
	while (buf[0] == '%')
	{
		if (!strncmp( buf, "%%BeginProlog", 13))
		{	/* This actually should not happen in DSC file */
			*seen_prolog = True;
			break;
		} else if (!strncmp( buf, "%%BoundingBox:", 14))
		{	*box_file = newstring( buf + 14);
			buf[0] = '\0';
		} else if (!strncmp( buf, "%%EndComments", 13))
		{	buf[0] = '\0';
			break;
		}

		fputs( buf, stdout);

		if (fgets( buf, Bufsize, stdin))
			n++;
		else	buf[0] = '\0';
       	}

	if (debuglevel >= 4)
		verbose( "Header copied from infile to stdout: %d lines\n",
			n-1);
	*linecnt = n;
	return( buf);
}

static FILE *open_prolog( char *name)
{
	FILE *pro;
	int c1, c2;
	char buf[ Bufsize];
	static char *dirs[] = { ".", "~/lib", "/usr/local/lib/ps",
				"/usr/local/lib", NULL };

	/* set default include file */
	if (!strcmp( name, "-"))
		name = "graphpro.ps";

	if (debuglevel >= 2)
		verbose( "open_prolog: search for '%s'...\n", name);

	if (name[0] == '/')
	{	strcpy( buf, name);
		pro = fopen( name, "r");
	} else
	{	int d;
		for (pro=NULL, d=0; !pro && dirs[d]; d++)
		{	if (dirs[d][0] == '~')
				sprintf( buf, "%s%s/%s", getenv( "HOME"),
					dirs[d]+1, name);
			else	sprintf( buf, "%s/%s", dirs[d], name);

			pro = fopen( buf, "r");
			if (debuglevel >= 3)
				verbose( "open_prolog: trying '%s'\n", buf);
		}
	}
	if (pro)
	{	if (debuglevel >= 1)
			verbose( "Copying prologue file '%s' to output.\n",
				buf);
	} else	error( "Prolog file '%s' not found!\n", name);

	/* check for postscript */
	c1 = getc( pro);
	if (c1 == '%')
	{	c2 = getc( pro);
		if (c2 == '!' || c2 == '%')
		{	postscript = True;
			if (debuglevel >= 4)
				verbose( "open_prolog: postscript = True\n");
		}
		ungetc( c2, pro);
	}
	ungetc( c1, pro);
	
	return( pro);
}

static void choose_bbox( char *box_file)
{
	/* some non-European media sizes in ps units (1/72 inch) */
	static char *mediatable[][2] =
	{{ "letter",   "612,792"},
	 { "legal",    "612,1008"},
	 { "tabloid",  "792,1224"},
	 { "ledger",   "792,1224"},
	 { "executive","522,756"},
	 { "monarch",  "279,540"},
	 { "c5",       "459,649"},
	 { "b4",       "728,1032"},
	 { "b5",       "499,708"},
	 { "dl",       "312,624"},
	 { NULL, NULL}
	};
	char *box_default = "a4";
	char *boxp[4];
	char *box_str, *p;
	int n;

	if (boxarg)
		box_str = newstring( boxarg);
	else if (box_file)
		box_str = box_file;
	else	box_str = newstring( box_default);

	if (debuglevel >= 2)
		verbose( "Box specification got as '%s'\n", box_str);

	/* count substrings in box */
	n = 0;
	for (p=box_str; *p && n<4; )
	{	/* skip white */
		while (*p && (*p == ' ' || *p == ',' || *p == '\t'))
		{	*p++ = ' ';
		}

		if (*p)
		{	boxp[ n++] = p;
			if (debuglevel >= 4)
				verbose( "Bounding box string [%d] = '%s'\n",
					n-1, boxp[n-1]);
		}

		while (*p && !(*p == ' ' || *p == ',' || *p == '\t'))
			p++;
	}
	*p = '\0'; /* force string termination */

	/***** determine x,y lower-left ********/
	if (n <= 2) /* assume lower-left is [0,0], only right-upper given */
	{	boxp[2] = boxp[0];
		boxp[3] = boxp[1];
		bbox[0] = bbox[1] = 0.0;
	}
	else
	{	int r = sscanf( boxp[0], "%f %f", &(bbox[0]), &(bbox[1]));
		if (r != 2) error( "Illegal box spec '%s'!\n", boxp[0]);
	}
	
	/***** determine upper-right *************/
	for (p=boxp[2]; *p; p++)
	{	if (isupper(*p)) *p=tolower(*p);
	}
	if (isdigit(*boxp[2]))
	{	/* numeric spec */
		int r = sscanf( boxp[2], "%f %f", &(bbox[2]), &(bbox[3]));
		if (r != 2) error( "Illegal box spec '%s'!\n", boxp[0]);
	} else	if (*boxp[2] == 'a' && isdigit(boxp[2][1]))
	{	/* symbolic European size spec */
		double A4length = ( 297    /* mm */
				  / 25.4) /* mm/inch */
				  * 72   /* points/inch */;
		int asize = atoi(boxp[2]+1);
		double sqrt2 = sqrt( (double)2.0 );
		double length, width;

		if (asize < 0 || asize > 12)
			error( "Illegal box spec '%s'!\n", boxp[2]);
		length = A4length * pow( sqrt2, (double)(4-asize));
		width = length / sqrt2;

		if (boxp[2][2] == 'l' || boxp[2][2] && boxp[2][3] == 'l')
		{	/* change to landscape sizing */
			bbox[2] = length;
			bbox[3] = width;
		} else /* portrait */
		{	bbox[2] = width;
			bbox[3] = length;
		}
	} else
	{	/* search for symbolic media size name */
		for (n=0; mediatable[n][0]; n++)
		{	int l = strlen( mediatable[n][0]);
			if (!strncmp( boxp[2], mediatable[n][0], l))
			{	sscanf( mediatable[n][1], "%f,%f",
				       &(bbox[2]), &(bbox[3]));
				if (boxp[2][l] == 'l')
				{	/* rotate for landscape mode */
					float f = bbox[2];
					bbox[2] = bbox[3];
					bbox[3] = f;
				}
				break;
			}
		}
		if (!mediatable[n][0])
		{	/* didn't find media spec */
			error( "Don't know what `%s' media is!\n",
			      boxp[2]);
		}
	}

	/* check validity of box */
	if (bbox[2] < bbox[0])
	{	float f = bbox[0];
		bbox[0] = bbox[2];
		bbox[2] = f;
	}
	if (bbox[3] < bbox[1])
	{	float f = bbox[1];
		bbox[1] = bbox[3];
		bbox[3] = f;
	}

	if (debuglevel >= 2)
		verbose( "Used bounding box: %g %g %g %g\n",
			bbox[0], bbox[1], bbox[2], bbox[3]);

}

static void make_header( int cnt, char *infile)
{
        char buf[ Bufsize];
        time_t t = time(0);
        struct passwd *pw;

        pw = getpwuid( getuid());
        gethostname( buf, Bufsize);

	if (cnt<=2) printf( "%%!PS-Adobe-3.0 EPSF-3.0\n");
	printf( "%%%%Creator: %s\n", myname);
	if (pw) printf( "%%%%For: %s@%s (%s)\n",
		       pw->pw_name, buf, pw->pw_gecos);
	printf( "%%%%Title: input file %s\n", infile);
	printf( "%%%%BoundingBox: %g %g %g %g\n",
		bbox[0], bbox[1], bbox[2], bbox[3]);
	printf( "%%%%CreationDate: %s\n\n", ctime( &t));
        /* Note that the ctime string itself probably contains a \n */
	/* and that the header comment lines MUST be contiguous */
}

/** copy in externally specified prolog file **/
static void copy_prolog( FILE *pro)
{
	int c;

	while ((c=getc(pro)) != EOF) putc( c, stdout);

	fclose( pro);

	putc( '\n', stdout);
}

/** pass through prolog from input file **/
static char *pass_prolog( char *text, int *linecnt)
{
	char buf[ Bufsize];
	Boolean in_prolog = True;

	if (text && text[0])
	{	fputs( text, stdout);
		in_prolog = strncmp( text, "%%EndProlog", 11);
	}

	while (in_prolog && fgets( buf, Bufsize, stdin))
	{	fputs( buf, stdout);
		in_prolog = strncmp( buf, "%%EndProlog", 11);
		(*linecnt)++;
	}

	if (in_prolog)
		warning( "Unexpected end of file while inside Prolog!\n");

	return( NULL); /* no unprocessed lines remain in buffer */
}

/*** now really parse file contents ***/
static void read_data( char *lasttext, int linecnt)
{
	Line **lastline = &(firstline);
	Line *line;
	char buf[Bufsize];
	int ln, le, hide;

	nodetable = hash_create( "nodetable", 379, 0);

	if (!lasttext)
	{	if (fgets( buf, Bufsize, stdin))
			lasttext = newstring(buf);
		else	lasttext = NULL;
		linecnt++;
	}

	ln = strlen( nodeword);
	le = strlen( edgeword);

	while (lasttext)
	{	if (!strncmp( lasttext, "%%BeginProlog", 13))
		{	/* Oooops, late detection of resident prolog */
			lasttext = pass_prolog( lasttext, &linecnt);
		} else
		{	char *p;

			*lastline = line = newstruct( Line);
			lastline = &(line->next);
			line->text = lasttext;
			if (debuglevel >= 6)
				verbose( "read_data: line=%s", line->text);
	
			/* scan back for last word */
			p = line->text + strlen( line->text) - 1;
			if (!(postscript && lasttext[0]=='%') &&
			    word_back( line->text, &p))
			{	
				if (debuglevel >= 5)
					verbose( "read_data: last word = %s",p);
				if (!strncmp(p,nodeword,ln) && !isgraph(p[ln]))
					read_node( line, p, linecnt);
				else
				{	hide = (*p=='-') ? 1 : 0;
					if (!strncmp( p+hide, edgeword, le) &&
					 !isgraph(p[le+hide]))
						read_edge( line, p, linecnt,
							   hide);
				}
			}
		}

		/* more to read ??  */
		if (fgets( buf, Bufsize, stdin))
			lasttext = newstring(buf);
		else	lasttext = NULL;
		linecnt++;
	}
}

static void read_node( Line *line, char *keyword, int linecnt)
{
	char *name = keyword-1;

	if (!word_back( line->text, &name))
	{	warning( "Line %d: nodename not found! (skipped)\n", linecnt);
		return;
	}

	line->node = get_node( name);

	/* cut text of line */
	*name = '\0';

	if (debuglevel >= 4)
		verbose( "read_node: nodename %s has adress %d\n",
			line->node->name, (int)(line->node));
}

static void read_edge( Line *line, char *keyword, int linecnt, int hide)
{
	char *toname = keyword-1;
	char *fromname = keyword-1;
	Edge *edge;
	Node *fromnode, *tonode;
	

	if (!word_back( line->text, &toname))
	{	warning( "Line %d: edge-to name not found! (skipped)\n",
			linecnt);
		return;
	}

	fromname = toname - 1;
	if (!word_back( line->text, &fromname))
	{	warning( "Line %d: edge-from name not found! (skipped)\n",
			linecnt);
		return;
	}

	fromnode = get_node( fromname);
	tonode = get_node( toname);

	/* cut text of line */
	*fromname = '\0';

	edge = newstruct( Edge);

	if (fromnode == tonode || hide)
	{	/* self-edges are not really added to the data structure */
		/* since the disturb current algorithmes, and do not */
		/* contribute to placement anyway */
		/* trick here to allow just later printing: */
		edge->to = tonode;
		edge->from = fromnode;
	} else	link_edge( edge, fromnode, tonode);

	line->edge = edge;
	if (debuglevel >= 4)
		verbose( "read_edge: edge from %s to %s has adress %d\n",
			fromnode->name, tonode->name, (int)(edge));
}
	
static Node *get_node( char *name)
{
	Node *node, *super;
	char buf[Bufsize];

	/* remove leading spaces and trailing spaces and more printable stuff */
	sscanf( name, "%s", buf);

	if (!hash_get( nodetable, buf, (void **)(&node))) /* not yet present */
	{	node = newstruct( Node);
		node->name = newstring( buf);
		hash_add( nodetable, node->name, node);
		link_node( &firstnode, node);
		n_nodes++;

		if (splitchar && (super = find_supernode( node->name)))
		{	Edge *e;
			node->super = super;
			e = newstruct( Edge);
			link_edge( e, super, node);
			if (debuglevel >=4)
				verbose( "get_node: Linked '%s' to '%s'\n",
					super->name, node->name);
		}
	}

	return( node);
}

static Node *find_supernode( char *name)
{	char *c;
	Node *super;

	c = strchr( name, splitchar);
	if (c && *(++c))
	{	super = get_node( c);
		super->flag = -1;
		return( super);
	} else return( NULL);
}

static Boolean word_back( char *first, char **pc)
{
	char *c = *pc;

	for (; c >= first; c--)
		if (!isspace(*c)) break;

	if (c < first)
	{	*pc = NULL;
		return( False);
	}

	for (; c >= first; c--)
		if (isspace(*c)) break;

	*pc = c+1;
	return( True);
}
