;% Copyright (c) 1990-1994 The MITRE Corporation
;% 
;% Authors: W. M. Farmer, J. D. Guttman, F. J. Thayer
;%   
;% The MITRE Corporation (MITRE) provides this software to you without
;% charge to use, copy, modify or enhance for any legitimate purpose
;% provided you reproduce MITRE's copyright notice in any copy or
;% derivative work of this software.
;% 
;% This software is the copyright work of MITRE.  No ownership or other
;% proprietary interest in this software is granted you other than what
;% is granted in this license.
;% 
;% Any modification or enhancement of this software must identify the
;% part of this software that was modified, by whom and when, and must
;% inherit this license including its warranty disclaimers.
;% 
;% MITRE IS PROVIDING THE PRODUCT "AS IS" AND MAKES NO WARRANTY, EXPRESS
;% OR IMPLIED, AS TO THE ACCURACY, CAPABILITY, EFFICIENCY OR FUNCTIONING
;% OF THIS SOFTWARE AND DOCUMENTATION.  IN NO EVENT WILL MITRE BE LIABLE
;% FOR ANY GENERAL, CONSEQUENTIAL, INDIRECT, INCIDENTAL, EXEMPLARY OR
;% SPECIAL DAMAGES, EVEN IF MITRE HAS BEEN ADVISED OF THE POSSIBILITY OF
;% SUCH DAMAGES.
;% 
;% You, at your expense, hereby indemnify and hold harmless MITRE, its
;% Board of Trustees, officers, agents and employees, from any and all
;% liability or damages to third parties, including attorneys' fees,
;% court costs, and other related costs and expenses, arising out of your
;% use of this software irrespective of the cause of said liability.
;% 
;% The export from the United States or the subsequent reexport of this
;% software is subject to compliance with United States export control
;% and munitions control restrictions.  You agree that in the event you
;% seek to export this software or any derivative work thereof, you
;% assume full responsibility for obtaining all necessary export licenses
;% and approvals and for assuring compliance with applicable reexport
;% restrictions.
;% 
;% 
;% COPYRIGHT NOTICE INSERTED: Mon Apr 11 11:42:27 EDT 1994


(herald CONSTRUCTORS)


(lset *constructors* '())

; A CONSTRUCTOR is an object that can be apply'ed to a list of components to
; yield a compound expression.  The logical constants are implemented by
; constructors.  

(define-predicate CONSTRUCTOR?)

; A constructor is a BINDING constructor if there are (ever) variables free in
; the components that are not free in the composite.  A binding expression is
; one with a binding constructor as its primary constructor. 

(define-predicate BINDING-CONSTRUCTOR?)

(define (NON-BINDING-CONSTRUCTOR? constructor)
  (and
   (constructor? constructor)
   (not
    (binding-constructor? constructor))))

(define (BINDING-EXPRESSION? expression)
  (and (expression? expression)
       (binding-constructor?
        (expression-constructor expression))))

; The GCT-ENTRIES record data on constructors.  These data include:
; 
;  1,2 Sexp-builder and -decoder, used in printing and reading expressions
;  3.  local-context-incrementer, a procedure which given a list of components
;      and an index computes the incremental local context for the component at
;      that index.
;  4.  A simplifier, which, given a list of components, returns an expression
;      that is equivalent to the result of applying the constructor to those 
;      components, but simpler if possible.
;  5.  Parity, which, given an index, returns -1, 1, or 0 depending as the
;      component at that index occurs negatively, positively, or neither.  An
;      occurrence is negative (positive) if the truth value of the whole is a
;      non-increasing (non-decreasing) function of the part at that index.
;  6.  A direct inference method, which is an applicable object that, given a
;      sequent, returns a list of sequents sufficient to entail it.
;  7.  A truth-value seeker, which, when given a context, expr, and numerical
;      persistence, attempts to reduce the expr to a truth value.  If
;      unsuccessful, it returns '#f.

(define-structure-type GCT-ENTRY
  sexp-decoder
  sexp-builder
  local-context-incrementer
  simplifier
  parity
  direct-inference
  antecedent-inference
  syllogistic-inference
  truth-value-seeker
  logical-transform)

(block
  (set (gct-entry-sexp-decoder (stype-master gct-entry-stype)) '#f)
  (set (gct-entry-sexp-builder (stype-master gct-entry-stype)) '#f)
  (set (gct-entry-local-context-incrementer (stype-master gct-entry-stype)) (always nil))
  (set (gct-entry-simplifier (stype-master gct-entry-stype)) '#f)
  (set (gct-entry-parity (stype-master gct-entry-stype)) (always 0))
  (set (gct-entry-direct-inference (stype-master gct-entry-stype)) '#f)
  (set (gct-entry-antecedent-inference (stype-master gct-entry-stype)) (always '#f))
  (set (gct-entry-syllogistic-inference (stype-master gct-entry-stype)) '#f)
  (set (gct-entry-logical-transform (stype-master gct-entry-stype)) '#f)
  (set (gct-entry-truth-value-seeker (stype-master gct-entry-stype)) (always '#f)))

(define-operation (gct-entry-value constructor)
  (stype-master gct-entry-stype))

(define SYMBOL->CONSTRUCTOR
  (let ((symbol-constructor-alist nil))
    (operation
	(lambda (sym)
	  (cond ((assq sym symbol-constructor-alist)
		 => cdr)
		(else nil)))
      ((setter self)
       (lambda (sym constr)
	 (cond ((assq sym symbol-constructor-alist)
		=>
		(lambda (p)
		  (set (cdr p) constr)
		  constr))
	       (else
		(push symbol-constructor-alist (cons sym constr))
		constr)))))))

; SEXP-BUILDER and SEXP-DECODER are used in reading and printing.
; SEXP-BUILDER, if non-nil, is a procedure which builds an sexp-form for a
; constructed expression in a language.  If nil, the default is to list the
; name of the constructor together with the sexp forms of the components in
; the language.  SEXP-DECODER, if non-nil, is a procedure which, called with
; an sexp, language and name-formal-symbol-alist, gets the components
; to which the constructor should be applied from the sexp.  It is responsible 
; for adding any new info to name-formal-symbol-alist in recursive calls.  If
; nil, the default is to map
; (lambda (s)
;   (sexp->expression-1 s language name-formal-symbol-alist))
; through the cdr of the sexp.   

(define (SEXP-BUILDER constructor)
  (gct-entry-sexp-builder (gct-entry-value constructor)))

(define (SET-SEXP-BUILDER constructor new-value)
  (set (gct-entry-sexp-builder (gct-entry-value constructor)) new-value))

(define (SEXP-DECODER constructor)
  (gct-entry-sexp-decoder (gct-entry-value constructor)))

(define (SET-SEXP-DECODER constructor new-value)
  (set (gct-entry-sexp-decoder (gct-entry-value constructor)) new-value))

; Procedures to retrieve and set the various other gct entry fields.

(define (CONSTRUCTOR-LC-INCREMENTER constructor)
  (gct-entry-local-context-incrementer (gct-entry-value constructor)))

(define (CONSTRUCTOR-SIMPLIFIER constructor)
  (gct-entry-simplifier (gct-entry-value constructor)))

(define (CONSTRUCTOR-PARITY constructor)
  (gct-entry-parity (gct-entry-value constructor)))

(define (CONSTRUCTOR-DIRECT-INFERENCE constructor)
  (gct-entry-direct-inference (gct-entry-value constructor)))

(define (CONSTRUCTOR-ANTECEDENT-INFERENCE constructor)
  (gct-entry-antecedent-inference (gct-entry-value constructor)))

(define (CONSTRUCTOR-SYLLOGISTIC-INFERENCE constructor)
  (gct-entry-syllogistic-inference (gct-entry-value constructor)))

(define (CONSTRUCTOR-TRUTH-VALUE-SEEKER constructor)
  (gct-entry-truth-value-seeker (gct-entry-value constructor)))

(define (CONSTRUCTOR-LOGICAL-TRANSFORM constructor)
  (gct-entry-logical-transform (gct-entry-value constructor)))


(define (SET-CONSTRUCTOR-LC-INCREMENTER constructor new-value)
  (set (gct-entry-local-context-incrementer (gct-entry-value constructor))
       new-value))

(define (SET-CONSTRUCTOR-SIMPLIFIER constructor new-value)
  (set (gct-entry-simplifier (gct-entry-value constructor))
       new-value))

(define (SET-CONSTRUCTOR-PARITY constructor new-value)
  (set (gct-entry-parity (gct-entry-value constructor))
       new-value))

(define (SET-CONSTRUCTOR-DIRECT-INFERENCE constructor new-value)
  (set (gct-entry-direct-inference (gct-entry-value constructor))
       new-value))

(define (SET-CONSTRUCTOR-ANTECEDENT-INFERENCE constructor new-value)
  (set (gct-entry-antecedent-inference (gct-entry-value constructor))
       new-value))

(define (SET-CONSTRUCTOR-TRUTH-VALUE-SEEKER constructor new-value)
  (set (gct-entry-truth-value-seeker (gct-entry-value constructor))
       new-value))

(define (SET-CONSTRUCTOR-SYLLOGISTIC-INFERENCE constructor new-value)
  (set (gct-entry-syllogistic-inference (gct-entry-value constructor))
       new-value))

(define (SET-CONSTRUCTOR-LOGICAL-TRANSFORM constructor new-value)
  (set (gct-entry-logical-transform (gct-entry-value constructor))
       new-value))

(define-operation constructor-hash)

; A CONSTRUCTOR is an object that, if apply'ed to components, constructs a
; compound expression (of an appropriate kind) having those as its
; components.  To make a constructor, we provide four parameters.  BINDER? is
; false if the constructor is not to bind variables.  If non-false, it is to
; be a procedure that, when apply'ed to components, returns the list of
; variables to be bound in the result.  COMPONENT-CHECKER is the procedure
; that the constructor will use to check the appropriateness of components to
; which it may be apply'ed, and to construct a sorting for the resulting
; compound from the sortings of the components.  It is expected to raise an
; error if the components are not appropriate.  SYMBOL-FORM is a symbol
; identifying the constructor; it is used in reading and printing, and as a
; key into the global constructor table.  

(define (MAKE-CONSTRUCTOR binder?
                          component-checker
                          symbol-form)
  (labels
      ((gct-entry (make-gct-entry))
       (alpha-hash (descriptor-hash (cons symbol-form symbol-form)))
       (session-independent-hash ((*value t-implementation-env 'string-hash)
				  (symbol->string symbol-form)))
       
       (constructor
        (object
            (lambda components
              (constructor-procedure components))

          ((constructor? self) '#t)
          ((binding-constructor? self) (true? binder?))
	  ((gct-entry-value self) gct-entry)
	  ((constructor-hash self) session-independent-hash)
          ((name self) symbol-form)
          ((print self port)
           (display symbol-form port))
	  ((two-d-table-hash self) alpha-hash)))

       (constructor-procedure
        (lambda (components)
	  (or
	   (retrieve-from-compound-table constructor components)
	   (let ((expr (constructor-construct constructor components)))
	     (insert-in-compound-table constructor components expr)
	     expr))))
	
       ((constructor-construct constructor components)
	(let ((new-sorting (apply component-checker components))
	      (new-bindings
	       (and binder? (binder? components)))
	      (new-height
	       (if (null? components)
		   0
		   (1+ (apply max (map expression-height components)))))
	      (new-home (compute-home-language components))
	      (expr (make-expression)))
	  (set (expression-components expr) components)
	  (set (expression-constructor expr) constructor)
	  (set (expression-constants expr)
	       (collect-set expression-constants components))
	  (set (expression-bound-variables expr)
	       (set-union (collect-set bound-variables components) new-bindings))
	  (set (expression-free-variables expr)
	       (set-diff (collect-set free-variables components) new-bindings))
	  (set (expression-newly-bound-variables expr) new-bindings)
	  (if (not (null? components))
	      (set (expression-lead-constant expr)
		   (expression-lead-constant (car components)))
	      (set (expression-lead-constant expr) 'no-lead-constant))
	  (set (expression-sorting expr) new-sorting)
	  (set (expression-home expr) new-home)
	  (set (expression-height expr) new-height)
	  (set (expression-descriptor-hash expr)	;ensure this hash is permanent
	       (descriptor-hash expr))
	  (set (expression-alpha-hash expr)
	       (hash-combine-two-fixnums
		alpha-hash
		(alpha-hash-combine-exprs components)))
	  (set (expression-alpha-root expr)
	       (if binder?
		   (compute-expression-alpha-root
		    expr
		    (hash-combine-two-fixnums
		     alpha-hash
		     (alpha-hash-combine-exprs components)))
		   (let ((roots (map expression-alpha-root components)))
		     (if (every? eq? components roots)
			 expr
			 (constructor-procedure roots)))))
	  (set (expression-var-name-conflict? expr)
	       (constructor-conflicting-vars components binder?))
	  expr)))
    (set
     (symbol->constructor symbol-form)
     constructor)
    (push *constructors* constructor)
    constructor))

(define-integrable (compute-expression-alpha-root uninterned-expression hash)
  (let ((entries (table-entry *alpha-root-table* hash)))
    (let ((root (find-alpha-root-in-list uninterned-expression entries)))
      (if (eq? uninterned-expression root)
	  (push (table-entry *alpha-root-table* hash)
		uninterned-expression))
      root)))

(lset *alpha-root-table* (make-table '*alpha-root-table*))

(lset *constructor-conflicting-vars-table*
      (make-table '*constructor-conflicting-vars-table*))

(define (constructor-conflicting-vars components binder?)
  (if binder?
      (let* ((nbvs (binder? components))
	     (bodies (set-difference components nbvs)))
	(set-difference 		 
	 (constructor-conflicting-vars bodies '#f)
	 nbvs))
      (let ((table *constructor-conflicting-vars-table*))
	(iterate iter ((components components)
		       (conflicts '()))
	  (if (null? components)
	      (make-set conflicts)
	      (let ((first (car components)))
		(iterate sub-iter ((vars (free-variables first))
				   (conflicts conflicts))
		  (if (null? vars)
		      (iter (cdr components)
			    (append (expression-var-name-conflict? first)
				    conflicts))
		      (let* ((n (name (car vars)))
			     (e (table-entry table n)))
			(if (and e
				 (not (eq? e (car vars))))
			    (sub-iter (cdr vars)
				      (cons* (car vars) e conflicts))
			    (bind (((table-entry table n) (car vars)))
			      (sub-iter (cdr vars)
					conflicts))))))))))))

; A TRANSPARENT constructor is a device to control reading and printing.  It
; introduces no logical difference from its single component, but may force the
; component to be read in a context in which the values of variables are known
; (as with the constructor WITH), and force their sortings to be printed, or
; similar shenanigans.  WITH is currently the only transparent constructor. 

(define-predicate transparent-constructor?)

(define (MAKE-TRANSPARENT-CONSTRUCTOR symbol-form sexp-decoder sexp-builder)
  (labels
      ((gct-entry (make-gct-entry))

       (constructor
        (object
            (lambda (component)
              (constructor-procedure component))

          ((constructor? self) '#t)
	  ((transparent-constructor? self) '#t)
          ((binding-constructor? self) '#f)
          ((gct-entry-value self) gct-entry)
	  ((name self) symbol-form)
          ((print self port)
           (display symbol-form port))))

       (constructor-procedure
        (lambda (component)
          (let ((expr (make-expression)))
            (set (expression-components expr) (list component))
            (set (expression-constructor expr) constructor)
            (set (expression-constants expr) (expression-constants component))
            (set (expression-bound-variables expr) (bound-variables component))
            (set (expression-free-variables expr) (free-variables component))
            (set (expression-newly-bound-variables expr) nil)
            (set (expression-lead-constant expr) (expression-lead-constant component))
	    (set (expression-sorting expr) (expression-sorting component))
            (set (expression-home expr) (home-language component))
            (set (expression-height expr) (1+ (expression-height component)))
            (set (expression-alpha-hash expr) (expression-alpha-hash component))
	    expr))))      
    (set
     (symbol->constructor symbol-form)
     constructor) 
    (set-sexp-decoder constructor sexp-decoder)
    (set-sexp-builder constructor sexp-builder)
    (push *constructors* constructor)
    constructor))

; A QUANTIFICATION will be any formula with a binding constructor. 

(define (QUANTIFICATION? formula)
  (and (formula? formula)
       (binding-expression? formula)))

; A BOOLEAN-COMBINATION is a formula that is neither atomic nor a
; quantification.  Perhaps the name is contentious.  

(define (BOOLEAN-COMBINATION? formula)
  (and (formula? formula)
       (not (quantification? formula))
       (not (atomic-formula? formula))))

; (CONSTRUCTOR-SIMPLIFY expression) finds the simplification procedure
; associated with the principal constructor for an expression and applies it to
; the components.

(define (CONSTRUCTOR-SIMPLIFY expression)
  (let ((constructor (expression-constructor expression)))
    (if constructor
	(let ((simplifier (constructor-simplifier constructor)))
	  (if simplifier
	      (simplifier (expression-components expression))
	      expression))
	expression)))

; (RECURSIVE-CONSTRUCTOR-SIMPLIFY expression) finds the simplification
; procedure associated with the principal contsructor for an expression and
; applies it to the result of recursively simplifying the components.

(define (RECURSIVE-CONSTRUCTOR-SIMPLIFY expression)
  (let ((constructor (expression-constructor expression)))
    (if constructor
	(let ((simplifier (constructor-simplifier constructor)))
	  (if simplifier
	      (simplifier
	       (map recursive-constructor-simplify
		    (expression-components expression)))
	      (apply constructor
		     (map recursive-constructor-simplify
			  (expression-components expression)))))
	expression)))
