;% Copyright (c) 1990-1994 The MITRE Corporation
;% 
;% Authors: W. M. Farmer, J. D. Guttman, F. J. Thayer
;%   
;% The MITRE Corporation (MITRE) provides this software to you without
;% charge to use, copy, modify or enhance for any legitimate purpose
;% provided you reproduce MITRE's copyright notice in any copy or
;% derivative work of this software.
;% 
;% This software is the copyright work of MITRE.  No ownership or other
;% proprietary interest in this software is granted you other than what
;% is granted in this license.
;% 
;% Any modification or enhancement of this software must identify the
;% part of this software that was modified, by whom and when, and must
;% inherit this license including its warranty disclaimers.
;% 
;% MITRE IS PROVIDING THE PRODUCT "AS IS" AND MAKES NO WARRANTY, EXPRESS
;% OR IMPLIED, AS TO THE ACCURACY, CAPABILITY, EFFICIENCY OR FUNCTIONING
;% OF THIS SOFTWARE AND DOCUMENTATION.  IN NO EVENT WILL MITRE BE LIABLE
;% FOR ANY GENERAL, CONSEQUENTIAL, INDIRECT, INCIDENTAL, EXEMPLARY OR
;% SPECIAL DAMAGES, EVEN IF MITRE HAS BEEN ADVISED OF THE POSSIBILITY OF
;% SUCH DAMAGES.
;% 
;% You, at your expense, hereby indemnify and hold harmless MITRE, its
;% Board of Trustees, officers, agents and employees, from any and all
;% liability or damages to third parties, including attorneys' fees,
;% court costs, and other related costs and expenses, arising out of your
;% use of this software irrespective of the cause of said liability.
;% 
;% The export from the United States or the subsequent reexport of this
;% software is subject to compliance with United States export control
;% and munitions control restrictions.  You agree that in the event you
;% seek to export this software or any derivative work thereof, you
;% assume full responsibility for obtaining all necessary export licenses
;% and approvals and for assuring compliance with applicable reexport
;% restrictions.
;% 
;% 
;% COPYRIGHT NOTICE INSERTED: Mon Apr 11 11:42:27 EDT 1994


(herald FP-READ)


;;; Functional Programming notation for IMPS.
;;; Syntactic support for disjoint unions is
;;; missing because I don't know what to do.
;;; This notation does not require the use of
;;; unary functions.  To make this change,
;;; Replace the function application production
;;; with
;;;
;;; Exp  ::= Exp Aexp                       ; unary function application
;;;
;;; Term* = zero or more comma separated Term's and
;;; Term+ = one or more comma separated Term's.
;;;
;;; An ATOM is a sequence of letters, digits, underscore, and period.
;;;
;;;------------------------
;;; The start symbol is Exp.
;;;
;;; Exp  ::= Aexp                           ; delimited expressions
;;;       |  Exp(Exp*)                      ; function application
;;;       |  Exp Op Exp                     ; binary ops
;;;       |  Exp:Sort                       ; sort assertion
;;;
;;; Aexp ::= ATOM                           ; vars or numbers or constants.
;;;       |  (Exp)
;;;       |  ()                             ; unit
;;;       |  (Exp, Exp+)                    ; tuples
;;;       |  Binder {Binding* | Exp}        ; binders
;;;       |  Cond {Exp, Exp, Exp+}          ; conditionals
;;;                                         ; requires an odd number of exprs.
;;;       |  [Exp*] Sort                    ; finite sequences of sort
;;;       |  + Exp                          ; Unary +
;;;       |  - Exp                          ; Unary -
;;;       |  ? Sort                         ; undefined from sort
;;;       |  @ Op                           ; Op as an expression
;;;
;;; Sort ::= ATOM                           ; base sorts
;;;       |  Sort -> Sort                   ; unary function constructor
;;;       |  {Sort, Sort+}                  ; nary function constructor
;;;       |  (Sort)
;;;       |  ()                             ; the type of unit
;;;       |  (Sort, Sort+)                  ; product sort
;;;       |  [Sort]                         ; sequence sort
;;;
;;; Binders ::= lambda | for_all | for_some | iota | iota_p | with
;;;
;;; Binding  ::= ATOM+ : Sort
;;;
;;; All but the last element of a binder must be a type assertion.
;;; For binders, the left operand of a type assignment must be a variable.
;;; as an extention, you can declare multiple bindings by giving a
;;; tuple of variables as the left operand.
;;;
;;; precedence
;;; 
;;; function application
;;; operators:
;;;          right        left      nonassoc
;;; high precedence
;;;          ->
;;;                                 @
;;;           ^
;;;                        *         /
;;;                        + -
;;;          ++
;;;                                  = < <= > >= ==
;;;                                  : :* ?
;;;                                  not 
;;;          and
;;;          or
;;;          implies iff
;;; low precedence
;;;---------

;  (fp-read language port)  reads an expression, ignoring the language (for now) 
;
;  semicolon    terminates input
;  comment character is # (comment goes to end of line)
;
;  f(x, y)  reads as  (f x y)
;
;  a:b      reads as  (is-defined-in-sort a b) where b is parsed a sort.
;  a::      reads as  (is-defined a)
;
;  x and y, x or y, not x  do the obvious thing
;
;  x + y    reads as  (+ x y)     - similarly for - * / = < > <= >=
;
;  x++y     reads as  (append x y)
;
;  [a, b, ...]e   reads as  (cons (cons .... nil(e))) where e is parsed
;                 as a sort.
;
;  ()            reads as  an%individual
;  (x, y, ...)   reads as  (pair x (pair y  ... ))
;
;  if{x,y,z}     reads as (if-term x y z)
;  if{x,y,z,w,v} reads as (if-term x y (if-term z w v)) ... etc.
;
;  for_all{x,y:s,z,w:t|e} reads as (forall (((x y) s) ((z w) t)) e)
;                where s and t are parsed as sorts.

; Sorts.
;
; {a,b,c,...z}  reads as (a b c ... z)
; a -> b        reads as (a b)
; (a,b,c)       reads as (a (b c unit%sort) unit%sort)
; ()            reads as unit%sort
; [a]           reads as (nn a)

(define sort-parsing-precedence 165)

;; False when parsing an expression, true when parsing a sort.
(lset *parsing-a-sort?* '#f)

(define (parse-sort stream)
  (bind ((*parsing-a-sort?* '#t))
	(parse sort-parsing-precedence stream)))

;;; The Lexer

(define fp-lexer-table (make-lexer-table))

(set-char-tokenization! (lexer-rtab fp-lexer-table)
			#\#
			(lambda (c port)
			  (ignore c)
			  (gobble-line port)
			  (read port))
			'#t)

(define (gobble-line port)
  (let ((c (read port)))
    (if (or (eof-object? c) (char=? #\newline c))
        '()
        (gobble-line port))))

;

(define (define-fp-keyword name op)
  (define-keyword fp-lexer-table name op))

(define (define-fp-punctuation string op)
  (define-punctuation fp-lexer-table string op))

;; Identifiers are a sequence of letters, digits,
;; underscore, and period that do not parse as numbers.
(let ((reserved-operator
       (make-op 'reserved '#f '#f '#f '#f)))
  (walk
   (lambda (char)
     (define-fp-punctuation (char->string char) reserved-operator))
   '(#\! #\@ #\# #\$ #\^ #\& #\* #\- #\+ #\= #\~
;	 #\%
	 #\| #\: #\< #\> #\? #\, #\/)))

; Arguments to make-op are: name prec asso null-method left-method

; f(x, y) reads as (f x y)
; f((x, y)) reads as (f (pair x y))

(define (open-paren-null-method operator stream)
  (ignore operator)
  (let ((right (prsmatch close-paren-operator stream)))
    (cond ((null? right)
	   (if *parsing-a-sort?* 'unit%sort 'an%individual)) ; ()
	  ((null? (cdr right)) (car right)) ; (x)
	  (else				; (x, y, ...)
	   (if *parsing-a-sort?*
	       (construct-pair-sort right)
	       (construct-pair right))))))

(define (construct-pair-sort right)	; the sort of (a,b)
  (list (car right)			; (a b unit%sort)
	(let ((right (cdr right)))
	  (if (null? (cdr right))
	      (car right)
	      (construct-pair-sort right)))
	'unit%sort))

(define (construct-pair right)		; (a,b) -> (pair a b)
  (list 'pair
	(car right)
	(let ((right (cdr right)))
	  (if (null? (cdr right))
	      (car right)
	      (construct-pair right)))))

(define (open-paren-left-method operator left stream)
  (ignore operator)
  (cons left (prsmatch close-paren-operator stream)))

(define-fp-punctuation "("
  (make-op 'open-paren '#f '#f open-paren-null-method open-paren-left-method))

(define-fp-punctuation "," comma-operator)

(define close-paren-operator
  (make-op 'close-paren 5 '#f delim-error extra-paren-err))
(define-fp-punctuation ")" close-paren-operator)

; variable binders
; for_all{exp, exp, exp... | exp }
;

(define (extra-bar-err token left stream)
  (ignore token left)
  (parse-error stream "too many vertical bars"))
(define vertical-bar-operator
  (make-op 'vertical-bar 5 '#f delim-error extra-bar-err))
(define-fp-punctuation "|" vertical-bar-operator)

(define (binding-null-method token stream)
  (let ((right (read stream)))
    (if (not (eq? right open-brace-operator))
        (parse-error stream "expecting { but got ~a" right)
	(let* ((varlist (bindize (prsmatch vertical-bar-operator stream)
				 stream))
	       (body (parse (operator-precedence close-brace-operator)
			    stream))
	       (right (read stream)))
	  (if (eq? right close-brace-operator)
	      (list (operator-name token) varlist body)
	      (parse-error stream "expecting } but got ~a" right))))))
	  
(define (bindize l stream)
  (iterate loop ((vars '()) (bs '()) (l l))
    (cond ((not (pair? l))
	   (if (null? vars)
	       (reverse bs)
	       (parse-error stream "bad binding")))
	  ((a-binder (car l) vars)
	   => (lambda (b)
		(loop '() (cons b bs) (cdr l))))
	  ((symbol? (car l))
	   (loop (cons (car l) vars) bs (cdr l)))
	  (else (parse-error stream "bad binding")))))

(define (a-binder thing vars)
  (and (pair? thing)
       (eq? 'is-defined-in-sort (car thing))
       (cons (caddr thing) (reverse (cons (cadr thing) vars)))))

(define (define-fp-binder name translation)
  (define-fp-keyword name
    (make-op translation '#f '#f binding-null-method '#f)))

(define-fp-binder 'for_all 'forall)
(define-fp-binder 'for_some 'forsome)
(define-fp-binder 'lambda 'lambda)
(define-fp-binder 'with 'with)
(define-fp-binder 'iota 'iota)
(define-fp-binder 'iota_p 'iota-p)
(define-fp-binder 'is_defined_in 'is-defined-in)

(define (cond-null-method token stream)
  (let ((right (read stream)))
    (if (not (eq? right open-brace-operator))
        (parse-error stream "expecting { but got ~a" right)
        (iterate loop ((right (condize (prsmatch close-brace-operator stream)
				       stream)))
	  (if (null? (cdr right))
	      (car right)
	      (list (operator-name token)
		    (car right)
		    (cadr right)
		    (loop (cddr right))))))))

(define (condize l stream)
  (let ((n (length l)))
    (if (and (odd? n) (> n 2))
	l
	(parse-error stream "Bad cond list ~a" l))))

(define (define-fp-cond name translation)
  (define-fp-keyword name
    (make-op translation '#f '#f cond-null-method '#f)))

(define-fp-cond 'if 'if)
;; formerly (define-fp-cond 'if 'if-term)
(define-fp-cond 'if_pred 'if)
;; fromerly
;;(define-fp-cond 'if_pred 'if-pred)
;; 

(define-fp-cond 'if_form 'if-form)

(define (open-brace-null-method operator stream)
  (ignore operator)
  (if *parsing-a-sort?*
      (prsmatch close-brace-operator stream)
      (parse-error stream "{...} is not an expression")))

(define open-brace-operator
  (make-op 'open-brace '#f '#f open-brace-null-method '#f))
(define-fp-punctuation "{" open-brace-operator)
(define close-brace-operator
  (make-op 'close-brace 5 '#f delim-error extra-brace-err))
(define-fp-punctuation "}" close-brace-operator)

; [Exp...] Sort reads as (cons Exp (cons .... nil(Sort)))
(define (open-bracket-null-method operator stream)
  (ignore operator)
  (let ((right (prsmatch close-bracket-operator stream)))
    (cond ((not *parsing-a-sort?*)
	   (construct-sequence
	    right
	    (parse-sort stream)))
	  ((null? right)
	   (parse-error stream "[] is an unrecognized sort"))
	  ((not (null? (cdr right)))
	   (parse-error stream "[x,y...] is an unrecognized sort"))
	  (else (cons 'nn right)))))

(define (construct-sequence right sort)
  (iterate loop ((right right))
    (if (null? right)
	(list 'nil sort)
	(list 'cons
	      (car right)
	      (loop (cdr right))))))

(define-fp-punctuation "["
  (make-op 'open-bracket '#f '#f open-bracket-null-method '#f))
(define close-bracket-operator
  (make-op 'close-bracket 5 '#f delim-error extra-bracket-err))
(define-fp-punctuation "]" close-bracket-operator)

(define-fp-punctuation "++"
  (make-op 'append 90 right-assoc '#f parse-nary))

; Operator as an expression
(define-fp-punctuation "@"
  (make-op '@ 160 nonassoc
	   (lambda (operator stream)
	     (ignore operator)
	     (let ((token (read stream)))
	       (if (operator? token)
		   (operator-name token)
		   token)))
	   '#f))

; Boolean operators

(define-fp-keyword 'true '(the-true))
(define-fp-keyword 'false '(the-false))

(define-fp-keyword 'not (make-op 'not 70 '#f parse-prefix '#f))
(define-fp-keyword 'and (make-op 'and 65 '#f '#f parse-nary))
(define-fp-keyword 'or  (make-op 'or  60 '#f '#f parse-nary))
(define-fp-keyword 'implies
  (make-op 'implies  55 right-assoc '#f parse-right-infix))
(define-fp-keyword 'iff
  (make-op 'iff  55 right-assoc '#f parse-right-infix))

; Arithmetic

(define-fp-punctuation "^"
  (make-op '^ 140 right-assoc '#f parse-right-infix))

(define-fp-punctuation "*"
  (make-op '* 120 left-assoc '#f parse-left-infix))

(define-fp-punctuation "/"
  (make-op '/ 120 '#f '#f parse-nonassoc-infix))

(define (parse-prefix-plus operator stream)
  (let ((right (parse (prec operator) stream)))
    (if (number? right)
	right
	(parse-error stream "~a is not unary." operator))))

(define-fp-punctuation "+"
  (make-op '+ 100 left-assoc parse-prefix-plus parse-left-infix))

(define (parse-prefix-minus operator stream)
  (let ((right (parse (prec operator) stream)))
    (if (number? right)
	(- right)
	(list '- right))))	

(define-fp-punctuation "-"
  (make-op 'sub 100 left-assoc parse-prefix-minus parse-left-infix))

(define-fp-punctuation "="
  (make-op '= 80 '#f '#f parse-nonassoc-infix))

(define-fp-punctuation ">"
  (make-op '> 80 '#f '#f parse-nonassoc-infix))

(define-fp-punctuation "<"
  (make-op '< 80 '#f '#f parse-nonassoc-infix))

(define-fp-punctuation ">="
  (make-op '>= 80 '#f '#f parse-nonassoc-infix))

(define-fp-punctuation "<="
  (make-op '<= 80 '#f '#f parse-nonassoc-infix))

(define-fp-punctuation "=="
  (make-op '== 80 '#f '#f parse-nonassoc-infix))

; Sorts

(define (parse-arrow-infix operator left stream)
  (if *parsing-a-sort?*
      (cdr (parse-right-infix operator left stream))
      (parse-error stream "-> not in a sort expression")))

(define-fp-punctuation "->"
  (make-op '-> (+ 5 sort-parsing-precedence)
	   right-assoc '#f parse-arrow-infix))

(define (parse-nonassoc-infix-right-sort operator left stream)
  (let ((left (list (operator-name operator)
		    left
		    (parse-sort stream)))
	(next-op (peek stream)))
    (if (> (prec operator) (prec next-op))
	left
	(parse-error
	 stream
	 "Ambiguous parse, ~a not expected"
	 next-op))))

(define colon-operator
  (make-op 'is-defined-in-sort 75 '#f '#f
	   parse-nonassoc-infix-right-sort))
(define-fp-punctuation ":" colon-operator)

(define-fp-punctuation ":*"
  (make-op 'is-defined 75 '#f '#f parse-postfix))

(define (parse-prefix-sort operator stream)
  (list (operator-name operator) (parse-sort stream)))

;(define-fp-punctuation "?"
;  (make-op 'undefined 75 '#f parse-prefix-sort '#f))

;;Quasi-constructor names 

(define-fp-keyword 'is_total 'total?)
(define-fp-keyword 'is_vacuous 'vacuous?)
(define-fp-keyword 'is_reflexive 'reflexive?)
(define-fp-keyword 'is_transitive 'transitive?)
(define-fp-keyword 'is_antisymmetric 'antisymmetric?)
(define-fp-keyword 'is_comparable 'comparable?)
(define-fp-keyword 'is_partial_order 'partial-order?)
(define-fp-keyword 'is_linear_order 'linear-order?)
(define-fp-keyword 'is_upper_bound 'upper-bound?)
(define-fp-keyword 'is_chain 'chain?)
(define-fp-keyword 'is_cpo 'cpo?)
(define-fp-keyword 'is_monotone 'monotone?)
(define-fp-keyword 'is_continuous 'continuous?)
(define-fp-keyword 'sub_function 'sub-function)


; End of input...

(define-fp-punctuation ";" end-of-input-operator)

; Read using operator precedence parser with fp tokenizer table

(define (fp-read language port)
  (ignore language)
  (set *parsing-a-sort?* '#f)
  (toplevel-parse (port->stream port fp-lexer-table)))

(define (qr-fp string)
  (sexp->expression
   (current-language)
   (fp-read (string->input-port string))))

; Read/print loop for testing.

(define (fp-rpl)
  (display "fp> " (standard-output))
  (force-output (standard-output))
  (let ((thing (fp-read (standard-input))))
    (if (eof? thing)
	'fp-rpl-done
	(block
	  (pretty-print thing (standard-output))
	  (newline (standard-output))
	  (fp-rpl)))))
