;% Copyright (c) 1990-1994 The MITRE Corporation
;% 
;% Authors: W. M. Farmer, J. D. Guttman, F. J. Thayer
;%   
;% The MITRE Corporation (MITRE) provides this software to you without
;% charge to use, copy, modify or enhance for any legitimate purpose
;% provided you reproduce MITRE's copyright notice in any copy or
;% derivative work of this software.
;% 
;% This software is the copyright work of MITRE.  No ownership or other
;% proprietary interest in this software is granted you other than what
;% is granted in this license.
;% 
;% Any modification or enhancement of this software must identify the
;% part of this software that was modified, by whom and when, and must
;% inherit this license including its warranty disclaimers.
;% 
;% MITRE IS PROVIDING THE PRODUCT "AS IS" AND MAKES NO WARRANTY, EXPRESS
;% OR IMPLIED, AS TO THE ACCURACY, CAPABILITY, EFFICIENCY OR FUNCTIONING
;% OF THIS SOFTWARE AND DOCUMENTATION.  IN NO EVENT WILL MITRE BE LIABLE
;% FOR ANY GENERAL, CONSEQUENTIAL, INDIRECT, INCIDENTAL, EXEMPLARY OR
;% SPECIAL DAMAGES, EVEN IF MITRE HAS BEEN ADVISED OF THE POSSIBILITY OF
;% SUCH DAMAGES.
;% 
;% You, at your expense, hereby indemnify and hold harmless MITRE, its
;% Board of Trustees, officers, agents and employees, from any and all
;% liability or damages to third parties, including attorneys' fees,
;% court costs, and other related costs and expenses, arising out of your
;% use of this software irrespective of the cause of said liability.
;% 
;% The export from the United States or the subsequent reexport of this
;% software is subject to compliance with United States export control
;% and munitions control restrictions.  You agree that in the event you
;% seek to export this software or any derivative work thereof, you
;% assume full responsibility for obtaining all necessary export licenses
;% and approvals and for assuring compliance with applicable reexport
;% restrictions.
;% 
;% 
;% COPYRIGHT NOTICE INSERTED: Mon Apr 11 11:42:27 EDT 1994


;; CONVERT  7/24/89  dpr
;;; modified 8/17/89  dpr
(herald convert)

;;; converts imps inequalities to a list of conditions.

;;; DEFINITIONS:

;;; condition:   any equality, non-equality, or inequality.
;;; conds-expr:  imps expression for a condition or conjunction of conditions.
;;; cond-expr:   imps-expression for a single condition.  A cond-expr must
;;;              have all variables on the left side.  Each side must be
;;;              a single term or a sum of terms, each term a constant,
;;;              a constant times a constant, a variable, or a constant
;;;              times a variable.  Any conditions not of this form will
;;;              not be used for optimization.
;;; AOB:         a list (A OB)
;;; A:           the A matrix of the linear programming problem.
;;; OB:          a list (O B-element) where O is a valid operator
;;;              ('<= '~<= '< '~<) and B-element is an element of the B vector.


;;; TOOLS:
;;; (see below for top level functions.)


;;; CONDS-EXPR->AOB converts a "conditions-expression," an imps expression
;;; expressing a conjunction of conditions, to a list of three elements, the
;;; first being the A matrix, the second being a vector of lists, each
;;; containing the condition Operator (< ~< <= or ~<=) and the B value of
;;; the corresponding condition-expression.  The third element is a list of
;;; any conditions which are not in the correct form (eg non-linear).

(define (conds-expr->AOB processor expr)
  (let* ((cond-expr-list (ultimate-conjuncts (list expr)))
	 (m (length cond-expr-list))
	 (vars (variables expr))
	 (types-list
	  (generate-list m (lambda (i)
			     (cond-expr->OB processor (list-ref cond-expr-list i)))))
	 (cond-vector-list
	  (generate-list m (lambda (i)
			     (if (equal? (list-ref types-list i) 'other)
				 'other
				 (cond-expr->vector processor
				  (list-ref cond-expr-list i) vars)))))
	 (screen (map (lambda(i) (not (equal? i 'other))) cond-vector-list)))
    (list (list->vector (screened-elements cond-vector-list screen))
	  (list->vector (screened-elements types-list screen))
	  (screened-elements cond-expr-list (map not screen)))))

;;; COND-EXPR->VECTOR converts an imps condition expression into a vector
;;; containing coefficients of variables.  Constants are disregarded.  The
;;; coefficients are in the same order as the variables in VARS, a list
;;; of imps variable expressions.  'other is returned if any terms of
;;; the expression are in other than the correct form.

(define (cond-expr->vector processor cond-expr vars)
  (let ((left-vector (terms->vector processor
				    (left-side-terms processor cond-expr) vars))
	(right-vector (terms->vector processor
				     (right-side-terms processor cond-expr) vars)))
    (if (or (equal? left-vector 'other)
	    (equal? right-vector 'other))
	'other
	(vector-sub left-vector right-vector))))

;;; COND-EXPR->OB takes a single imps condition expression and returns
;;; an OB, a list containing the Operator, which is one of
;;; ( '<  '~<  '<=  '~<= ), where "~" means NOT, and the sum of the
;;; constant terms, considered to be on the right hand side of the
;;; equation.  If the expression has an invalid condition type,
;;; 'other is returned.

(define (cond-expr->OB processor cond-expr)
  (let ((condition-type (cond-expr-type processor cond-expr)))
    (if (not (equal? condition-type 'other))
	(list condition-type (cond-expr->constant processor cond-expr))
	condition-type)))

;;; COND-EXPR-TYPE returns the type of a condition expression, one of
;;; ( '<  '~<  '<=  '~<=).  If the expression is another type, 'other
;;; is returned.
    
(define (cond-expr-type processor expr)
  (let ((raw-expr (raw-expr expr)))
    (cond ((equation? expr)
	   'other)   ;;  '= not supported for now.
	  ((less-than? processor expr)
	   '<)
	  ((less-than-or-equals? processor expr)
	   '<=)
	  ((equation? raw-expr)
	   'other)   ;; '~= not supported for now.
	  ((less-than? processor raw-expr)
	   '~<)
	  ((less-than-or-equals? processor raw-expr)
	   '~<=)
	  (else 'other))))

;;; COND-EXPR->CONSTANT returns the sum of the constant terms of an imps
;;; condition expression, as if expressed on the right hand side of the 
;;; expression.

(define (cond-expr->constant processor expr)
  (let ((left-constant (terms->constant processor (left-side-terms processor expr)))
	(right-constant (terms->constant processor (right-side-terms processor expr))))
    (- right-constant left-constant)))

;;; TERMS->VECTOR takes a list of imps expressions for terms and returns a
;;; vector as specified above.  Each term must either be a constant or a
;;; product of two constants (which will be ignored), a variable, or a
;;; constant times a variable.  If any terms are not of this form, 'other
;;; will be returned in place of the vector.

(define (terms->vector processor terms vars)
  (let ((v (make-vector (length vars) 0)))
    (iterate loop ((i (-1+ (length terms))))
      (let ((term (list-ref terms i)))
	(cond ((scalar-constant? processor term))
	      ((memq? term vars)
	       (vset v (memqn term vars) '1))
	      ((not (= (length (expression-components term)) 3))
	       (set v 'other))
	      ((not (multiplication? processor term))
	       (set v 'other))
	      ((and (scalar-constant? processor (1starg term))
		    (scalar-constant? processor (2ndarg term))))
	      ((and (scalar-constant? processor (1starg term))
		    (memq? (2ndarg term) vars))
	       (vset v (memqn (2ndarg term) vars)
		     (scalar-constant->numerical-object processor (1starg term))))
	      (else (set v 'other)))
	(if (and (> i 0)
		 (not (equal? v 'other)))
	    (loop (-1+ i))
	    v)))))

;;; TERMS->CONSTANT accepts a list of imps expression terms and returns
;;; the sum of any constant terms.  Acceptable forms for terms are as for
;;; terms->vector.  However, bad forms will be treated as zero, and a number
;;; will always be returned.
	 
(define (terms->constant processor terms)
  (iterate loop ((i (-1+ (length terms)))
		 (constant 0))
    (if (>= i 0)
	(let ((term (list-ref terms i)))
	  (cond ((scalar-constant? processor term)
		 (loop (-1+ i) (+ constant (name term))))
		((and (multiplication? processor term)
		      (scalar-constant? processor (1starg term))
		      (scalar-constant? processor (2ndarg term)))
		 (loop (-1+ i)
		       (+ constant
			  (* (scalar-constant->numerical-object processor (1starg term))
			     (scalar-constant->mumerical-object processor (2ndarg term))))))
		(else (loop (-1+ i) constant))))
	constant)))	  


;;; LEFT-SIDE-TERMS returns a list of the terms on the left side of an
;;; imps condition expression, each term being an imps expression.
;;; LEFT-SIDE returns the imps expression for the entire left side of an
;;; imps condition expression.  An error is produced if the expr is not
;;; a cond-expr.

(define (left-side-terms processor expr)
  (expression->terms processor (left-side (raw-expr expr))))

(define (right-side-terms processor expr)
  (expression->terms processor (right-side (raw-expr expr))))

(define (left-side cond-expr)
  (cond ((application? cond-expr)
	 (1starg cond-expr))
	((equation? cond-expr)
	 (car (expression-components cond-expr)))
	(else (error "Invalid expression to left-side"))))

(define (right-side cond-expr)
  (cond ((application? cond-expr)
	 (2ndarg cond-expr))
	((equation? cond-expr)
	 (cadr (expression-components cond-expr)))
	(else (error "Invalid expression to right-side"))))

(define (expression->terms processor expr)
  (cond ((null? (expression-components expr))  ;single term
	 (list expr))
	((multiplication? processor expr)                ;single product term
	 (list expr))
	((addition? processor expr)                ;sum of terms
	 (associative-arguments expr))
	(error "Bad type in expression->terms")))

;;; RAW-EXPR, if given a NOT expression, will return the expression
;;; without the NOT.  If given an equation or an application, it will
;;; return the same expression.  Otherwise, garbage or error.

(define (raw-expr expr)
  (cond ((equation? expr) expr)
	((application? expr) expr)
	((negation? expr) (car (expression-components expr)))
	(else (error "Invalid expression to raw expr"))))
  

;;; LESS-THAN? returns #T if EXPR is an application and indicates
;;; a less-than condition.

(define (less-than? processor expr)
  (and (application? expr) (eq? (operator expr) (<r processor))))

(define (less-than-or-equals? processor expr)
  (and (application? expr) (eq? (operator expr) (<=r processor))))

;;; IMPS-***-EXPRESSION is the imps form of ***.

;;; SCREENED-ELEMENTS returns a list of all elements in data-list whose
;;; corresponding element in screen is true.  The two lists should be
;;; the same length.

(define (screened-elements data-list screen-list)
  (if (not (= (length data-list) (length screen-list)))
      (error "unequal lengths to screened-elements")
      (iterate loop ((data-list data-list)
		     (screen-list screen-list)
		     (output '()))
	(if data-list
	    (if (car screen-list)
		(loop (cdr data-list) (cdr screen-list)
		      (cons (car data-list) output))
		(loop (cdr data-list) (cdr screen-list) output))
	    (reverse output)))))
		


;;; TOP-LEVEL-FUNCTIONS:
;;; The following procedures ASSUME THAT THERE WILL BE NO = OR ~= CONDITIONS !!
;;; Or more accurately, that all conditions will be expressed in terms of
;;; either < or ~<.

;;; SOLVE-IMPS-EXPR-LPP takes a linear programming problem in the form
;;; of an AB imps expression and a C imps expression, optimizes,
;;; and returns a list with three elements.  The first is the solution
;;; vector X and the second is the max value, CX.  If the C expression
;;; contains constants, they are disregarded for optimization and added
;;; to the final CX value.  If the problem has no feasible solution or
;;; is unbounded, 'No-feasible-solution or 'Unbounded is returned in place
;;; of X and '() in place of CX.  The third element is a list of the
;;; conditions that were not in the correct form (eg non-linear) and were
;;; disregarded for optimization.

(define (solve-imps-expr-lpp processor C-expr AB-expr)
  (if (not (list-subset? (variables C-expr) (variables AB-expr)))
      (error "Incompatible variables")
      (let* ((vars (variables AB-expr))
	     (C (C-expr->vector processor C-expr vars))
	     (C-const (expr->const processor C-expr))
	     (AB (conds-expr->AB processor AB-expr))
	     (A (car AB))
	     (B (cadr AB))
	     (others (caddr AB))
	     (sol (solve-lpp C A B)))
	(if (pair? sol)
	    (let ((X (car sol))
		  (CX (cadr sol)))
	      (list (match-variables processor vars X) (+ CX C-const) others))
	    (list sol '() others)))))
(define norm-min solve-imps-expr-lpp)

(define (norm-max processor C-expr AB-expr)
  (if (not (list-subset? (variables C-expr) (variables AB-expr)))
      (error "Incompatible variables")
      (let* ((vars (variables AB-expr))
	     (C (vneg (C-expr->vector processor C-expr vars)))
	     (C-const (- (expr->const processor C-expr)))
	     (AB (conds-expr->AB processor AB-expr))
	     (A (car AB))
	     (B (cadr AB))
	     (others (caddr AB))
	     (sol (solve-lpp C A B)))
	(if (pair? sol)
	    (let ((X (car sol))
		  (CX (cadr sol)))
	      (list (match-variables processor vars X) (+ CX C-const) others))
	    (list sol '() others)))))


;;;THE FOLLOWING DOESN'T SEEM TO BE CALLED BY ANYTHING.
;;; REMOVE-IMPS-EXPR-REDUNDANCY takes an imps AB expression and returns
;;; an A matrix and a B vector with all redundancy removed.  If the
;;; conditions represent a non-feasible region, an error is produced.

(define (remove-imps-expr-redundancy processor AB-expr)
  (let* ((AB (conds-expr->AB processor AB-expr))
	 (A (car AB))
	 (B (cadr AB)))
    (remove-redundancy2 A B)))

;;; C-EXPR->VECTOR takes an imps C-expr and returns a vector of the
;;; coefficients of C in the same order as the variables of VARS, a list
;;; of imps expression variables.

(define (C-expr->vector processor C-expr vars)
  (let ((terms (associative-arguments C-expr)))
    (terms->vector processor terms vars)))

;;; CONDS-EXPR->AB uses conds-expr->AOB to convert an imps conditions
;;; expression to a list (A-matrix B-vector)

(define (conds-expr->AB processor expr)
  (let* ((AOB (conds-expr->AOB processor expr))
	 (A (car AOB))
	 (OB (cadr AOB))
	 (others (caddr AOB))
	 (B (make-vector (vector-length OB)))
	 (op# (lambda (i) (car (vref OB i))))
	 (val# (lambda (i) (cadr (vref OB i)))))
    (iterate loop ((i (-1+ (vector-length B))))
      (cond ((eq? (op# i) '<)
	     (vset B i (- (val# i) *small-fudge*)))  ;; *****
	    ((eq? (op# i) '<=)
	     (vset B i (val# i)))
	    ((eq? (op# i) '~<)
	     (block (vset B i (- (val# i)))
		    (negate-row! A i)))
	    ((eq? (op# i) '~<=)
	     (block (vset B i (- (+ (val# i) *small-fudge*)))
		    (negate-row! A i)))
	    (else (error "Invalid conditions")))
      (if (> i 0)
	  (loop (-1+ i))
	  (list A B others)))))

(lset *small-fudge* 0) ;; for converting < and > to <= and >=.

;;; MATCH-VARIABLES takes a list of imps-expression variables and a
;;; vector of corresponding values and returns a list of
;;; (variable-name value) lists.

;;;(define (match-variables processor vars-list value-vector)
;;;  (pair-elements vars-list
;;;		 (map (lambda (i) (find-constant (processor-language processor) i))
;;;		      (vector->list value-vector))))

;;;This returns a substitution ((var1 . val1)...)
(define (match-variables processor vars-list value-vector)
  (map cons vars-list
       (map (lambda (i) (find-constant (processor-language processor) i))
	    (vector->list value-vector))))

;;; EXPR->CONST returns the sum of the constant terms of an imps addition
;;; expression.

(define (expr->const processor expr)
  (terms->constant processor (expression->terms processor expr)))



;;; TEST: ERIC'S IDEA OF ELIMINATING X<Y TERMS.

(define (solve-imps-expr-lpp-eb-test processor C-expr AB-expr)
  (let* ((vars (variables AB-expr))
	 (C (C-expr->vector processor C-expr vars))
	 (C-const (expr->const processor C-expr))
	 (AB (conds-expr->AB processor AB-expr))
	 (A (car AB))
	 (B (cadr AB))
	 (others (caddr AB))
	 (A'B' (eb-subset a b))
	 (A' (car A'B'))
	 (B' (cadr A'B'))
	 (sol (solve-lpp C A' B')))
    (if (pair? sol)
	(let ((X (car sol))
	      (CX (cadr sol)))
	  (list (match-variables procssor vars X) (+ CX C-const) others))
	(list sol '() others))))

(define eb-min solve-imps-expr-lpp-eb-test)

(define (eb-max processor C-expr AB-expr)
  (let* ((vars (variables AB-expr))
	 (C (vneg (C-expr->vector processor C-expr vars)))
	 (C-const (- (expr->const processor C-expr)))
	 (AB (conds-expr->AB processor AB-expr))
	 (A (car AB))
	 (B (cadr AB))
	 (others (caddr AB))
	 (A'B' (eb-subset a b))
	 (A' (car A'B'))
	 (B' (cadr A'B'))
	 (sol (solve-lpp C A' B')))
c    (if (pair? sol)
	(let ((X (car sol))
	      (CX (cadr sol)))
	  (list (match-variables processor vars X) (+ CX C-const) others))
	(list sol '() others))))

(define (eb-subset a b)
  (iterate loop ((i (-1+ (vector-length b)))
		 (subset-a-list '())
		 (subset-b-list '()))
    (if (>= i 0)
	(if (and (= (vector-components-sum (vref a i)) 0)
		 (= (vref b i) 0))
	    (loop (-1+ i) subset-a-list subset-b-list)
	    (loop (-1+ i) (cons (vref a i) subset-a-list)
		  (cons (vref b i) subset-b-list)))
	(list (list->vector subset-a-list)
	      (list->vector subset-b-list)))))


(define (MAKE-LINEAR-INEQUALITY-TESTER processor)
  (lambda (context expr)
    (let* ((simp (context-simplify context expr (context-simplification-persistence)))
	   (AB (conds-expr->AB processor simp)))
      (if (feasible? (car AB ) (cadr AB)) simp (the-false)))))
