;% Copyright (c) 1990-1994 The MITRE Corporation
;% 
;% Authors: W. M. Farmer, J. D. Guttman, F. J. Thayer
;%   
;% The MITRE Corporation (MITRE) provides this software to you without
;% charge to use, copy, modify or enhance for any legitimate purpose
;% provided you reproduce MITRE's copyright notice in any copy or
;% derivative work of this software.
;% 
;% This software is the copyright work of MITRE.  No ownership or other
;% proprietary interest in this software is granted you other than what
;% is granted in this license.
;% 
;% Any modification or enhancement of this software must identify the
;% part of this software that was modified, by whom and when, and must
;% inherit this license including its warranty disclaimers.
;% 
;% MITRE IS PROVIDING THE PRODUCT "AS IS" AND MAKES NO WARRANTY, EXPRESS
;% OR IMPLIED, AS TO THE ACCURACY, CAPABILITY, EFFICIENCY OR FUNCTIONING
;% OF THIS SOFTWARE AND DOCUMENTATION.  IN NO EVENT WILL MITRE BE LIABLE
;% FOR ANY GENERAL, CONSEQUENTIAL, INDIRECT, INCIDENTAL, EXEMPLARY OR
;% SPECIAL DAMAGES, EVEN IF MITRE HAS BEEN ADVISED OF THE POSSIBILITY OF
;% SUCH DAMAGES.
;% 
;% You, at your expense, hereby indemnify and hold harmless MITRE, its
;% Board of Trustees, officers, agents and employees, from any and all
;% liability or damages to third parties, including attorneys' fees,
;% court costs, and other related costs and expenses, arising out of your
;% use of this software irrespective of the cause of said liability.
;% 
;% The export from the United States or the subsequent reexport of this
;% software is subject to compliance with United States export control
;% and munitions control restrictions.  You agree that in the event you
;% seek to export this software or any derivative work thereof, you
;% assume full responsibility for obtaining all necessary export licenses
;% and approvals and for assuring compliance with applicable reexport
;% restrictions.
;% 
;% 
;% COPYRIGHT NOTICE INSERTED: Mon Apr 11 11:42:27 EDT 1994


(herald simplex)

;;; SIMPLEX.T  7-11-89  dpr
;;; Solve linear programming problems using simplex method.

;;; A linear programming problem in the form:
;;;     Maximize:    CX
;;;     Subject to:  AX <= B,  X >= 0.
;;; is solved by calling (SOLVE-LPP C A B).
;;; C A and B are placed into a SIMPLEX-TABLEAU matrix, and the
;;; simplex method is employed.  References to the simplex-tableau
;;; are 1-based.

;;; FEASIBLE? returns #T if the system AX <= B has a feasible solution.

(define (feasible? A B)
  (let ((type (condition-types B)))
    (if (eq? type 'simple)
	'#t
        (let ((st-out (iterate-simplex (lpp->phase-1-st A B type))))
	  (if (<= (abs (st-optimum st-out)) *round-off-limit*)
	      '#t
	      '#f)))))

;;; SOLVE-LPP is called with the A matrix and the B and C vectors and,
;;; after solving the linear programming problem, returns output as specified by FORMAT-ST-OUT.  The problem is assumed to be maximization, all of the constraints being of the <= type.  >= constraints must be converted to <=.  Strict equality is handled by including both a >= and a <=, and strict inequality is not permitted.

(define (solve-lpp C A B)
  (format-st-out (optimize-lpp C A B) (num-cols A)))

;;; FORMAT-ST-OUT takes a simplex-tab and a number of variables
;;; and returns a list of two elements, the first being the X
;;; vector and the second being the value of CX, the optimum value.
;;; If there is no feasible solution, 'No-feasible-solution is returned.
;;; If the solution is unbounded, 'Unbounded is returned.

(define (format-st-out st-out num-vars)
  (if (matrix? st-out)
      (let ((ans (make-vector num-vars 0)))
	(iterate loop ((n num-vars))
	  (let ((position-number (unit-vector? (nth-col st-out (-1+ n)))))
	    (if position-number
		(vset1 ans n (st-B-ref st-out (1+ position-number))))
	    (if (> n 1)
		(loop (-1+ n))
		ans)))
	(list ans (st-optimum st-out)))
      st-out))
				   
 
;;; OPTIMIZE-LPP first determines if the lpp is "simple" (if all of the B
;;; values are positive, meaning either that all conditions are of the
;;; <= type or that any >= conditions have been dealt with during
;;; phase one of the two phase method).  If so, the lpp is solved directly,
;;; otherwise, >= conditions are taken into account and the two phase
;;; method is employed.
;;; A SIMPLEX-TABleau is defined to be a matrix, a vector of row vectors.
;;; It is a matrix representing a table containing an A, B, C and max,
;;; on which the simplex method can work.


(lset *round-off-limit* 0)   ;*****

(define (optimize-lpp C A B)
  (let ((type (condition-types B)))
    (if (eq? type 'simple)
	(iterate-simplex (lpp->simplex-tab C A B))
	(let* ((st1 (lpp->phase-1-st A B type))
	       (st2 (remove-artificial-vars (iterate-simplex st1))))
	  (if (not (< (abs (st-optimum st2)) *round-off-limit*))
	      'No-feasible-solution	
	      (iterate-simplex (canonize (replace-C st2 (vneg C)))))))))

;;; ITERATE-SIMPLEX: Given a simple simplex-tableau, returns an
;;; optimized simplex-tableau.

(define (iterate-simplex simplex-tab)
  (iterate loop ((st simplex-tab))
	   (let ((j (choose-entering st)))
;;	     (format t "~% ~a" (objective-row st))
	     (if (eq? j 'all-non-neg)
		 st
		 (let* ((ratio-vector (find-ratios st j))
			(i (find-least-non-neg ratio-vector)))
		   (if (eq? i 'all-neg)
		       'Unbounded
		       (loop (GJ-reduce st (1+ i) j))))))))

;;; LPP->SIMPLEX-TAB returns a simplex tableau for optimization 
;;; formed from C, A, and B.  LPP->PHASE-1-ST returns a simplex
;;; tableau for phase one of the two phase method, finding an
;;; initial basis.

(define (lpp->simplex-tab C A B)
  (let* ((m (num-rows A))
	 (n (num-cols A))
	 (mm (1+ m))
	 (nn (+ 1 m n))
	 (st (make-matrix mm nn 0)))
    (replace-A st A)
    (replace-B st B)
    (replace-C st (vneg C))
    (st-replace-region st 1 (+ n 1) (identity-matrix m))))
    
(define (lpp->phase-1-st A B type)
  (let* ((>=condition? (lambda (i)
			 (vref1 type i)))
	 (m (num-rows A))
	 (n (num-cols A))
	 (num>=s (vref1 type (1+ m)))
	 (mm (1+ m))
	 (nn (+ m n num>=s 1))
	 (st (make-matrix mm nn 0)))
    (replace-A st A)
    (replace-B st B)
    (switch>=sto<=s st m type)
    (st-replace-region st 1 (+ n 1) (identity-matrix m))
    (iterate loop ((i 1) (j 1))
	     (if (<= i m)
		 (if (>=condition? i)
		     (block (st-set st i (+ j m n) -1)
			    (st-C-set st (+ i n) 1)
			    (loop (1+ i) (1+ j)))
		     (loop (1+ i) j))
		 (canonize st)))))

(define (switch>=sto<=s st m type)
  (let ((>=condition? (lambda (i) (vref1 type i))))
    (iterate loop ((i m))
      (if (>=condition? i)
	  (negate-row! st (-1+ i)))
      (if (> i 1)
	  (loop (-1+ i))))))
    
;;; CANONIZE checks to be sure that a ST is in canonical form: any
;;; column vector of the A part of the ST that is a unit vector must
;;; have a 0 in the corresponding position of the objective row.  If not,
;;; GJ-reduction is employed to make it 0.

(define (canonize st)
  (let ((m (num-rows st))
	(n (num-cols st)))
    (iterate loop ((index (-1+ n)))
	     (let* ((col (vcopy (nth-col st (-1+ index))))
		    (r (unit-vector? col)))
	       (vset1 col m 0)
	       (let ((s (unit-vector? col)))
		 (if (and s (not r))
		     (block ;(format t "~% ~a" '+) commented out by jt
			    (set st (GJ-reduce st (1+ s) index)))
		     ;(format t "~% ~a" '-)
		     )
		 (if (> index 1)
		     (loop (-1+ index))
		     st))))))

;;; REMOVE-ARTIFICIAL-VARS removes artificial variables from ST by setting to 0
;;; all entries in the corresponding column.

(define (remove-artificial-vars st)
  (let ((m (num-rows st))
	(n (num-cols st))
	(objective (objective-row st)))
    (iterate loop ((j (-1+ n)))
	     (if (not (= (vref1 objective j) 0))
		 (iterate loop2 ((i m))
			  (st-set st i j 0)
			  (if (> i 1)
			      (loop2 (-1+ i)))))
	     (if (> j 1)
		 (loop (-1+ j))
		 st))))

;;; CONDITION-TYPES either returns 'SIMPLE is all of the B-values are positive,
;;; otherwise returns a vector one element longer then B, the last element
;;; being the number of >= conditions, and other elements indicating
;;; with a '>= that the corresponding row of A and B indicates a >=
;;; or with a '() that the corresponding condition is <=.

(define (condition-types B)
  (let* ((B-len (vector-length B))
	 (equations>=? (make-vector (1+ B-len))))
    (vset1 equations>=? (1+ B-len) 0)
    (iterate loop ((index B-len))
	     (if (> index 0)
		 (block (vset1 equations>=? index
			       (if (< (vref1 B index) 0)
				   (block
				    (vset1 equations>=? (1+ B-len)
					   (1+ (vref1
						equations>=? (1+ B-len))))
				    '>=)
				   '()))
			(loop (-1+ index)))
		 (if (= (vref1 equations>=? (1+ B-len)) 0)
		     'simple
		     equations>=?)))))

;;; CHOOSE-ENTERING  Find the row index of the entering variable by
;;; finding the column with the most negative value in the objective row.
;;; If no elements of the objective row are negative, then 'ALL-NON-NEG is
;;; returned, indicating that the optimum solution has been found.

(define (choose-entering simplex-tab)
  (let ((objective (objective-row simplex-tab))
	(n (num-cols simplex-tab)))
    (iterate loop ((j (-1+ n)) (i 'all-non-neg) (least 0))
	     (if (> j 0)
		 (if (< (vref1 objective j) least)
		     (loop (-1+ j) j (vref1 objective j))
		     (loop (-1+ j) i least))
		 i))))

;;; FIND-RATIOS returns a vector of ratios between the B column and the
;;; column of the entering variable.

(define (find-ratios simplex-tab i)
  (let ((m (-1+ (num-rows simplex-tab))))
    (generate-vector m (lambda (mm)
			 (if (> (st-ref simplex-tab (1+ mm) i) 0)
			     (/ (st-B-ref simplex-tab (1+ mm))
				(st-ref simplex-tab (1+ mm) i))
			     -1)))))

;;; CHOOSE-EXITING finds the column index of the exiting variable, which is
;;; the basic variable with a 1 in the pivot row.

(define (choose-exiting simplex-tab j)
  (iterate loop ((index (-1+ (num-cols simplex-tab))))
	   (if (> index 0)
	       (let ((v (nth-col simplex-tab (-1+ index))))
		 (if (and (unit-vector? v)
			  (= (vref1 v j) 1))
		     index
		     (loop (-1+ index))))
	       (error "no unit vector"))))

;;; GJ-REDUCE uses Gauss-Jordan elimination to produce a unit vector in row
;;; J with a 1 in the Ith row.

(define (GJ-reduce st i j)
  (let* ((mm (num-rows st))
	 (nn (num-cols st)))
    (generate-matrix
     mm nn
     (lambda (x y)
       (let ((m (1+ x))
	     (n (1+ y)))
	 (cond ((= m i)
		(/ (st-ref st m n)
		   (st-ref st i j)))
	       (else (- (st-ref st m n)
			(* (st-ref st i n)
			   (/ (st-ref st m j)
			      (st-ref  st i j)))))))))))


;;; 1-based reference and simplex-tableau support.

;;; Simplex-tableaus are one-based; following are some one-based
;;; vector commands to make coding easier.

(define (vref1 v n)
  (vref v (-1+ n)))
(define (vset1 v n new)
  (vset v (-1+ n) new))
(define (mref1 m i j)
  (mref m (-1+ i) (-1+ j)))
(define (mset1 m i j new)
  (mset m (-1+ i) (-1+ j) new))

(define (st-ref simplex-tab m n)
  (mref1 simplex-tab m n))
(define (st-set simplex-tab m n new)
  (mset1 simplex-tab m n new))

(define (st-A-ref simplex-tab m n)
  (st-ref simplex-tab m n))
(define (st-A-set simplex-tab m n new)
  (st-set simplex-tab m n new))

(define (st-B-ref simplex-tab m)
  (vref1 (nth-col simplex-tab (-1+ (num-cols simplex-tab))) m))
(define (st-B-set simplex-tab m new)
  (st-set simplex-tab m (num-cols simplex-tab) new))

(define (st-C-ref simplex-tab n)
  (vref1 (nth-row simplex-tab (-1+ (num-rows simplex-tab))) n))
(define (st-C-set simplex-tab n new)
  (vset1 (nth-row simplex-tab (-1+ (num-rows simplex-tab))) n new))

;;; REPLACE-A [B,C] replaces the A portion of ST with A.

(define (replace-A st A)
  (iterate loop ((m (num-rows A)) (n (num-cols A)))
	   (st-A-set st m n (mref1 A m n))
	   (if (> m 1)
	       (loop (-1+ m) n)
	       (if (> n 1)
		   (loop (num-rows A) (-1+ n))
		   st))))

(define (replace-B st B)
  (iterate loop ((m (vector-length B)))
	   (st-B-set st m (vref1 B m))
	   (if (> m 1)
	       (loop (-1+ m))
	       st)))

(define (replace-C st C)
  (iterate loop ((n (vector-length C)))
	   (st-C-set st n (vref1 C n))
	   (if (> n 1)
	       (loop (-1+ n))
	       st)))

;;; ST-REPLACE-REGION replaces a portion of ST with MATRIX, beginning
;;; at element (top-ref, left-ref).

(define (st-replace-region st top-ref left-ref matrix)
  (let ((m (num-rows matrix))
	(n (num-cols matrix)))
    (iterate loop ((i top-ref) (j left-ref))
	     (st-set st i j (mref matrix (- i top-ref) (- j left-ref)))
	     (if (< j (+ left-ref n -1))
		 (loop i (1+ j))
		 (if (< i (+ top-ref m -1))
		     (loop (1+ i) left-ref)
		     st)))))

(define (objective-row simplex-tab)
  (vref1 simplex-tab (num-rows simplex-tab)))
(define (st-optimum st)
  (let ((objective (objective-row st)))
    (vref1 objective (vector-length objective))))

;;;Data for use in testing procedures

;;;(define C '#(4 3))
;;;(define A '#(#(3 4) #(3 3) #(4 2)))
;;;(define B '#(12 10 8))
;;;
;;;(define C1 '#(4 3))
;;;(define A1 '#(#(3 4) #(3 3) #(4 2) #(-1 -1)))
;;;(define B1 '#(12 10 8 -1))
;;;
;;;;;NO FEASIBLE SOLUTION
;;;(define C2 '#(4 3))
;;;(define A2 '#(#(3 4) #(-1 -1) #(4 2)))
;;;(define B2 '#(12 -4 8))
;;;
;;;;;UNBOUNDED:
;;;(define C3 '#(1 2))
;;;(define A3 '#(#(-4 1) #(-1 -1) #(-1 -2) #(1 -1)))
;;;(define B3 '#(2 -3 -4 2))
