/*******************************************************************************
  anf_add_mod.c


  This file contains :

  anf_add_mod
  anf_z_add_mod

********************************************************************************/
#include "kant.h"
#include "integer.e"
#include "anf.h"


anf_elt
anf_add_mod  WITH_4_ARGS(
	order,		ord,
        integer_big,    modul,
	anf_elt,	alpha,
	anf_elt,	beta
)
/*******************************************************************************

Description:                               
 
	Addition of two algebraic numbers with coefficients over an arbitrary 
        order modulo a given integer modul..
 
	The summation of the coefficients (referring to the basis of the 
        order ord) is executed straightforward. The special cases that one of 
        the submitted numbers is a rational integer is handled separately. 
        We assume that all used alg. numbers do not have any denominators.
 
	If the coefficient order is Z the routine anf_z_add_mod is called.
 
	For the reason of speed this routine doesn't care for simplifying 
	coefficients/denominator. Call anf_simplify if denominators could occur.
 
	So if a non-integer is submitted the result will definitely be a 
        non-integer.


Calling sequence:

        gamma = anf_add_mod (ord,modul,alpha,beta);

              order             ord       : all arithmetic will be done in this
                                            order.                     
              integer_big       modul     : all operations are done modulo modul
              anf_elt           alpha     : an alg. number of order.
              anf_elt           beta      : dito.
              anf_elt           gamma     : the summ of alpha and beta.


History:
        MD November 1991    Version 10.11.91 (first modular version)
        JS August   1991    Version 20.09.91 (non modular)

*******************************************************************************/
{
	block_declarations;
 
	anf_elt		gamma;
	integer_small	deg;
        order		ordcoef;
        integer_big	multa, multb,h1;
	anf_elt		temp, tempaa, tempbb;
	integer_small	onepos;
	integer_small	i;
        
/*
    Special case: beta (and alpha?) is integer
*/
	if (anf_elt_is_integer(beta)) 
	{	
        	if (anf_elt_is_integer(alpha)) 
                {       
                  multa = integer_rem (alpha,modul);
                  multb = integer_rem (beta,modul);

                  h1 = modint_add (modul,multa,multb);     

                  integer_delref (multa);
                  integer_delref (multb);

                  return h1;
                }
		temp  = beta;
		beta  = alpha;
		alpha = temp;
	}
  
/*  
    Is the coefficient order equal Z? 
*/
 
        ordcoef = order_coef_order(ord);
	if (ring_type(ordcoef) == RING_Z) return anf_z_add_mod(ord, modul, alpha, beta);
 
/*
    Special case: only alpha is integer
*/
        
	if (anf_elt_is_integer(alpha))
        {
        	deg = order_rel_degree(ord);
		onepos = order_one_position(ord);
		if (onepos <= 0)
                        error_internal("anf_add_mod : One not in basis.");
		else
		{
                	anf_elt_alloc(gamma, deg);

			anf_elt_den(gamma) = 1;

                        for (i=1; i<=deg; ++i)
                        {
                           	if (i==onepos) continue;
				temp = anf_elt_coef(beta, i);
				anf_elt_coef(gamma,i) = anf_elt_incref(temp);
 			}
			temp = alpha; 
                        anf_elt_coef(gamma, onepos) = 
                          anf_add_mod(ordcoef, modul, temp, anf_elt_coef(beta,onepos));
			return gamma;
		}
	}
  
/*
    General case: Both are not integers
*/ 
	deg = order_rel_degree(ord);
 
 	anf_elt_alloc(gamma, deg);
 
	anf_elt_den(gamma) = 1;
 
 
	for (i=1; i<=deg; ++i)
	{
		tempaa = anf_elt_coef(alpha, i);
		tempbb = anf_elt_coef(beta,  i);
		anf_elt_coef (gamma, i) = anf_add_mod (ordcoef, modul, tempaa, tempbb);
	        anf_elt_delete(ordcoef, &tempaa);
        	anf_elt_delete(ordcoef, &tempbb);
	}
 
	integer_delref(multa);
	integer_delref(multb);
 
	return gamma;
}



anf_elt
anf_z_add_mod WITH_4_ARGS(
	order,		ord,
        integer_big,    modul,
	anf_elt,	alpha,
	anf_elt,	beta
)
/*******************************************************************************

Description:                               
 
                                               
Description

	Addition of two algebraic numbers with coefficients over Z modulo a given integer modul
	The user should call the generic routine anf_add_mod.
 
	The summation of the coefficients (referring to the basis of the order ord) 
	is executed straightforward. The special cases that one of the submitted numbers
	is a rational integer is handled separately. The function will not care 
	for any denominators.
 
	For the reason of speed this routine doesn't care for simplifying 
	coefficients/denominator. Call anf_simplify if denominators could occur.
        

Calling sequence:

        gamma = anf_z_add_mod (ord,modul,alpha,beta);

              order             ord       : all arithmetic will be done in this
                                            order.                     
              integer_big       modul     : all operations are done modulo modul
              anf_elt           alpha     : an alg. number of order.
              anf_elt           beta      : dito.
              anf_elt           gamma     : the summ of alpha and beta.

        The speciality of this function is, that we assume order to be defined 
        over Z.


History:
	92-09-09   JS     bug fixed
	91-10-30          first modular version
        JS August  1991   first version of anf_add          
	 

*******************************************************************************/
{
	block_declarations;
 
	anf_elt		gamma;
	integer_small	deg;
        integer_big	denom;	
        integer_big	temp, tempa, tempb;	
	anf_elt		tempaa,tempab;
	integer_small	onepos;
	integer_small	i;
  
/*
    Special case: beta (and alpha?) is integer
*/            
        puts (" ANF_Z_ADD_MOD");                  

	if (anf_elt_is_integer(beta)) 
	{	
        	if (anf_elt_is_integer(alpha)) 
                {       
                  
                  tempaa = integer_rem (alpha,modul);
                  tempab = integer_rem (beta,modul);

                  return modint_add(modul,tempaa, tempab);
                }
		
                tempaa = beta;
		beta   = alpha;
		alpha  = tempaa;
	}

/*
    Special case: only alpha is integer
*/
        
	if (anf_elt_is_integer(alpha))
        {                                          
                printf ("alpha ist integer");integer_write (alpha);puts ("");
               	tempa = integer_rem (alpha,modul);
                printf ("Rest ");integer_write (tempa);puts ("");

                deg = order_rel_degree(ord);                     
	   	onepos = order_one_position(ord);
	   	if (onepos <= 0)
		 	error_internal("anf_add_mod: One not in basis.");
               	else
	 	{
                    	anf_elt_alloc(gamma, deg);
	   		denom = 1;
			anf_elt_den(gamma) = 1;
                        for (i=1; i<=deg; ++i)
                        {
                        	if (i==onepos) continue;
				temp = integer_rem (anf_elt_coef(beta, i),modul);
				anf_elt_coef(gamma,i) = integer_incref(temp);
 		        }
                        tempb = integer_rem (anf_elt_coef(beta, onepos), modul);
                        anf_elt_coef(gamma, onepos) = 
                              modint_add (modul, temp, anf_elt_coef(beta, onepos));
		        return gamma;
		}
	}
  
/*
    General case: Both are not integers
*/ 
	deg = order_rel_degree(ord);
 	anf_elt_alloc(gamma, deg);
 
	anf_elt_den(gamma) = 1;
 
	for (i=1; i<=deg; ++i)
	{
		tempa = integer_rem (anf_elt_coef(alpha, i),modul);
		tempb = integer_rem (anf_elt_coef(beta, i),modul);
		
                printf ("koeff nr. %d von alpha",i);integer_write (tempa);puts ("");
                printf ("koeff nr. %d von beta ",i);integer_write (tempb);puts ("");
                
                anf_elt_coef(gamma, i) = modint_add(modul,tempa, tempb);
		integer_delref(tempa);
		integer_delref(tempb);
	}
 
	return gamma;
}
