/******************************************************************************
  anf_ideal_different.c
******************************************************************************/
 
#include "stdio.h"        
#include "kant.h"
#include "mat.h"
#include "anf.h"

t_logical
anf_ideal_different WITH_3_ARGS(
	order,		ord,
	anf_ideal,	id1,
	anf_ideal,	id2
)
/*******************************************************************************
 
Description:

	We compare two ideals.
	If both ideals are prime and their degrees of inertia are known we
	try to check if the two generators of id1 are also elements of id2.
	In all other cases we compare the transition matrices of the ideals.
	If an ideal is not given in Z-Basis-Presentation, the transition
	matrix will be computed in upper hnf.
	If one matrix is given in upper hnf and the other one in lower
	hnf, we will kill this lower hnf and compute the upper one.
	Remark that your input data may be changed !

  
Calling sequence:
 
        anf_ideal_different(ord, id1, id2);
 
        order   	ord:        order over which id1 and id2 are defined
        anf_ideal       id1,id2:    ideals to be compared

 
History:

	92-09-02 CO	case both ideals are prime, degrees
			of inertia are known
	92-07-13 CO     anf_ideal_simplify
	92-06-02 CO	anf_ideal_tran_assure_hnf_upper
	92-05-11 CO     first version

*******************************************************************************/
{       
	block_declarations;
 
	integer_small	n,i,j;
 
	order_must_be_over_z(ord);
 
/*******  special case: both ideals are prime, degrees of inertia ***/
/*******  are known, one ideal has 2-Element-Presentation         ***/

	if( (anf_ideal_degree_known(id1)) && (anf_ideal_degree_known(id2)) )
	{
	    if( anf_ideal_degree(id1) == anf_ideal_degree(id2) )
	    {
		if (anf_ideal_is_2(id1))
		{
		    if( (anf_elt_in_ideal(ord, anf_ideal_gen1(id1), id2))
		      &&(anf_elt_in_ideal(ord, anf_ideal_gen2(id1), id2)) )
			return 0;
		    else return 1;
		}
		if (anf_ideal_is_2(id2))
		{
		    if( (anf_elt_in_ideal(ord, anf_ideal_gen1(id2), id1))
		      &&(anf_elt_in_ideal(ord, anf_ideal_gen2(id2), id1)) )
			return 0;
		    else return 1;
		}
	    }
	    else return 1;
	}



/*******  compute Z-Basis-Presentations if not already given and ****/
/*******  compute upper hnf if transition matrix is not in hnf ******/

	anf_ideal_z_assure(ord, id1);
	if (!(anf_ideal_tran_hnf(id1)))
		anf_ideal_tran_assure_hnf_upper( ord, id1 );
	anf_ideal_z_assure(ord, id2);
	if(!anf_ideal_tran_is_hnf(id2))
		anf_ideal_tran_assure_hnf_upper( ord, id2 );
	                                                                  
/*******  case both ideals have transition matrix in lower hnf :        */
/*******  compare denominators and as many matrix elements as necessary */

	if( anf_ideal_tran_is_lower_hnf(id1) &&
	    anf_ideal_tran_is_lower_hnf(id2)    )
	{
		anf_ideal_simplify( ord, &id1 );
		anf_ideal_simplify( ord, &id2 );

		if( integer_compare( anf_ideal_tran_den(id1),
				     anf_ideal_tran_den(id2)) != 0 )
			return 1;                            

		n = order_abs_degree( ord );
		for( i = 1; i <= n; i++ )
			for( j = i; j <= n; j++ )
				if( integer_compare( mat_elt( anf_ideal_tran( id1 ), j,i ),
						     mat_elt( anf_ideal_tran( id2 ), j,i )) != 0)
          				return 1;
		return 0;
	}

/*******  One of our ideals has transition matrix in upper hnf.  *******/
/*******  If the other one has transition matrix in lower hnf,   *******/
/*******  we throw it away and compute the upper hnf, so that    *******/
/*******  we can compare.                                        *******/

	anf_ideal_tran_assure_hnf_upper( ord, id1 );
	anf_ideal_tran_assure_hnf_upper( ord, id2 );

/*******  now both ideals have transition matrix in upper hnf :         */
/*******  compare denominators and as many matrix elements as necessary */

	if( integer_compare( anf_ideal_tran_den(id1),
			     anf_ideal_tran_den(id2)) != 0 )
		return 1;                            

	n = order_abs_degree( ord );
	for( i = 1; i <= n; i++ )
		for( j = i; j <= n; j++ )
			if( integer_compare( mat_elt( anf_ideal_tran( id1 ), i,j ),
					     mat_elt( anf_ideal_tran( id2 ), i,j )) != 0)
       				return 1;
	return 0;

}
