/******************************************************************************
  anf_norm.c
 
  This file contains:
  anf_elt_norm_abs
  anf_norm
 
******************************************************************************/
   
#include "kant.h"
 
t_void
anf_norm WITH_4_ARGS(
	order,		ord,
	anf_elt,	alpha,
	integer_big *,	norm,
	integer_big *,	den
)
/*******************************************************************************
 
Description:

	Norm (over Q) of an algebraic number.
 
	This routine computes the norm of alpha over Q by computing iterated 
        relative norms.
 
	If alpha is given via its conjugates these are muliplied and the result 
        rounded. No check for precision is done.
                                           
 
Calling sequence:
 
	anf_elt_norm(ord, alpha, &norm, &den);
 
      	order         ord      = t_handle of order 
        anf_elt       alpha    = algebraic number
        integer_big   norm     = denominator of norm
        integer_big   den      = numerator

 
History:
                
        92-09-09 JS    no absolute value anymore (take anf_elt_norm_abs!)
	91-10-11 JS    minor changes
	91-09-01 JS    first version
 
*******************************************************************************/
{
	block_declarations;
 
	order		ordcoef;
	t_handle		reals;
	anf_elt		temp;
	t_real          temp0, temp2, tempr, tempi, prod; 
	integer_small	deg, i, j, r1, r2, r12;


/*  
    Is the order equal Z?
*/
 
	if (ring_type(ord) == RING_Z) 
	{
		*norm = integer_abs(alpha);
		*den = 1;   
		return;
	}
 
/*
    Is it a conjugate vector?
*/
 
	if (anf_elt_is_con(alpha)) 
	{
		deg = order_abs_degree(ord);
		r1  = order_r1(ord);
		r2  = order_r2(ord);
		r12 = r1 + r2;
		reals = order_reals(ord);
 
		prod = conv_int_to_real(reals, 1);
 
		for (i=1; i<=r1; ++i)
		{
			temp0 = prod;
			prod  = real_mult(reals, temp0, anf_con(alpha, i));
 
			real_delete(&temp0);
		}  
 
		for (i=r1+1, j=r12+1; i<=r12; ++i, ++j)
		{
			temp0 = prod;
			tempr = real_mult(reals, anf_con(alpha, i), 
						 anf_con(alpha, i));
			tempi = real_mult(reals, anf_con(alpha, j), 
						 anf_con(alpha, j));
			temp2 = real_add(reals, tempr, tempi);
			prod  = real_mult(reals, temp0, temp2);
 
			real_delete(&temp0);
			real_delete(&tempr);
			real_delete(&tempi);
			real_delete(&temp2);
		}
 
		if (r2)
		{
			temp  = integer_power(2, r2);
			temp2 = conv_int_to_real(reals, temp);
			temp0 = prod;
			prod  = real_divide(reals, temp0, temp2);
 
			integer_delref(temp);
			real_delete(&temp0);
			real_delete(&temp2);
		}
                         
		if (anf_print_level > 2)
		{
			printf("\nNorm before rounding: \n");
			real_write(reals, prod, 80);
			printf("\n");
		}
 
		*den = 1;
		*norm = conv_real_to_int_round(reals, prod);
 
		real_delete(&prod);
 
		return;
	}
 
/*
    We compute the absolute norm of the relative norm of alpha
*/
	anf_norm_rel(ord, alpha, &temp, den);
 
        ordcoef = order_coef_order(ord);
 
	if (ring_type(ordcoef) == RING_Z)
	{
		*norm = integer_incref(temp);
	}
	else    
	{
		anf_elt_norm(ordcoef, temp, norm, den);
	}
 
	anf_elt_delete(ordcoef, &temp);
 
	return;
}
 
 
t_void
anf_elt_norm_abs WITH_4_ARGS(
	order,		ord,
	anf_elt,	alpha,
	integer_big *,	norm,
	integer_big *,	den
)
/*******************************************************************************
 
Description:

	Absolute value of norm (over Q) of an algebraic number.
 
	This routine computes the (absolute) norm of alpha
	by computing iterated relative norms.
 
	If alpha is given via its conjugates these are muliplied and the result 
        rounded. No check for precision is done.
 
 
Calling sequence:
 
	anf_elt_norm_abs(ord, alpha, &norm, &den);
 
      	order         ord      = t_handle of order 
        anf_elt       alpha    = algebraic number
        integer_big   norm     = denominator of norm
        integer_big   den      = numerator
 
 
History:
                
        92-09-09 JS    first version
 
*******************************************************************************/
{
	block_declarations;
 
	integer_big	temp;
 
        anf_elt_norm(ord, alpha, &temp, den);
  
        *norm = integer_abs(temp);
 
        integer_delref(temp);
 
        return;
}
