/******************************************************************************
  lat_lll_reduce.c
******************************************************************************/
#include "kant.h"
#include "lattice.h"

lattice
lat_lll_reduce WITH_3_ARGS (
	lattice,	lat,
	matrix*,	trans,
	matrix*,	inv_trans
)
/******************************************************************************
 
Description:	Computes a LLL-reduced basis of a lattice.

Calling sequence:
 
	lll_lat = lat_lll_reduce(lat,&trans,&inv_trans)

	lattice		lll_lat		= t_handle of the new lattice
					  with LLL-reduced
					  basis (lll_lat = lat !!).  
	lattice		lat		= t_handle of lattice
	matrix		trans		= t_handle of matrix
				  	  lat_basis(lll_lat) = lat_basis(lat)*trans
	matrix		inv_trans	= t_handle of the inverse to trans
 
History:
 
	92-09-10 JS     removed bug when calling ring_elt_incref
	92-03-19 KW	minor changes
	92-03-06 KW	written
 
******************************************************************************/
{
	block_declarations;

	integer_small	i;            
	matrix		mat1,mat2;
	lattice		llllat;
	t_handle		Z;

	*trans = *inv_trans = MEM_NH;

	Z = m_z_str_incref(structure_z);

	llllat = lat_create();
	lat_rank(llllat) = lat_rank(lat);

	if (lat_basis_is_over_z(lat))
	{
		lll_int_d_reduce(lat_basis_ring(lat),
					lat_basis(lat),0.75,&lat_basis(llllat),trans);
	}
	else
	{
		mat1 = MEM_NH;             


		lll_real_d_reduce(lat_basis_ring(lat),
					lat_basis(lat),0.75,&lat_basis(llllat),&mat1);
/*
*		Remark: mat1 is over R. 
*/
		*trans = mat_real_to_mat_z(lat_basis_ring(lat),mat1);
		mat_delref(lat_basis_ring(lat),&mat1);
	}
	lat_basis_ring(llllat) = ring_incref(lat_basis_ring(lat));

	if (lat_gram(lat) && lat_gram_is_over_z(lat))
	{
/*
*		Save Z and beware of R.
*/
		mat1 = mat_ring_trans(Z,*trans);
		mat2 = mat_ring_mult(Z,lat_gram(lat),*trans);
		lat_gram(llllat) = mat_ring_mult(Z,mat1,mat2);

		lat_gram_ring(llllat) = ring_incref(lat_gram_ring(lat));
		
		mat_delref(Z,&mat1);
		mat_delref(Z,&mat2);
	}

	*inv_trans = mat_z_inverse(Z,*trans);
                
/*
*	administration of succ. mins.
*/

	if (lat_disc_known(lat))
	{
		lat_disc(llllat) = ring_elt_incref(lat_disc_ring(lat), lat_disc(lat));
		lat_disc_ring(llllat) = ring_incref(lat_disc_ring(lat));
	}
	if (lat_succ_mins(lat) != MEM_NH)
	{
		lat_succ_mins(llllat) = dyn_arr_incref(lat_succ_mins(lat));
	}
	lat_succ_min_vec_count_set(llllat,lat_succ_min_vec_count(lat));

	for (i=1;i<=lat_succ_min_vec_count(lat);i++)
	{
		lat_succ_min_vec(llllat,i)
		= lat_elt_move(lat_succ_min_vec(lat,i),*inv_trans);
	}

	ring_delete(&Z);

	return (llllat);
}
