/******************************************************************************
  order_disc_calc.c
 
  This file contains:
  order_disc_calc           (generic)
  order_disc_calc_pure      (internal)
  order_disc_calc_power     (internal)
  order_disc_calc_trans     (internal)
  order_disc_calc_real      
 
******************************************************************************/
 
#include "kant.h"
 

t_void
order_disc_calc WITH_1_ARG(
	order,		ord
)
/*******************************************************************************
 
Description:

Computes the discriminant of an order. Dependent on the situation this
routine calls order_disc_calc_pure/power/trans or real.
 
Until now is implemented:
   - pure equation order
   - orders with power basis
   - order given via transition matrix of a suborder 
   - arbitrary order if real basis is given (and perhaps no multiplication)
  
  
Calling sequence:
 
	order_disc_calc(ord);
 
      	order       ord      = t_handle of order 
 

History:                                 
                                                             
 	92-06-07 JS    order_mult need not be known if power basis
 	92-03-19 JS    deleting order_disc if it exists
	92-03-10 JS    slightly different strategy
 	91-10-01 JS    first version

*******************************************************************************/
{
	block_declarations;
 
	t_void order_disc_calc_pure();
	t_void order_disc_calc_power();
	t_void order_disc_calc_trans();
	t_void order_disc_calc_real();
                                   
 
	order_must_be_over_z(ord);
 
/*
    maybe the discriminant is known. In this case we have a restart
    with new precision. We then have to delete the discrimant first.
*/
        if (order_disc_known(ord)) integer_delref(order_disc(ord));
 
	if (order_basis_is_pure(ord))
	{
 		order_disc_calc_pure(ord);
	}
	else if (order_basis_is_rel(ord))
        {
		order_disc_calc_trans(ord);
        }
	else if (order_basis_is_power(ord))
	{
		order_disc_calc_power(ord);
	}
	else if (order_reals_known(ord))
 	{
	 	order_disc_calc_real(ord);
	}
	else
	{
		error_internal("order_disc_calc: Unknown situation.");
	}
 
	if (anf_print_level > 1)
	{
		printf("Order discriminant = ");
		integer_write(order_disc(ord));
		printf("\n");
	}
 
	return;
}
 
               
t_void
order_disc_calc_pure WITH_1_ARG(
	order,		ord
)
/*******************************************************************************
 
Description:

        Computes the discriminant of a pure equation order.
        Let X^n-a be the minimal polynial. Then the discriminant is
        n^n*a^(n-1).
                     
 
Calling sequence:
 
	order_disc_calc_pure(ord);
 
      	order       ord      = t_handle of order 
 

History:                                 
                                                             
 	92-04-02 JS    signature
 	91-10-01 JS    first version
  
*******************************************************************************/
{
	block_declarations;
 
	integer_small	deg, r1, r2;
	integer_big	temp0, temp1, temp2, temp3, alpha;
 
   
	deg   = order_rel_degree(ord);
	alpha = order_pure_gen(ord);
	r2    = order_r2(ord);
 
	if (r2 < 0) 
        {
                if (ring_type(order_coef_order(ord)) == RING_Z)
                { 
                        if (deg % 2)
                        { 
                          if (integer_sign(order_pure_gen(ord)) > 0) r1 = 2;
                          else  r1 = 0;
                        }
                        else r1 = 1;
                        r2 = deg - r1 - r1;
                        anf_order_set_sig(ord, r1, r2);
                }
                else
                {
                        error_internal("order_disc_calc: Signature not known.");
                }
        }
 
	temp0 = integer_abs(alpha);
	temp1 = integer_power(temp0, deg-1);
	temp2 = integer_power(deg, deg);
	temp3 = integer_mult(temp1, temp2);
 
	if (order_r2(ord) % 2) 
		order_disc(ord) = integer_negate(temp3);
	else
		order_disc(ord) = integer_incref(temp3);
 
	integer_delref(temp0); 
	integer_delref(temp1); 
	integer_delref(temp2); 
	integer_delref(temp3); 
  
	return;
}
 
 
 
t_void
order_disc_calc_power WITH_1_ARG(
	order,		ord
)
/*******************************************************************************
 
Description:

        Computes the discriminant of an order with power basis.
        In case the order is given over Z we can use ipoly_disc. In the
        other case we compute the first derivative of the generating polynomial. 
        If we substitute the variable by a root of the polynomial the 
        absolute value of the norm of this element is the absolute value 
        of the discriminant. To compute the signature we (in this 
        implementation) have to know the conjugates of a basis.
        These real values are computed automatically.
                     
 
Calling sequence:
 
	order_disc_calc_power(ord);
 
      	order       ord      = t_handle of order 
 

History:                                 
                                
        92-09-09 JS    anf_elt_norm_abs                
 	92-04-02 JS    ipoly_disc
	92-03-10 JS    order_set_reals if not done before (this is dirty!)
 	91-10-01 JS    first version

  
*******************************************************************************/
{
	block_declarations;
 
	t_poly	deriv;
	anf_elt		alpha;
	integer_big	norm, den, r1, r2;
 
 

        if (ring_type(order_coef_order(ord)) == RING_Z)
        {
		/*  this is the fast way    */
 
		order_disc(ord) = ipoly_disc(order_poly(ord), &r1, &r2);
                anf_order_set_sig(ord, r1, r2);
        }
        else
        {    
	        if (order_r2(ord) < 0) order_set_real_values(ord);

		r2    = order_r2(ord);
	 
		/*  this is nonsense since ipoly_... needs Pol over Z...   */
	 	deriv = poly_z_ith_deriv(structure_pring_z, order_poly(ord), 1);
		alpha = poly_to_anf_elt(ord, deriv);
		anf_elt_norm_abs(ord, alpha, &norm, &den);
		if (order_r2(ord) % 2) 
			order_disc(ord) = integer_negate(norm);
		else
			order_disc(ord) = integer_incref(norm);
 
		m_poly_z_delref(structure_pring_z, deriv); 
		anf_elt_delete(ord, &alpha);
		integer_delref(norm); 
        }

	return;
}
 
 
 
 
t_void
order_disc_calc_trans WITH_1_ARG(
	order,		ord
)
/*******************************************************************************
 
 
Description:

        Computes the discriminant of an order which is given
        via a transformation matrix referring to a suborder. Using
        the discriminant of the suborder the discriminant is 
        derived.
 
Calling sequence:
 
	order_disc_calc_trans(ord);
 
      	order       ord      = t_handle of order 
 

History:                                 
                                                             
 	91-10-01 JS    first version
  
*******************************************************************************/
{
	block_declarations;
 
	order		subord;
	integer_big	den;
 
                             
	subord   = order_suborder(ord);

	if (!order_disc_known(subord)) order_disc_calc(subord);
 
	den = integer_mult(order_index(ord), order_index(ord));
 
	order_disc(ord) = integer_div(order_disc(subord), den);
	
	integer_delref(den); 
  
	return;
}
 
 
 
 

t_void
order_disc_calc_real WITH_1_ARG(
	order,		ord
)
/*******************************************************************************
 
Description:

        Computes the discriminant of an order by using real arithmetic.
        The determinant of the conjugate matrix of the basis is computed.en
        The caller is responsible for precision.
 
Calling sequence:
 
	order_disc_calc_real(ord);
 
      	order       ord      = t_handle of order 
 

History:                                 
                                                             
 	91-10-01 JS    first version
 
*******************************************************************************/
{
	block_declarations;
 
	integer_small	deg;
	t_handle		reals;
	t_real		det, detsq, rdisc;
 

	deg   = order_rel_degree(ord);
	reals = order_reals(ord);

	mat_fld_det_sub(reals, order_basis_real(ord), &det);
	detsq  = real_mult(reals, det, det);
 
	if(anf_print_level > 2)
	{
		printf("order_disc_calc_real: Square Determinant = ");
		real_write(reals, detsq, 80);
		printf("\n");
	}
 
	if (order_r2(ord) % 2) 
	{
		rdisc = real_negate(reals, detsq);
		order_disc(ord) = conv_real_to_int_round(reals, rdisc);
		real_delete(&rdisc);
	}
	else
	{  
		order_disc(ord) = conv_real_to_int_round(reals, detsq);
	}
 
	real_delete(&det);
	real_delete(&detsq);
 
	return;
}
 
