#ifdef KANT
/******************************************************************************
  order_read.c
 
  This file contains:
  order_read
  v1_get_cmd
  v1_dyn_arr_read
  v1_dyn_arr_to_poly
  v1_str_clean

******************************************************************************/

#include <stdio.h>
#include <ctype.h>
#include <strings.h>
               
#include "kant.h"
#include "mat.h"
#include "ring.h"
#include "anf.h"
#include "poly.h"
#include "dyn_arr.h"

 
  
#define isdigsign(c)		(isdigit(c) || c=='-' || c=='+') 
#define ex(i)           	(dyn_arr_element(arrex,i))
#define dyn_arr_kill(a, length) {int ii; for (ii=0; ii<length; ++ii) \
				    integer_delref(dyn_arr_element(a, ii)); \
                                    dyn_arr_delete(&a);} \

#define STRLMAX         10000
 
t_logical
order_read WITH_1_ARG(
	order*,		ord
)
/*******************************************************************************
 
Description:
       
        Reading an order from stdin. The format of the order is as in
        the COMINP format of KANT Version 1. Not all features of KANT V1 are
        supported. Also if the units are integer_bigs they have to be 
        given with all coefficients in a row (no LWRITE!).
 
        This routine is completely compatible to order_write.
        For the input of an equation order you can simply type
        (F=)
        <coefs, decreasing, without leading coefficients>
        (END=)
        or using the (FLD=)-format (see KANT V1).
        The only other commands which are allowed (and should be used before
        the input of any order) are
        (PRECIS=)
        ..
        and
        (TSTFLG=)
        ..
        These commands initialize the real precision of the actual order and
        anf_print_level.
  
        The commands are not case dependent.
 
        Galois group and class group structures are not supported yet.
 
  
Calling sequence:
 
	if (order_read(&ord)) {...}
 
      	order         ord      = t_handle of order (if read) 
      
 
History:
 
	92-06-05 JS     exflag(1) = 0 treated as if -1
	92-03-19 JS	class group
	92-03-16 JS    	first version

*******************************************************************************/
{
	block_declarations;
 
	order		ord0;
        integer_small   deg, ucnt, i, j, w, prec, r1, r2, len, num;
        integer_big     den, locden, cden, fac, temp;
        anf_elt         alpha, beta;
        anf_ideal	id;
        matrix          mat0, trans;
        t_poly       pol;
	dyn_arr_handle  arr, arrex;
        char            str[STRLMAX]; 
        t_handle          Z;
        t_logical         order_was_read;
        void            v1_get_cmd(), v1_str_clean();
             
 
        order_was_read = FALSE; prec = 0;

/*                                                      
    Looking for a command string (containing "=)")
*/                             
 
        do 
        {
	        v1_get_cmd(str);
	                              
	        if (!strncmp(str,"(TSTFLG=)",9))
	        {       
	                /*  reading anf_print_level */
	                gets(str);
	                sscanf(str,"%d", &anf_print_level);
	        }
	        else if (!strncmp(str,"(PRECIS=)",9))
	        {
	                /*  Reading the precision for real computations*/
	                gets(str);
	                sscanf(str,"%d", &prec);
	        }
	        else if (!strncmp(str,"(F=)",4))
	        {
	                /*  Reading a polynomial */
	                Z = m_z_str_incref(structure_z);
                        gets(str);
                        arr = v1_dyn_arr_read(str);
                        pol = v1_dyn_arr_to_poly(Z, arr);
 
                        *ord = order_equation_create(Z, pol);
 
			dyn_arr_kill(arr, order_abs_degree(*ord));
	                m_poly_z_delref(structure_pring_z, pol);
	                ring_delete(&Z);
                                  
                        /* overread the next line (which should be (END=))  */
                        gets(str);
                        order_was_read = TRUE;
	        }
	        else if (!strncmp(str,"(FLD=)",6))
	        {       
                        /*  We start with the field degree and the Exflags */
	                Z = m_z_str_incref(structure_z);
                        arrex = v1_dyn_arr_read(str);
                        deg = dyn_arr_element(arrex, 0);
                                                        
                        /* Now the polynomial */
                        gets(str);                      
                        arr = v1_dyn_arr_read(str);
                        pol = v1_dyn_arr_to_poly(Z, arr);
                        dyn_arr_kill(arr, deg);

		        ord0 = order_equation_create(Z, pol); 
	                if (abs(ex(1)) == 1 || !ex(1))             
	                {           
	                        /* equation order */
	                        *ord = order_incref(ord0);
                                if (ex(1) == 1) order_set_is_maximal(*ord);
	                }
	                else if (abs(ex(1)) == 2)             
	                {
	                        /* Trafo matrix */      
                                /* first reading the matrix with denominators */
                                mat0  = mat_new(deg+1, deg);
				trans = mat_new(deg, deg);

                                /* storing the matrix and lcm of all denominators into cden */ 
                                cden = 1;
				for (j=1; j<=deg; ++j)
				{
		                        gets(str);                      
                		        arr = v1_dyn_arr_read(str);
                                        for (i=1; i<=deg+1; ++i)
	                                        mat_elt(mat0, i, j) = dyn_arr_element(arr, i-1);
                                        dyn_arr_delete(&arr);

                                        temp = cden;
                                        cden = integer_lcm(temp, mat_elt(mat0, deg+1, j));
                                        integer_delref(temp);
                                }     
                                
                                /* the common denominator now will be cden */
				for (j=1; j<=deg; ++j)
				{
					locden = mat_elt(mat0, deg+1, j);
					fac    = integer_div(cden, locden);
  
					for (i=1; i<=deg; ++i)
					mat_elt(trans, i, j) = integer_mult(fac, mat_elt(mat0, i, j));
 
					integer_delref(fac);
				}                    
    
                                /* we now have the order */
				*ord = order_trans(ord0, trans, cden);
		                if (ex(1) > 0) order_set_is_maximal(*ord);
 
                                integer_delref(cden);
				mat_delref(Z, &trans);
                                mat_delref(Z, &mat0);
	
                                if (ex(1) == 2) order_set_is_maximal(*ord);
	                }
	                else
	                {
	                        puts("Wrong EXFLAG(1)!");
		                m_poly_z_delref(structure_pring_z, pol);
	        	        order_delete(&ord0);
	                }              
	 
	                if (ucnt=ex(2))             
	                {            
	                        /* units */
                                order_units_count_set(*ord, ucnt);
                                for (j=1; j<=ucnt; ++j)
                                {
		                        gets(str);                      
                		        arr = v1_dyn_arr_read(str);
                                        anf_elt_alloc(alpha, deg);
                                        anf_elt_den(alpha) = 1;
                                        for (i=1; i<=deg; ++i) 
                                                anf_elt_coef(alpha, i) = dyn_arr_element(arr, i-1);
                                        order_unit(*ord, j) = alpha;

		                        dyn_arr_delete(&arr);
                                }
                                if (ex(5) == ucnt)
                                        order_set_units_are_fund(*ord);
	                }
 
	                if (ex(3)>0)             
	                {            
	                        /* long units */
                                for (i=ex(3)*deg*2+ex(3)+1; i; i--) gets(str);
	                        puts("Sorry, long units overread.");
	                } 
                         
                        /* torsion subgroup */
                        if (ex(4) == 2)
                        {
				order_torsion_rank(*ord) =  2;
				order_torsion_unit(*ord) = -1;
                        }
			else if(ex(4) > 2)
			{
				order_torsion_rank(*ord) = ex(4);
 
				anf_elt_alloc(alpha, deg);
				anf_elt_den(alpha) = 1;
	                        gets(str);                      
               		        arr = v1_dyn_arr_read(str);
				for (i=1; i<=deg; ++i)
					anf_elt_coef(alpha, i) = dyn_arr_element(arr, i-1);
				order_torsion_unit(*ord) = alpha;
                                dyn_arr_delete(&arr);
			}
 
                        if (ex(6))
            		{
            			gets(str);
                                printf("Sorry, Galois group structure overread.\n");
            		}
                        if (ex(7) && ex(7) !=1)
                        {
	                        /* class group, first class number and orders */
	                        gets(str);                      
               		        arr = v1_dyn_arr_read(str);
                                len = dyn_arr_curr_length(arr);
                                order_class_number(*ord) = dyn_arr_element(arr, 0);
                                order_class_group_order_set(*ord, len-2);
                                for (i=2; i<len; i++)
			             order_class_group_factor_order(*ord,i-1) = dyn_arr_element(arr, i);
                                dyn_arr_delete(&arr);
                                len -=2;
                                /* now the number of ideals */
				gets(str); sscanf(str,"%d",&num);
                                order_class_group_gen_count_set(*ord, num);
                                /* now all ideals */
                                for (i=1; i<=num; ++i)
                                {
					gets(str);
		              	        arr = v1_dyn_arr_read(str);
                                        /* second generator */
                                        anf_elt_alloc(alpha, deg);
                                        anf_elt_den(alpha) = 1;
                                        for (j=1; j<=deg; ++j)
                                                anf_elt_coef(alpha, j) = dyn_arr_element(arr, j+1);
                                        /* first generator */
                                        beta = dyn_arr_element(arr, 1);
                                        id = anf_ideal_2_create(*ord, beta, alpha);
                                        /* minimum and degree are both known */
                                        anf_ideal_min(id) = beta;
                                        anf_ideal_degree(id) = dyn_arr_element(arr, 0);
                                        /* we are done */
                                        order_class_group_gen(*ord, i) = id;
	                                dyn_arr_delete(&arr);
                                        anf_elt_delete(*ord, &alpha);
                                }
                                /* now the exponent matrix */
                                mat0 = mat_new(num, len);
                                for (j=1; j<=len; j++)
                                {
		                        gets(str);                      
               			        arr = v1_dyn_arr_read(str);
                                        for (i=1; i<=num; i++)
                                            mat_elt(mat0, i, j) =  dyn_arr_element(arr, i-1);
	                                dyn_arr_delete(&arr);
                                }
				order_class_group_genexps(*ord) = mat0;
                        } 
                        else if(ex(7) == 1)
                        {
                                order_class_number(*ord) = 1;
                        }

                        /* We might have a chance to set the signature */
        
                        if (ex(10) > 0)
                        {
                                r1 = ex(10);
                                r2 = (deg - r1)/2;
                                anf_order_set_sig(*ord, r1, r2);
                        }
	
	                m_poly_z_delref(structure_pring_z, pol);
                        dyn_arr_kill(arrex, 11);
	                order_delete(&ord0);
	                ring_delete(&Z);
 
                        order_was_read = TRUE;
	        }
	        else if (strlen(str)>3)
	        {
	                /*  Error */
	                puts("Sorry, this format is not supported.");
	        }
	        else 
	        {
	                return FALSE;
	        }
        }
        while(!order_was_read); 

        if (prec>0) order_set_reals(*ord, prec);
 
        return TRUE;
}         
 
 
           
          
void
v1_get_cmd WITH_1_ARG(
        char *, str
)
/*******************************************************************************
 
Description:
       
       returns a Command in KANT V1 COMINP style.
       The command starts at position 0.
 
  
History:
 
	92-03-13 JS    first version

*******************************************************************************/
{
        char            s1[STRLMAX], s2[STRLMAX];
        integer_small   i, len1, len2;

        for(;;)
        {       
                s1[0] = EOF; s1[1] = '\0';
           
                gets(s1);
 
                if ((strlen(s1) == 1 && s1[0] == 'e') || s1[0] == EOF) 
                {
                        /* This looks like EOF */
			strcpy(str, s1);
                        return;
                }
                if (index(s1,'*'))
                {                                
                        /* Comments start with a '*' */
                        len1 = strlen(index(s1,'*'));
                        len2 = strlen(s1) - len1;
                   	strncpy(s1,s1,len2);
                        s1[len2] = '\0';

                }
 
                if (index(s1,'='))
		{          
                   	strcpy(s2,index(s1,'='));
			if(s2[1] == ')') 
			{
	                        /* Commands end with "=)" */
				strcpy(str, index(s1,'('));
				for(i=0; i<strlen(str); ++i) 
				if (islower(*(str+i))) *(str+i)=toupper(*(str+i));
				return;
                        }
                }
        }
 
}
 
 
dyn_arr_handle
v1_dyn_arr_read WITH_1_ARG(
        char *, 	str
)
/*******************************************************************************
 
Description:
       
       Reads an array of integers out of a string. The string will be cleaned
       before.
  
History:
 
	92-03-16 JS    first version

*******************************************************************************/
{
	block_declarations;
 
        int             i, len1, len2, cnt;
        dyn_arr_handle  arr;
        char            s1[STRLMAX];
	t_void          v1_str_clean();
                                                              
/*
    The string must be cleaned
*/
        v1_str_clean(str);
 
        len1 = strlen(str);
  
/*
    How many numbers are in str? (= # of blanks)
*/
        for (cnt=0, i=0; i<len1; ++i) if(*(str+i)==' ') cnt++;
 
	arr = dyn_arr_alloc(cnt);
                               
/*
    We are reading from the back
*/ 
        strcpy(s1, str); len2 = 0;
	for (i=cnt-1; i>=0; --i)
        {
                strcpy(s1, rindex(s1, ' '));
		dyn_arr_element(arr, i) = integer_read(&s1[1]);
                len2 = len2 + strlen(s1);
                strncpy(s1, str, len1-len2);
                s1[len1-len2] = '\0';
        }
	dyn_arr_curr_length(arr) = cnt;
   
        return arr;
}
 
   
t_poly
v1_dyn_arr_to_poly WITH_2_ARGS(
        t_handle,         Z,
        dyn_arr_handle, arr
)
/*******************************************************************************
 
Description:
       
       Converts a dynamic array (as read with v1_dyn_arr_read) into a monic 
       integer polynomial.
       The coefficients are as in KANT V1 COMINP.
  
History:
 
	92-03-16 JS    first version

*******************************************************************************/
{
	block_declarations;
 
        t_poly       pol;
        dyn_arr_handle  arr1;
        integer_small   deg, i, len1, len2;
 
        deg = dyn_arr_curr_length(arr);
        
	arr1 = dyn_arr_alloc(deg+1);
	dyn_arr_element(arr1, deg) = 1;
                                
	for (i=0; i < deg; ++i)
		dyn_arr_element(arr1, i) = dyn_arr_element(arr, deg-i-1);
 
	dyn_arr_curr_length(arr1) = deg + 1;
 
	pol = dyn_arr_to_poly(Z, arr1);
 
	dyn_arr_delete(&arr1);
 
        return pol;
}
 
          
void
v1_str_clean WITH_1_ARG(
        char *, str
)
/*******************************************************************************
 
Description:
       
       Cleans a string. On exit, str only consists of (signed) numbers,
       between each two and at the beginning is exactly one blank.
       The last character is no blank. So, on exit, there are exactly as many
       blanks in str as numbers and every blank is followed by a number.
  
History:
 
	92-03-13 JS    first version

*******************************************************************************/
{
        int             i, len, delta;

        len = strlen(str);
 
/*
    shifting: A blank must be at the beginning
*/
        for (i=len; i; i--) *(str+i) = *(str+i-1); *str = ' '; len++;
  
/*
    killing double blanks
*/
        if (!isdigsign(*str)) *str = ' ';
        for (i=1, *str++, delta=0; i<len; ++i, *str++)
        {                        
                if (!isdigsign(*str)) *str = ' ';
                if (*(str-delta-1) == ' ' && *str == ' ') delta++;
               	*(str-delta) = *str;
        }      
        *(str - delta) = '\0';
 
/*
    clearing the end
*/
        if (*(str-delta-1) == ' ')  *(str-delta-1) = '\0';

 
        return; 
}
#endif
