/*******************************************************************************
  order_relations_find.c 

    order_relations_find
    next_ideal_in_fac_basis
    eval_t2_bound_const
    eval_t2_bound
********************************************************************************/

                                                             
#include <stdio.h>
#include "kant.h"
#include "integer.e" 
#include "dyn_arr.h"
#include "poly.h"      /* Kein Wort darueber */
#include "poly_z_faclst.h"
#include "anf.h"
#include "lattice.h"
#include "conv.e"




void     
order_relations_find WITH_3_ARGS (order         , ord,
                                  integer_big   , t2_bound_flag,
                                  integer_small , no_of_anf_elts)
/*******************************************************************************
Description:
                        
   Finds "no_of_anf_elts" algebraic numbers in each ideal of the
   factor basis stored in order which splitt over this basis.


Calling sequence:                                            

       order         ord             : the order we are working with
       integer_big   t2_bound_flag   : We are looking for alg. numb. by searching
                                       for points in lattices.
                                       there are two cases  :
                                           1) flag == 0 : we compute a bound
                                                          for the lattice. 
                                           2) flag != 0 : We take this bound.
  
       integer_small no_of_anf_elts  : the number of alg.numbers we 
                                       are looking for
                                                           


    order_relations_find (ord,t2_bound_flag,no_of_anf_elts):


                            
History:                                 
           

 	92-06-12 JS 	direction 'down' instead of 'up'
        92-04-92 MD     first version

********************************************************************************/
{

	block_declarations;         
        t_handle              R,Z;     

        anf_ideal           ideal;   
        lattice             lat,lll_lat;
        lat_enum_env        lat_env,lll_lat_env; 
        lat_elt             lat_vec;                         

        matrix              trans,inv_trans;

        anf_elt             alpha;

               
        integer_big         t2_bound;
        integer_small       bound_l,bound_u; 
        integer_small       found,found_total;
        integer_small       pos_prime,pos_ideal;


        float               hit_rate;
 
        if (anf_print_level > 0)
        {
           	cay_print
		("order_relations_find: Looking for %d relations...", 
		  no_of_anf_elts * order_fac_basis_ideals_count(ord));
           	if (t2_bound_flag) cay_print
		("\nUser supplied T2 bound: %d.", t2_bound_flag);
        }      
 
        order_reals_assure(ord);

/* We want to find a certain number of alg. numbers in each prime ideal of the */
/* factor basis that split completely over the factor basis given with the     */
/* order ord.                                                                  */
        
  R = order_reals(ord);

  pos_prime           = 0;          /* Pos. of last rat. prime in fac_basis    */
  pos_ideal           = 0;          /* Pos. of last prime ideal in fac_basis   */

  found_total = 0;                  /* The total number of alg. numbers        */
                                    /* found in all ideals.                    */
                                                               
    
/* Now set the value of t2_bound :                                             */
/* This depends on the value of t2_bound_flag  :                               */
/*                                                                             */
/*                   t2_bound_flag == 0  : compute a t2_bound                  */
/*                                 != 0  : t2_bound = t2_bound_flag            */
           
  if  (t2_bound_flag == 0) 
    t2_bound = eval_t2_bound_const (ord);
  else
    t2_bound = integer_incref (t2_bound_flag);                                         


                                   
  while ((ideal = next_ideal_in_fac_basis (ord,&pos_prime,&pos_ideal)) != MEM_NH)
  {                
    if (anf_print_level >1)
    {
      puts ("");anf_ideal_2_write (ord,ideal);printf ("  ");
    }

    found   = 0;       /* No alg. numbers have been found              */
                       /* (numbers which splitt over the factor basis) */

    bound_l = 0;       /* Set the searching bound`s for the lattice    */
    bound_u = eval_t2_bound (ord,ideal,t2_bound,t2_bound_flag);



    anf_ideal_lat (ord, ideal, &lat, &lat_env);

    lll_lat = lat_lll_reduce(lat,&trans,&inv_trans);
    lll_lat_env = lat_enum_create(lll_lat);

    lat_enum_request_set_next(lll_lat_env);
    lat_enum_strategy_set_down(lll_lat_env);          
    
    while (found < no_of_anf_elts)     /* We still need alg. numbers */
    {                     
      if (anf_print_level >2) printf ("Upper T2 bound: %d   Lower T2 bound: %d\n",bound_u,bound_l);

      if (bound_l) lat_enum_lbound(lll_lat_env) = conv_int_to_real(R,bound_l);
                   lat_enum_ubound(lll_lat_env) = conv_int_to_real(R,bound_u);

      lat_enum_status_set_new(lll_lat_env);
 
      while (lat_enum(lll_lat, lll_lat_env) && (found <no_of_anf_elts))
      {    
        found_total++;          

        lat_vec  = lat_elt_move (lll_lat,lat_enum_act_coefs(lll_lat_env),trans);
        alpha    = lat_elt_ideal_to_anf_elt (lat, lat_vec, ord,ideal);

               
       
        if (anf_elt_factorize (ord,alpha)) 
        { 
          if (anf_print_level >2) printf ("* ");      
          found ++;        
        }
        else
        {
          if (anf_print_level >2) printf (". ");      
          else
            if (anf_print_level >4)
            {
              printf("(%d) Number: ",found); anf_elt_write(ord, alpha);puts ("");
            }
        }


                    
        lat_elt_delete (lll_lat,&lat_vec);
        anf_elt_delete(ord, &alpha);
      }

      if(bound_l)       
        real_delete(&lat_enum_lbound(lll_lat_env));

      real_delete(&lat_enum_ubound(lll_lat_env));
                      
/* Now change the upper and lower bound for searching */
           
      integer_delref (bound_l);
      bound_l = bound_u;
 
      bound_u = integer_mult (2,bound_l);
         
      if (anf_print_level >1) printf ("| ");      

    }     

    Z = m_z_str_incref(structure_z);
    mat_delref(Z,&trans);
    mat_delref(Z,&inv_trans);
    ring_delete(&Z);	



    integer_delref (bound_l);
    integer_delref (bound_u);

    lat_enum_delete(lll_lat, &lll_lat_env);
    lat_enum_delete(lat, &lat_env);

    lat_delete(&lll_lat);
    lat_delete(&lat); 


  }
  
  integer_delref (t2_bound);

  if (anf_print_level >0) 
  {
    hit_rate = (order_fac_basis_ideals_count (ord)*no_of_anf_elts) / (1.0*found_total);
    printf("\nNumbers checked: %d   and factorized: %d  ==>",found_total,
            order_fac_basis_ideals_count (ord)*no_of_anf_elts);
    printf("   Hit rate: %f\n",hit_rate);
  }
 
  
} 




anf_ideal
next_ideal_in_fac_basis WITH_3_ARGS (order           , ord      ,
                                     integer_small*  , pos_prime,
                                     integer_small*  , pos_ideal )
/*******************************************************************************
Description:
                        
   returns the next ideal relative to the given position in the factor basis.

Calling sequence:                                            

       order         ord             : the order we are working with
       integer_small pos_prime       : the index of the actual rat. prime in
                                       the factor basis
       integer_small pos_ideal       : the index of the actual idealin
                                       the list of ideals belonging to 
                                       the actual rat. prime.
                                                           


    next_ideal_in_fac_basis (ord,&pos_prime,&pos_ideal):


                            
History:                                 
           

   MD 09.04.92    written                    

********************************************************************************/
{ 
    anf_ideal       ideal;

                   

/* This part computes the next ideal - position              */
/* It will not be checked whether or not this is a valid one */
/* This will be done later !!                                */

  if (*pos_prime == 0)      
/* The next possible ideal will be at position : */
  {
    *pos_prime = 1;
    *pos_ideal = 0;
  }                                            
  else
    if ((*pos_ideal+1) == m_poly_z_faclst_len (order_fac_basis_ideals (ord,*pos_prime)))
/* There are no more ideals in the factor basis above the actual rat. prime */
    {
      *pos_prime= *pos_prime+1;
      *pos_ideal = 0;
    }                
    else
/* There are more ideals above the actual prime number. */
      {
        *pos_ideal= *pos_ideal+1;
      }

  if (*pos_prime > order_fac_basis_len (ord)) /* There are no more ideals in the */
                                              /* factor basis                    */
    ideal = MEM_NH;
  else
    ideal = m_poly_z_faclst_factor (order_fac_basis_ideals (ord,*pos_prime),*pos_ideal);
    
  return ideal;
}
                 




integer_big 
eval_t2_bound_const WITH_1_ARG (order,   ord)
/*******************************************************************************
Description:
                                  
   returns a constant for the lattice belonging to all ideals in the factor 
   basis. 

Calling sequence:                                            

       order         ord             : the order we are working with


    eval_t2_bound_const (ord);


                            
History:                                 
           

   MD 09.04.92    written                    

********************************************************************************/
{
       block_declarations;         

       integer_big     h1,h2;
       integer_small   sign;



  integer_root (order_fac_basis_prime (ord,1),order_abs_degree (ord),&h1,&sign);                           

  h2 = integer_power (h1,4);
  integer_delref (h1);                                     
  
  h1 = integer_mult (order_abs_degree (ord),h2);
  integer_delref  (h2);
  

  return h1;

}



integer_big 
eval_t2_bound  WITH_4_ARGS (order        , ord            ,
                            anf_ideal    , ideal          ,
                            integer_big  , t2_bound_const ,
                            integer_big  , t2_bound_flag   )
/*******************************************************************************
Description:
                                  
   returns the upper bound for the lattice enumeration for the lattice belonging 
   to the ideal "ideal".

Calling sequence:                                            

       order         ord            : the order we are working with
       anf_ideal     ideal          : the actual ideal.
       integer_big   const          : the constant computed by 
                                         "eval_t2_bound_const"
       integer_big   flag           : same def. as decribed in 
                                           "order_relations_find"


    eval_t2_bound (ord,ideal,const,flag);


                            
History:                                 
           

   MD 09.04.92    written                    

********************************************************************************/
{
       block_declarations;                 



       integer_big      temp1,temp2;
       integer_big      value;
                                    

       integer_small    sign,n;              

  if (t2_bound_flag == 0) 
  {
    anf_ideal_norm (ord,ideal,&temp1,&temp2); 
    integer_delref (temp2);    /* We care just for integral ideals */
    n = order_abs_degree (ord);
    integer_root (temp1,n,&temp2,&sign);                           
    integer_delref (temp1);

    temp1 = integer_power (temp2,2);
    integer_delref (temp2);                       

    value = integer_mult (t2_bound_const,temp1);
    integer_delref (temp1);               

  }                          
  else
    value = integer_incref (t2_bound_flag);



  return value;

}
