#include "defs.h"

Logical
integer_is_prime_small WITH_1_ARG(
    register t_int,    p
)
/*
** Given a small integer p which is odd, return TRUE if p is prime.
** Small in this instance means single precision (i.e. < IBETA).  If
** IBETA changes, the list of exceptions will need to be changed as
** well.
**
** This uses the strong pseudoprime test outlined in H. Riesel,
** "Prime Numbers and Computer Methods for Factorization", Birkhauser,
** 1985, pp 98-99.
*/
{
	register t_int    s;
	register t_int    d;
	register t_int    base;
	register t_int    k;
	register t_int    r;

	/*
	 * Test for small cases, and do simple trial division
	 */
	if (p == 2 || p == 3 || p == 5)
		return TRUE;
	if (p == 1 || integer_gcd(p, 30) > 1)
		return FALSE;
	/*
	 * Calculate p - 1 = d * 2^s, d odd
	 */
	s = 0;
	d = p - 1;
	while (d % 2 == 0)
	{
		s++;
		d /= 2;
	}
	for (base = 2; base <= 5; base++)
	{
		if (base == 4)
			continue;
		if ((k = mintd_exp(p, base, d)) == 1)
			continue;
		for (r = 0; r < s; r++)
		{
			if (k == p - 1)
				break;
			if (r < s - 1)
				k = mintd_exp(p, k, 2);
		}
		if (r >= s)
			/* Failed pseudoprime test */
			return FALSE;
	}
	/*
	 * Check the four exceptions < IBETA.
	 *
	 * Note that Riesel gives the remaining 9 exceptions for p < 25 * 10^9:
	 *	 3,215,031,751
	 *	 5,764,643,587
	 *	 6,770,862,367
	 *	14,386,156,093
	 *	15,579,919,981
	 *	18,459,366,157
	 *	19,887,974,881
	 *	21,276,028,621
	 */
	if
	(
		p ==   25326001 ||
		p ==  161304001 ||
		p ==  960946321 ||
		p == 1157839381
	)
		return FALSE;
	/*
	 * Passed all pseudoprime tests, and not one of the exceptions
	 */
	return TRUE;
}

