/*  inthdl_quot_rem.c
*/

#include "defs.h"
#include "inthdl.e"
#include "intbig.h"
 
/*
The following checks whether the sign of a remainder r is inappropriate for
the sign of a divisor b, that is, whether the remainder is nonzero with sign
opposite to that of the divisor.  It assumes that b is known to be non-zero,
so that if not positive it is negative.
*/

#define wrong_sign(b, r)	((b) > 0 ? (r) < 0 : (r) > 0)


void
inthdl_quot_rem_beta	WITH_4_ARGS(
    inthdl_handle,	ahdl,
    t_int,	b,
    inthdl_handle,	qhdl,
    t_int  *,	rptr
)
/*
Given big integer a in block ahdl, and nonzero small integer b, calculate the
quotient q and remainder r as the unique integers such that

    a = q * b + r,  with    0 <= r <  b if b > 0
		    and     b <  r <= 0 if b < 0	.

This is equivalent to saying that q is the floor of (largest integer not
exceeding) a/b, and r = a - q * b.

Return r (a beta-digit) in *rptr; return q in the pre-allocated integer
block qhdl, which must have room for at least intbig_curr_size(ahdl)
digits (this is sufficient since always abs(q) <= abs(a)).

Signal an error via bh_error_general() if b is zero.
*/
{
    inthdl_sign			asign, bsign;
    inthdl_length		len; 
    t_int 		q, r;
    inthdl_length		j;


    DEBUG_INTHDL_BETA("+inthdl_quot_rem_beta", ahdl, b);

    DENY(b == 0);

    asign = intbig_sign(ahdl);
    if (asign == 0)
    {
	/*  dividend is zero  */

	intbig_sign(qhdl) = 0;
	*rptr = 0;

	DEBUG_INTHDL_BETA("-inthdl_quot_rem_beta", qhdl, *rptr);
	return;
    }

    if (b > 0)
	bsign = 1;
    else
    {
	bsign = -1;
	b = -b;
    }

    len = intbig_curr_size(ahdl);
    if (len == 1)
    {
	/*  ahdl is a single-precision integer  */

	t_int	a = intbig_digit(ahdl, 0);

	q = asign * bsign * (a / b);
	r = asign * a - q * bsign * b;

	if (r < 0 && bsign > 0)
	{
	    r += b;
	    q--;
	}
	else if (r > 0 && bsign < 0)
	{
	    r -= b;			/* i.e. r += (the real b) */
	    q--;
	}

	if (q > 0)
	{
	    intbig_sign(qhdl) = 1;
	    intbig_curr_size(qhdl) = 1;
	    intbig_digit(qhdl, 0) = q;
	}
	else if (q < 0)
	{
	    intbig_sign(qhdl) = -1;
	    intbig_curr_size(qhdl) = 1;
	    intbig_digit(qhdl, 0) = -q;
	}
	else
	    intbig_sign(qhdl) = 0;

	*rptr = r;

	DEBUG_INTHDL_BETA("-inthdl_quot_rem_beta", qhdl, *rptr);
	return;
    }

    /*  ahdl is multi-precision  */

    r = 0;
    for (j = len - 1; j >= 0; j--)
    {
	ib_quot_rem(r, intbig_digit(ahdl, j), b, &q, &r);
	intbig_digit(qhdl, j) = q;
    }

    /* NB: It's easily shown that intbig_digit(qhdl, len - 2) != 0 */
    if (intbig_digit(qhdl, len - 1) == 0)
	len--;

    intbig_sign(qhdl) = asign * bsign;
    intbig_curr_size(qhdl) = len;

    if (asign < 0)
	r = -r;

    if (r < 0 && bsign > 0)
    {
	/*
	Adjust r so that it has the same sign as b and compensate by
	subtracting one from the quotient.
	*/

	r += b;

	inthdl_add_beta(qhdl, -1, qhdl);
    }
    else if (r > 0 && bsign < 0)
    {
	/*
	Similarly.
	*/

	r -= b;					/* i.e. r += (the real b) */
	inthdl_add_beta(qhdl, -1, qhdl);
    }

    *rptr = r;

    DEBUG_INTHDL_BETA("-inthdl_quot_rem_beta", qhdl, *rptr);
}


void
inthdl_quot_rem     WITH_4_ARGS(
    inthdl_handle,  ahdl,
    inthdl_handle,  bhdl,
    inthdl_handle,  qhdl,
    inthdl_handle,  rhdl
)
/*
Given big integers a and b in blocks ahdl and bhdl, calculate the quotient q
and remainder r as the unique integers such that

    a = q * b + r,  with    0 <= r <  b if b > 0
		    and     b <  r <= 0 if b < 0	.

This is equivalent to saying that q is the floor of (largest integer not
exceeding) a/b, and r = a - q * b.

Return q in the pre-allocated integer block qhdl, which must have room for
at least

	intbig_curr_size(ahdl) - intbig_curr_size(bhdl) + 2

digits (this upper bound can be attained: for example, if a = BETA**2 - 1
with length 2 and b = -BETA with length 2 then q = -BETA has length 2).

Return r in the pre-allocated integer block qhdl, which must have room for
at least intbig_curr_size(bhdl) digits.

Signal an error via bh_error_general if b is zero.
*/
{
    t_int	r;
    inthdl_sign		asign, bsign;
    register t_int  a0;
    register t_int  b0;
    register t_int  b1;
    register t_int  a1;
    register t_int  a2;
    t_int       c0;
    t_int       c1;
    t_int       c2;
    t_int       c1dash;
    t_int       ctemp;
    t_int       prod;

    register t_int  diff;
    register t_int  signa;
    register t_int  signb;
    register t_int  lastb;
    register t_int  norm;

    inthdl_handle   adashhdl;
    inthdl_handle   bdashhdl;
    inthdl_handle   tmphdl;

    register t_int  least;
    register t_int  most;
    t_int           qdig;
    register t_int  carry;
    register t_int  nexta;
    register t_int  aindex;
    register t_int  bindex;
    register t_int  nextb;

    register inthdl_length  alen;
    register inthdl_length  blen;
    register inthdl_length  qlen;
    register inthdl_length  rlen;
    register inthdl_length  adlen;

    DEBUG_INTHDL_2("+inthdl_quot_rem", ahdl , bhdl);

    asign = intbig_sign(ahdl);
    bsign = intbig_sign(bhdl);

    DENY(bsign == 0);

    if (asign == 0)
    {
	intbig_sign(qhdl) = 0;
	intbig_sign(rhdl) = 0;
	DEBUG_INTHDL_2("-inthdl_quotrem", qhdl, rhdl);
	return;
    }

    alen = intbig_curr_size(ahdl);
    blen = intbig_curr_size(bhdl);

    if (blen == 1)
    {
	inthdl_quot_rem_beta(ahdl, intbig_digit(bhdl, 0), qhdl, &r);

	if (r > 0)
	{
	    intbig_sign(rhdl) = 1;
	    intbig_curr_size(rhdl) = 1;
	    intbig_digit(rhdl, 0) = r;
	}
	else if (r < 0)
	{
	    intbig_sign(rhdl) = -1;
	    intbig_curr_size(rhdl) = 1;
	    intbig_digit(rhdl, 0) = -r;
	}
	else
	    intbig_sign(rhdl) = 0;

	DEBUG_INTHDL_2("-inthdl_quotrem", qhdl, rhdl);
	return;
    }

    if (alen < blen)
    {
	if (asign != bsign)
	{
	    /*
	    Set q = -1 and r = a + b.
	    */

	    intbig_sign(qhdl) = -1;
	    intbig_digit(qhdl, 0) = 1;
	    intbig_curr_size(qhdl) = 1;

	    inthdl_add(ahdl, bhdl, rhdl);
	}
	else
	{
	    intbig_sign(qhdl) = 0;
	    intbig_sign(rhdl) = bsign;
	    intbig_copy_digits(ahdl, 0, alen, rhdl, 0);
	    intbig_curr_size(rhdl) = alen;
	}

	DEBUG_INTHDL_2("-inthdl_quotrem", qhdl, rhdl);
	return;
    }

    /*
    compute signs and normalize a giving adash, b giving bdash
    */

    lastb = intbig_digit(bhdl, blen - 1);

    norm = BETA / (lastb + 1);

    adashhdl = inthdl_buf_alloc(alen + 1);
    inthdl_mult_beta(ahdl, asign * norm, adashhdl);

    adlen = intbig_curr_size(adashhdl);
    if (adlen == alen)
    {
	adlen++;
	intbig_digit(adashhdl, adlen - 1) = 0;
    }

    bdashhdl = inthdl_buf_alloc(blen);
    inthdl_mult_beta(bhdl, bsign * norm, bdashhdl);

    /*
    The following is the main loop of the function.  The pen-and-paper
    long-division method is used.  Each time through the loop, we are dividing
    blen+1 digits of adash by the blen digits of bdash.  adash is successively
    transformed so that on all except the first time through the loop, the
    blen+1 digits of adash being divided are the remainder from the previous
    division plus the next most-significant digit of adash.  least is the
    index of the least significant digit, and most is the index of the most
    significant digit from the section of adash currently being worked with.
    least is also the index of the next digit to be added to the result qhdl.
    */

    tmphdl = inthdl_buf_alloc(blen);

    /*
    obtain leading digits of divisor bdash
    */

    b0 = intbig_digit(bdashhdl, blen - 2);
    b1 = intbig_digit(bdashhdl, blen - 1);

    qlen = adlen - blen;
    least = qlen - 1;
    most = adlen - 1;

    /*
     Start of repeat loop to obtain each least'th quotient digit
    */

    do
    {
	a0 = intbig_digit(adashhdl, most - 2);
	a1 = intbig_digit(adashhdl, most - 1);
	a2 = intbig_digit(adashhdl, most);

	/*
	compute quotient digit approximation qdig
	*/

	if (a2 == b1)
	    qdig = BETA - 1;
	else
	    ib_quot_rem(a2, a1, b1, &qdig, &r);

	/*
	start of repeat loop to make sure approx is no more than one too large
	get qdig to be <= (a2, a1, a0)/(b1, b0)
	*/

	do
	{
	    ib_mult(b0, qdig, &c1, &c0);
	    ib_mult(b1, qdig, &c2, &c1dash);
	    c1 += c1dash;
	    if (c1 >= BETA)
	    {
		c1 -= BETA;
		c2++;
	    }
	    diff = a2 - c2;
	    if (diff == 0)
	    {
		diff = a1 - c1;
		if (diff == 0)
		    diff = a0 - c0;
	    }
	    qdig--;
	}
	while (diff < 0);

	/*
	end of repeat loop to obtain quotient digit approximation
	*/

	qdig++;

	do
	{
	    /*
	    compute tmphdl by subtracting qdig * bdash from adash [least..most]
	    (use temporary storage tmphdl since may not have correct qdig)
	    */

	    for (bindex = 0, aindex = least, carry = 0;
		 bindex < blen;
		 bindex++, aindex++)
	    {
		nextb = intbig_digit(bdashhdl, bindex);
		ib_mult(nextb, qdig, &ctemp, &prod);
		nexta = intbig_digit(adashhdl, aindex) - prod - carry;
		carry = ctemp;

		/*
		make sure each digit of tmphdl > 0
		*/

		while (nexta < 0)
		{
		    nexta += BETA;
		    carry++;
		}

		intbig_digit(tmphdl, bindex) = nexta;
	    }

	    nexta = intbig_digit(adashhdl, aindex) - carry;

	    /*
	    change qdig and recompute tmphdl if necessary
	    */

	    qdig--;
	}
	while (nexta < 0);

	qdig++;

	/*
	now qdig is correct quotient digit and tmphdl has been computed
	*/

	intbig_digit(qhdl, least) = qdig;

	intbig_copy_digits(tmphdl, 0, blen, adashhdl, least);
	intbig_curr_size(adashhdl) = blen + least + 1;

	/*
	continue quotient digit loop with new section of adash, if more left
	*/

	least--;
	most--;
    }
    while (least >= 0);

    /*
    end of loop to compute quotient digits
    */

    if (qlen >= 1 && intbig_digit(qhdl, qlen-1) == 0)
	qlen--;

    /*
    obtain remainder from adash [0..most]
    */

    rlen = most;

    /*
    set rlen to position of most significant non-zero
    */

    while (rlen >= 0 && intbig_digit(adashhdl, rlen) == 0)
	rlen--;

    /*
    adash [0..rlen] is the remainder.
    Copy digits into tmphdl first, so that rhdl can take the result from
    inthdl_quot_rem_beta directly.  Change from index of most sig digit
    to length.
    */

    if (++rlen)
    {
	intbig_sign(tmphdl) = 1;
	intbig_copy_digits(adashhdl, 0, rlen, tmphdl, 0);
	intbig_curr_size(tmphdl) = rlen;
    }
    else
	intbig_sign(tmphdl) = 0;

    /*
    Unnormalize remainder and make its sign the same as a's
    and give q the correct sign for a/b instead of abs(a)/abs(b).
    */

    inthdl_quot_rem_beta(tmphdl, asign * norm, rhdl, &r);

    if (qlen)
    {
	intbig_sign(qhdl) = asign * bsign;
	intbig_curr_size(qhdl) = qlen;
    }
    else
	intbig_sign(qhdl) = 0;


    /*
    if necessary, modify qhdl and rhdl so that rhdl has the
    same sign as b
    */

    if (intbig_sign(rhdl) && bsign != intbig_sign(rhdl))
    {
	DEBUG_INTHDL_0(
	    "inthdl_quot_rem: adjusting result for negative remainder");

	inthdl_add_beta(qhdl, -1, qhdl);
	inthdl_add(rhdl, bhdl, rhdl);
    }
 
    inthdl_buf_delete(tmphdl);
    inthdl_buf_delete(adashhdl);
    inthdl_buf_delete(bdashhdl);

    DEBUG_INTHDL_2("-inthdl_quotrem", qhdl, rhdl);
}

void
inthdl_rem     WITH_3_ARGS(
    inthdl_handle,  ahdl,
    inthdl_handle,  bhdl,
    inthdl_handle,  rhdl
)
/*
Stores in rhdl the remainder upon dividing ahdl by bhdl.
*/
{
    inthdl_handle	quot;

    quot = inthdl_buf_alloc(intbig_sign(ahdl)? intbig_curr_size(ahdl): 1);
    inthdl_quot_rem(ahdl, bhdl, quot, rhdl);

    inthdl_buf_delete(quot);
}
