#include "defs.h"
#include "integer.e"
#include "inthdl.e"
#include "intbig.h"

void
inthdl_sqrt WITH_3_ARGS(
    inthdl_handle,     ahdl,
    inthdl_handle,     roothdl,
    t_int *,   sign
)
/*
**  Input:  ahdl - t_handle to block containing integer to be square rooted
**  Output: roothdl - t_handle to block to contain the square root
**                  - must have been preallocated by caller to at least
**                          CEIL(alen / 2)
**          sign    - sign of (integer ahdl - square of integer roothdl)
**  sqrt of ahdl is   FLOOR(\/ ahdl)
*/
{
    inthdl_sign		asign;
    inthdl_length	alen;
    t_int	tdig;
    t_int	hdig;
    inthdl_handle	thdl;
    inthdl_handle	qhdl;

    DEBUG_INTHDL_1("+inthdl_sqrt", ahdl);

    asign = intbig_sign(ahdl);
    if (asign == 0)
    {
	intbig_sign(roothdl) = 0;
	*sign = 0;
	DEBUG_INTHDL_BETA("-inthdl_sqrt", roothdl, *sign);
	return;
    }

    DENY(asign < 0);

    alen = intbig_curr_size(ahdl);
    if (alen == 1)
    {
	ib_sqrt(intbig_digit(ahdl, 0), &tdig, sign);

	intbig_sign(roothdl) = 1;
	intbig_digit(roothdl, 0) = tdig;
	intbig_curr_size(roothdl) = 1;

	DEBUG_INTHDL_BETA("-inthdl_sqrt", roothdl, *sign);
	return;
    }

    /*
     *      Case alen > 1
     *      Allocate storage for temporary results
     */

    thdl = inthdl_buf_alloc(alen + 1);
    qhdl = inthdl_buf_alloc(alen);

    /*
     *      Compute single-precision approximation to square-root
     */

    tdig = inthdl_log2(ahdl);
    hdig = tdig - ZETA;

    if (hdig & 1)
	hdig++;

    /*
    Set thdl = ahdl / (2^hdig).
    */

    inthdl_shift_right(ahdl, thdl, hdig / ZETA, hdig % ZETA);

    ib_sqrt(intbig_digit(thdl, 0), &tdig, sign);
    intbig_digit(thdl, 0) = tdig + 1;
    hdig >>= 1;

    inthdl_shift_left(thdl, roothdl, hdig / ZETA, hdig % ZETA);

    /*
     *      Start of loop to compute square-root using modified Newton
     *	method
     */

    for (;;)
    {
	t_int   s;
	inthdl_quot_rem(ahdl, roothdl, qhdl, thdl);
	if ((s = inthdl_compare(roothdl, qhdl)) <= 0)
		break;
	DEBUG_INTEGER_1(".value of comparison", s);
	inthdl_add(roothdl, qhdl, thdl);
	inthdl_quot_rem_beta(thdl, 2, roothdl, &tdig);
    }

    /*
     *      End of loop     (roothdl contains square-root)
     *      
     *      Compute sign
     */

    inthdl_mult(roothdl, roothdl, thdl);
    *sign = inthdl_compare(ahdl, thdl);

    inthdl_buf_delete(thdl);
    inthdl_buf_delete(qhdl);

    DEBUG_INTHDL_BETA("-inthdl_sqrt", roothdl, *sign);
}
