#include "defs.h"
#include "ring.h"
#include "mat.h"
#include "integer.e"


/*
 * This file includes
 *
 *	mat_ring_row_swap()	   - Swaps row i and j
 *	mat_ring_col_swap()	   - Swaps col i and j
 * 	mat_ring_row_add()         - Adds n times row i to row j
 * 	mat_ring_row_mult()        - Replaces row i by n times row i
 * 	mat_ring_col_add()         - Adds n times col i to col j
 * 	mat_ring_col_mult()        - Replaces col i by n times col i
 */


void
mat_ring_row_swap WITH_6_ARGS(
	t_handle,		cring,
	matrix,		a,
	t_int,		arow,
	t_int,		brow,
	integer_small,	from,
	integer_small,	to
)
/*
** swaps the two row vectors in the matrix,
** (row arow from a) <--> (row brow from a)
*/
{
	register integer_small    temp;
	register integer_small    len;
	register integer_small    i;
	register integer_small    n;
	register integer_small    aposn;
	register integer_small    bposn;
	register integer_small    *ptr_mat;
	matrix           mat = 0;

	len = mat_col( a );
	n = mat_row( a );

	if (arow < 1 || arow > n || brow < 1 || brow > n || from < 1 || to < from || to > len )
		error_internal("out of bounds row nos. in mat_ring_row_swap");

	mat_create_unpkd(cring, a, mat, n, len);
	ptr_mat = mat_elt0_ptr(mat);

	aposn = (arow - 1) * len + from;
	bposn = (brow - 1) * len + from;
	for ( i=from; i<=to; ++i, ++aposn, ++bposn )
	{
		temp = ptr_mat[ aposn ];
		ptr_mat[ aposn ] = ptr_mat[ bposn ];
		ptr_mat[ bposn ] = temp;
	}

	mat_free_unpkd(a, mat);
}


void
mat_ring_col_swap WITH_6_ARGS(
	t_handle,		cring,
	matrix,		a,
	t_int,		acol,
	t_int,		bcol,
	integer_small,	from,
	integer_small,	to
)
/*
** swaps the two column vectors in the matrix,
** (column acol from a) <--> (column bcol from a)
*/
{
	register integer_small    temp;
	register integer_small    len;
	register integer_small    i;
	register integer_small    n;
	register integer_small    aposn;
	register integer_small    bposn;
	register integer_small    *ptr_mat;
	matrix           mat = 0;

	len = mat_row( a );
	n = mat_col( a );

	if ( acol < 1 || acol > n || bcol < 1 || bcol > n || from < 1 || to < from || to > len )
		error_internal("out of bounds column nos. in mat_ring_col_swap");

	mat_create_unpkd(cring, a, mat, len, n);
	ptr_mat = mat_elt0_ptr(mat);

	aposn = acol + (from - 1) * n;
	bposn = bcol + (from - 1) * n;
	for ( i=from; i<=to; ++i, aposn += n, bposn += n )
	{
		temp = ptr_mat[ aposn ];
		ptr_mat[ aposn ] = ptr_mat[ bposn ];
		ptr_mat[ bposn ] = temp;
	}

	mat_free_unpkd(a, mat);
}


void
mat_ring_row_add WITH_7_ARGS(
	t_handle,         ring,
	matrix,         a,
	integer_small,  arow,
	integer_small,  brow,
	integer_big,    mult,
	integer_small,  from,
	integer_small,	to
)
/*
** Adds (mult * row arow) of a to row brow of a.
** Does the addition left to right from column "from".
*/
{
	register integer_small    i;
	register integer_small    len;
	register integer_small    numrows;
	matrix           mat = 0;
	t_ring_elt      temp1;
	t_ring_elt      temp2;
	t_ring_elt	(*ringadd)();
	t_ring_elt	(*ringmult)();
	Logical 	(*ringztest)();

	switch( ring_type( ring ))
	{
	case RING_Z:
		mat_z_row_add( ring, a, arow, brow, mult, from, to );
		return;

	case RING_ZM:
		mat_zm_row_add( ring, a, arow, brow, mult, from, to );
		return;

	default:
		break;
	}

	numrows = mat_row( a );
	len = mat_col( a );

	if (arow < 1 || arow > numrows || brow < 1 || brow > numrows || from < 1 || to < from || to > len )
		error_internal("out of bounds row nos. in mat_ring_row_add");

	ringadd = ring_addition( ring );
	ringmult = ring_multiplication( ring );
	ringztest = ring_zero_test( ring );

	mat_create_unpkd(ring, a, mat, mat_row(a), mat_col(a));

	for ( i=from; i<=to; ++i )
	{
		temp1 = mat_elt( mat, arow, i );
		if ( (*ringztest) ( ring, temp1 ))
			continue;
		temp1 = (*ringmult)( ring, temp1, mult );
		temp2 = mat_elt( mat, brow, i );
		mat_elt( mat, brow, i ) = (*ringadd)( ring, temp1, temp2 );

		ring_elt_delete( ring, &temp1 );
		ring_elt_delete( ring, &temp2 );
	}

	mat_free_unpkd(a, mat);
}





void
mat_ring_row_mult WITH_6_ARGS(
	t_handle,         ring,
	matrix,         a,
	integer_small,  arow,
	t_ring_elt,     umult,
	integer_small,  from,
	integer_small,	to
)
/*
** replaces row arow of a with (umult * row arow) of a, where
** umult is a unit of the ring.
** Only does the replacing left to right from column "from".
*/
{
	register integer_small    i;
	register integer_small    len;
	register integer_small    numrows;
	matrix           mat = 0;
	t_ring_elt      temp1;
	t_ring_elt	(*ringmult)();

	switch( ring_type( ring ))
	{
	case RING_Z:
		mat_z_row_mult( ring, a, arow, umult, from, to );
		return;

	case RING_ZM:
		mat_zm_row_mult( ring, a, arow, umult, from, to );
		return;

	default:
		break;
	}

	ringmult = ring_multiplication( ring );

	numrows = mat_row( a );
	len = mat_col( a );
	if ( arow < 1 || arow > numrows || from < 1 || to < from || to > len )
		error_internal("out of bounds row nos. in mat_ring_row_mult");

	mat_create_unpkd(ring, a, mat, mat_row(a), mat_col(a));

	for ( i=from; i<=to; ++i )
	{
		temp1 = mat_elt( mat, arow, i );	/* transfer */
		mat_elt( mat, arow, i ) = (*ringmult)( ring, temp1, umult );
		ring_elt_delete( ring, &temp1 );
	}

	mat_free_unpkd(a, mat);
}





void
mat_ring_col_add WITH_7_ARGS(
	t_handle,		ring,
	matrix,		a,
	integer_small,	acol,
	integer_small,	bcol,
	t_ring_elt,	mult,
	integer_small,	from,
	integer_small,	to
)
/*
** adds (mult * column acol) of a to column bcol of a.
*/
{
	register integer_small    i;
	register integer_small    len;
	register integer_small    numcols;
	matrix           mat = 0;
	t_ring_elt      temp1;
	t_ring_elt      temp2;
	t_ring_elt	(*ringadd)();
	t_ring_elt	(*ringmult)();
	Logical 	(*ringztest)();

	switch( ring_type( ring ))
	{
	case RING_Z:
		mat_z_col_add( ring, a, acol, bcol, mult, from, to );
		return;

	case RING_ZM:
		mat_zm_col_add( ring, a, acol, bcol, mult, from, to );
		return;

	default:
		break;
	}

	len = mat_row( a );
	numcols = mat_col( a );
	if ( acol < 1 || acol > numcols || bcol < 1 || bcol > numcols || from < 1 || to < from || to > len )
		error_internal("out of bounds column nos. in mat_ring_col_add");

	ringadd = ring_addition( ring );
	ringmult = ring_multiplication( ring );
	ringztest = ring_zero_test( ring );

	mat_create_unpkd(ring, a, mat, mat_row(a), mat_col(a));

	for ( i=from; i<=to; ++i )
	{
		temp1 = mat_elt( mat, i, acol );
		if ( (*ringztest) ( ring, temp1 ))
			continue;
		temp1 = (*ringmult)( ring, temp1, mult );
		temp2 = mat_elt( mat, i, bcol );
		mat_elt( mat, i, bcol ) = (*ringadd)( ring, temp1, temp2 );

		ring_elt_delete( ring, &temp1 );
		ring_elt_delete( ring, &temp2 );
	}

	mat_free_unpkd(a, mat);
}





void
mat_ring_col_mult WITH_6_ARGS(
	t_handle,         ring,
	matrix,         a,
	integer_small,  acol,
	integer_big,    umult,
	integer_small,	from,
	integer_small,	to
)
/*
** replaces column acol of a with (umult * column acol) of a, where
** umult is a unit of the ring.
*/
{
	register integer_small    i;
	register integer_small    len;
	register integer_small    numcols;
	matrix           mat = 0;
	t_ring_elt      temp1;
	t_ring_elt	(*ringmult)();

	switch( ring_type( ring ))
	{
	case RING_Z:
		mat_z_col_mult( ring, a, acol, umult, from, to );
		return;

	case RING_ZM:
		mat_zm_col_mult( ring, a, acol, umult, from, to );
		return;

	default:
		break;
	}

	len = mat_row( a );
	numcols = mat_col( a );
	if (acol < 1 || acol > numcols || from < 1 || to < from || to > len )
		error_internal("out of bounds column nos. in mat_ring_col_mult");

	ringmult = ring_multiplication( ring );

	mat_create_unpkd(ring, a, mat, mat_row(a), mat_col(a));

	for ( i=from; i<=to; ++i )
	{
		temp1 = mat_elt( mat, i, acol );
		mat_elt( mat, i, acol ) = (*ringmult)( ring, umult, temp1 );
		ring_elt_delete( ring, &temp1 );
	}

	mat_free_unpkd(a, mat);
}

