#include "defs.h"
#include "ring.h"
#include "mat.h"
#include "integer.e"


/*
 * This file includes
 *
 * 	mat_z_row_add()         - Adds n times row i to row j
 * 	mat_z_row_mult()        - Replaces row i by n times row i
 * 	mat_z_col_add()         - Adds n times col i to col j
 * 	mat_z_col_mult()        - Replaces col i by n times col i
 */



void
mat_z_row_add WITH_7_ARGS(
	t_handle,         ring,
	matrix,         a,
	integer_small,  arow,
	integer_small,  brow,
	integer_big,	mult,
	integer_small,  from,
	integer_small,	to
)
/*
** Adds (mult * row arow) of a to row brow of a.
** Does the addition left to right from column "from".
*/
{
	block_declarations;
	register t_ring_elt      temp1;
	register t_ring_elt      temp2;
	register integer_small    i;
	register integer_small    len;
	register integer_small    numrows;
	matrix           mat = 0;

	numrows = mat_row( a );
	len = mat_col( a );
	if (arow < 1 || arow > numrows || brow < 1 || brow > numrows || from < 1 || to < from || to > len )
		error_internal("out of bounds row nos. in mat_z_row_add");

	mat_create_unpkd(ring, a, mat, mat_row(a), mat_col(a));

	for ( i=from; i<=to; ++i )
	{
		temp1 = mat_elt( mat, arow, i );
		if (temp1 == 0)
			continue;
		temp1 = integer_mult( temp1, mult );
		temp2 = mat_elt( mat, brow, i );
		mat_elt( mat, brow, i ) = integer_add( temp1, temp2 );

		integer_delref( temp1 );
		integer_delref( temp2 );
	}

	mat_free_unpkd(a, mat);
}





void
mat_z_row_mult WITH_6_ARGS(
	t_handle,         ring,
	matrix,         a,
	integer_small,  arow,
	integer_big,    umult,
	integer_small,  from,
	integer_small,	to
)
/*
** replaces row arow of a with (umult * row arow) of a, where
** umult is a unit of the ring.
** Only does the replacing left to right from column "from".
*/
{
	block_declarations;
	register integer_big      temp1;
	register integer_small    i;
	register integer_small    len;
	register integer_small    numrows;
	matrix           mat = 0;

	numrows = mat_row( a );
	len = mat_col( a );
	if ( arow < 1 || arow > numrows || from < 1 || to < from || to > len )
		error_internal("out of bounds row nos. in mat_z_row_mult");

	mat_create_unpkd(ring, a, mat, mat_row(a), mat_col(a));

	for ( i=from; i<=to; ++i )
	{
		temp1 = mat_elt( mat, arow, i );	/* transfer */
		mat_elt( mat, arow, i ) = integer_mult( umult, temp1 );
		integer_delref( temp1 );
	}

	mat_free_unpkd(a, mat);
}





void
mat_z_col_add WITH_7_ARGS(
	t_handle,         ring,
	matrix,         a,
	integer_small,  acol,
	integer_small,  bcol,
	integer_big,    mult,
	integer_small,	from,
	integer_small,	to
)
/*
** adds (mult * column acol) of a to column bcol of a.
*/
{
	block_declarations;
	register integer_big      temp1;
	register integer_big      temp2;
	register integer_small    i;
	register integer_small    len;
	register integer_small    numcols;
	matrix           mat = 0;

	len = mat_row( a );
	numcols = mat_col( a );
	if (acol < 1 || acol > numcols || bcol < 1 || bcol > numcols || from < 1 || to < from || to > len )
		error_internal("out of bounds column nos. in mat_z_col_add");

	mat_create_unpkd(ring, a, mat, mat_row(a), mat_col(a));

	for ( i=from; i<=to; ++i )
	{
		temp1 = integer_mult( mat_elt( mat, i, acol ), mult );
		temp2 = mat_elt( mat, i, bcol );
		mat_elt( mat, i, bcol ) = integer_add( temp1, temp2 );
		integer_delref( temp1 );
		integer_delref( temp2 );
	}

	mat_free_unpkd(a, mat);
}




void
mat_z_col_mult WITH_6_ARGS(
	t_handle,         ring,
	matrix,         a,
	integer_small,  acol,
	integer_big,    umult,
	integer_small,	from,
	integer_small,	to
)
/*
** replaces column acol of a with (umult * column acol) of a, where
** umult is a unit of the ring.
*/
{
	block_declarations;
	register integer_big      temp1;
	register integer_small    i;
	register integer_small    len;
	register integer_small    numcols;
	matrix           mat = 0;

	len = mat_row( a );
	numcols = mat_col( a );
	if ( acol < 1 || acol > numcols || from < 1 || to < from || to > len )
		error_internal("out of bounds column nos. in mat_z_col_mult");

	mat_create_unpkd(ring, a, mat, mat_row(a), mat_col(a));
	integer_incref(umult);

	for ( i=from; i<=to; ++i )
	{
		temp1 = mat_elt( mat, i, acol );
		mat_elt( mat, i, acol ) = integer_mult( umult, temp1 );
		integer_delref( temp1 );
	}

	mat_free_unpkd(a, mat);
}
