#include "defs.h"
#include "debug.e"
#include "integer.e"
#include "poly.h"
#include "dyn_arr.h"
#include "zm.e"
#include "error.e"

#undef DEPTH_TESTING
#ifdef DEPTH_TESTING
private t_int DEpth;
#endif

t_void
modpoly_gcd_cofactor WITH_7_ARGS(
    t_handle,         pring,
    t_int,    pdig,
    t_poly,        Apoly,
    t_poly,        Bpoly,
    t_poly *,      Cpoly,
    t_poly *,      Abar,
    t_poly *,      Bbar
)
/*
** Modular integral polynomial greatest common divisor and cofactors
**
** pdig prime beta integer
** Apoly, Bpoly modular integer polynomials over Z_pdig
** *Cpoly = GCD( Apoly, Bpoly )
** if *Cpoly != 0 then *Abar = Apoly / Cpoly, *Bbar = Bpoly / Cpoly
*/
{
    block_declarations;
    t_poly         a;
    t_poly         Ahat;
    t_poly         b;
    t_poly         Bhat;
    t_poly         c;
    t_int     ahat;
    t_int     bhat;
    t_poly         chat;
    dyn_arr_handle    U;
    t_int     Ulen;
    dyn_arr_handle    V;
    t_int     Vlen;
    t_int     g;
    t_int     d;
    dyn_arr_handle    W;
    dyn_arr_handle    Wstar;
    t_int     cstar;
    t_int     princv;
    t_int     lpvar;
    t_poly         Astar;
    t_poly         Bstar;
    dyn_arr_handle    Ustar;
    dyn_arr_handle    Vstar;
    t_poly         Cstar;
    t_poly         Astarhat;
    t_poly         Bstarhat;
    t_poly         Cstarhat;
    t_poly         temp;
    t_poly         Aprime;
    t_poly         Bprime;
    t_poly         Cprime;
    t_poly         Q;
    t_handle            linph;
    t_poly         linpoly;
    t_poly         aprime;
    t_poly         bprime;
    t_poly         cprime;
    t_int     i;
    t_int     t;
    t_int     qprime;
    t_int     cprimeprime;
    t_poly         cprimehat;
    t_poly         abar;
    t_poly         bbar;
    t_handle            Aph;
    t_handle            Bph;
    t_handle            temp_hdl;

#ifdef DEPTH_TESTING
DEpth ++;
cay_print ("Depth = %d\n", DEpth);
#endif

    if ( ! integer_is_single( pdig ))
    {
	error_internal( "cannot take GCD over large modulus" );
        return;
    }

    IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
		cay_print( "Entering modpoly_gcd()\nAp = " );
		poly_z_write(pring, Apoly);
		cay_print( "\nBp = " );
		poly_z_write(pring, Bpoly);
		cay_print( "\n" );
	)

/* STEP 1 : A = B = 0 */
	IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
		cay_print( "step 1" );
		dmplin();
	)

    if ( poly_z_is_zero_poly (pring, Apoly)
      && poly_z_is_zero_poly (pring, Bpoly))
    {
	*Cpoly = m_modpoly_incref (pring, Apoly);
	*Abar = m_modpoly_incref (pring, Apoly);
	*Bbar = m_modpoly_incref (pring, Apoly);

    IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
	    cay_print( "modpoly_gcd( 0, 0 ) = 0\n" );
	)

	goto EXIT1;
    }

/* STEP 2 A = 0 or B = 0 */
	IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
		cay_print( "step 2" );
		dmplin();
	)

    if ( poly_z_is_zero_poly (pring, Apoly))
    {
	*Cpoly = modpoly_monic( pring, pdig, Bpoly );

	*Abar = m_modpoly_incref (pring, Apoly);
	*Bbar = poly_z_constant_poly (pring, Bpoly,
                                   poly_z_lbase_coefft( pring, Bpoly ));

    IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
	    cay_print( "modpoly_gcd( B, 0 ) = B\n" );
	)

	goto EXIT1;
    }

    if ( poly_z_is_zero_poly (pring, Bpoly) )
    {
	*Cpoly = modpoly_monic( pring, pdig, Apoly );

	*Abar = poly_z_constant_poly (pring, Apoly,
                                   poly_z_lbase_coefft( pring, Apoly ));
	*Bbar = m_modpoly_incref (pring, Bpoly);

    IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
	    cay_print( "modpoly_gcd( A, 0 ) = A\n" );
	)

	goto EXIT1;
    }

/* Step 2a : constant polynomial ? */

    if ( m_poly_const (Apoly) || m_poly_const (Bpoly) )
    {
#ifdef DEBUG
        if ( m_poly_not_const (Apoly) || m_poly_not_const (Bpoly) )
        {
            error_internal ("modpoly_gcd: polys not lifted\n");
            return;
        }
#endif /* DEBUG */
	*Cpoly = 1;
	*Abar = Apoly;
	*Bbar =  Bpoly;

    IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
	    cay_print( "modpoly_gcd( const, const ) = 1\n" );
	)

	goto EXIT1;
    }

    Aph = m_poly_poly_to_handle( Apoly );
    Bph = m_poly_poly_to_handle( Bpoly );

    princv = m_poly_princvar (Aph);
    lpvar  = m_poly_least_pvar (Aph);
#ifdef DEBUG
    if ( princv != m_poly_princvar (Bph) )
    {
	IF_DEBUG_FLAG
	(
		DEBUG_MODPOLY_GCD,
	    cay_print( "princvar( Aph ) != princvar( Bph )\n" );
	)
        error_internal ("modpoly_gcd: polys not lifted\n");
        return;
    }
    if ( lpvar != m_poly_least_pvar (Bph) )
    {
    	IF_DEBUG_FLAG
		( 
			DEBUG_MODPOLY_GCD,
	    	cay_print( "least_pvar( Aph ) != least_pvar( Bph )\n" );
		)
        error_internal ("modpoly_gcd: polys not lifted\n");
        return;
    }
#endif /* DEBUG */


/* STEP 3 : univariate polynomials ? */
	IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
		cay_print( "step 3" );
		dmplin();
	)

    if (m_poly_univariate (Aph))
    {
	*Cpoly = poly_u_zm_gcd( pring, pdig, Apoly, Bpoly );

	*Abar = modpoly_div( pring, pdig, Apoly, *Cpoly );
	*Bbar = modpoly_div( pring, pdig, Bpoly, *Cpoly );

	goto EXIT1;
    }

/* STEP 4 :  Compute univariate contents and primitive parts */
	IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
		cay_print( "step 4" );
		dmplin();
	)

    modpoly_uni_contpp( pring, pdig, Apoly, &a, &Ahat );

	IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
		cay_print( "a = " );
		poly_z_write( pring, a );
		cay_print( "\nAhat = " );
		poly_z_write( pring, Ahat );
		cay_print( "\n" );
	)

    modpoly_uni_contpp( pring, pdig, Bpoly, &b, &Bhat );

	IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
		cay_print( "b = " );
		poly_z_write( pring, b );
		cay_print( "\nBhat = " );
		poly_z_write( pring, Bhat );
		cay_print( "\n" );
	)

    c = poly_u_zm_gcd( pring, pdig, a, b );

	IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
		cay_print( "c = " );
		poly_z_write( pring, c );
		cay_print( "\n" );
	)

/* STEP 5 : Compute normalisation factor */
	IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
		cay_print( "step 5" );
		dmplin();
	)

    ahat = poly_z_lead_coefft_svar( pring, Ahat, lpvar-1 );
    bhat = poly_z_lead_coefft_svar( pring, Bhat, lpvar-1 );

    chat = poly_u_zm_gcd( pring, pdig, ahat, bhat );

	IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
		cay_print( "ahat = " );
		poly_z_write( pring, ahat );
		cay_print( "\nbhat = " );
		poly_z_write( pring, bhat );
		cay_print( "\nchat = " );
		poly_z_write( pring, chat );
		cay_print( "\n" );
	)

/* STEP 6 : Compute degrees */
	IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
		cay_print( "step 6" );
		dmplin();
	)

    U = poly_degree_n_vector(Ahat, lpvar );
    Ulen = poly_deg_sp_var (Ahat, lpvar);
    dyn_arr_curr_length( U ) = lpvar;

	IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
		cay_print( "U = " );
		dyn_int_arr_print(U);
		cay_print( "\n" );
	)

    V = poly_degree_n_vector(Bhat, lpvar );
    Vlen = poly_deg_sp_var (Bhat, lpvar);
    dyn_arr_curr_length( V ) = lpvar;

	IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
		cay_print( "V = " );
		dyn_int_arr_print(V);
		cay_print( "\n" );
	)

    g = poly_deg( chat ) + integer_max( Ulen, Vlen );

    if (g >= pdig)
    {
	cay_print( "\nmultivariate GCD failed.\nelements of Z_p exhausted in modpoly_gcd_cofactor\n" );
	*Cpoly = *Abar = *Bbar = 0;
	goto EXIT2;
    }

/* STEP 7 :  Initialize element and degree vector */
	IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
		cay_print( "step 7" );
		dmplin();
	)

    d = -1;

    W = dyn_arr_alloc( lpvar );
    dyn_arr_curr_length(W) = lpvar;

    for (i = 0; i < lpvar; ++i)
	dyn_arr_element( W, i ) = dyn_arr_element( U, i );

    /* should this 0 be princv ? */
    dyn_arr_element( W, princv ) = dyn_arr_element( W, princv ) + 1;

	IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
		cay_print( "W = " );
		dyn_int_arr_print(W);
		cay_print( "\n" );
	)

    m_poly_create_empty (&temp_hdl, lpvar, lpvar, 1);
    m_poly_coefft (temp_hdl, 0) = 1;
    m_poly_expt (temp_hdl, 0) = 0;
    Q = m_poly_handle_to_poly (temp_hdl);
    Aprime = poly_z_zero_poly (pring, Apoly);
    Bprime = m_modpoly_incref (pring, Aprime);
    Cprime = m_modpoly_incref (pring, Aprime);

    for ( ;; )
    {

/* STEP 8 : Obtain next element */
	IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
		cay_print( "step 8" );
		dmplin();
	)

	++d;

	IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
	    cay_print( "increase d to %d\n", d );
	)

	if ( d == pdig )
	{
	    cay_print( "\nmultivariate GCD failed.\nelements of Z_p exhausted in modpoly_gcd_cofactor\n" );
	    *Cpoly = *Abar = *Bbar = 0;
	    goto EXIT3;
	}

/* STEP 9 : Map normalization factor */
	IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
		cay_print( "step 9" );
		dmplin();
	)

	cstar = modpoly_eval_princvar( pring, pdig, chat, d );

	IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
	    cay_print( "cstar = " );
	    poly_z_write( pring, cstar );
	    cay_print( "\n" );
	)

        /* since chat is univariate, cstar is const. */
        /* no need to call poly_z_is_zero_poly        */
	if ( cstar == 0 )
	{
		IF_DEBUG_FLAG
		( 
			DEBUG_MODPOLY_GCD,
			cay_print( "continue 9\n" );
		)

	    continue;	/* goto Step 8 */
	}

/* STEP 10 : Map Ahat and Bhat */
	IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
		cay_print( "step 10" );
		dmplin();
	)

	Astar = modpoly_eval_lpvar( pring, pdig, Ahat, d );

	Ustar = poly_degree_n_vector(Astar, lpvar );
	t = dyn_arr_vcomp( U, Ustar );
	mem_delete_hptr( &Ustar );

	if ( t )
	{
	    m_modpoly_delref( pring, Astar );
		IF_DEBUG_FLAG
		( 
			DEBUG_MODPOLY_GCD,
			cay_print( "continue 10a\n" );
	    )

	    continue;		/* goto step 8 */
	}

	Bstar = modpoly_eval_lpvar( pring, pdig, Bhat, d );

	Vstar = poly_degree_n_vector(Bstar, lpvar );
	t = dyn_arr_vcomp( V, Vstar );
	mem_delete_hptr( &Vstar );

	if ( t )
	{
	    m_modpoly_delref( pring, Astar );
	    m_modpoly_delref( pring, Bstar );
		IF_DEBUG_FLAG
		( 
			DEBUG_MODPOLY_GCD,
			cay_print( "continue 10b\n" );
	    )

	    continue;		/* goto step 8 */
	}

/* STEP 11 : Compute G.C.D. */
	IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
		cay_print( "step 11" );
		dmplin();
	)

	modpoly_gcd_cofactor( pring, pdig, Astar, Bstar, &Cstar, &Astarhat, &Bstarhat );

	m_modpoly_delref( pring, Astar );
	m_modpoly_delref( pring, Bstar );

/* STEP 12 : Test for constant G.C.D. */
	IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
		cay_print( "step 12" );
		dmplin();
	)

	if ( poly_z_is_one_poly (pring, Cstar))
	{
            /* lift c into Apoly world */
	    poly_z_lift (pring, c, Apoly, Cpoly, Abar);
	    m_poly_z_delref (pring, *Abar);

	    *Abar = modpoly_uni_quot( pring, pdig, Apoly, c );
	    *Bbar = modpoly_uni_quot( pring, pdig, Bpoly, c );

	    m_modpoly_delref( pring, Cstar );
	    m_modpoly_delref( pring, Astarhat );
	    m_modpoly_delref( pring, Bstarhat );

	    goto EXIT3;
	}

/* STEP 13 : Conditional initialization of the interpolation process */
	IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
		cay_print( "step 13" );
		dmplin();
	)

	Wstar = poly_degree_n_vector(Cstar, lpvar );
	t = dyn_arr_vcomp( W, Wstar );

	if ( t >= 2 )
	{
            m_modpoly_delref( pring, Cprime );
            m_modpoly_delref( pring, Aprime );
            m_modpoly_delref( pring, Bprime );
            m_modpoly_delref( pring, Q );

            m_poly_create_empty (&temp_hdl, lpvar, lpvar, 1);
            m_poly_coefft (temp_hdl, 0) = 1;
            m_poly_expt (temp_hdl, 0) = 0;
            Q = m_poly_handle_to_poly (temp_hdl);
	    Aprime = poly_z_zero_poly (pring, Apoly);
	    Bprime = m_modpoly_incref (pring, Aprime);
	    Cprime = m_modpoly_incref (pring, Aprime);

	    temp = W;
	    W = dyn_arr_vmin( temp, Wstar );
	    mem_delete_h( temp );
	}

	mem_delete_h( Wstar );

/* STEP 14 : Test for unlucky element */
	IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
		cay_print( "step 14" );
		dmplin();
	)

	if ( t == 1 || t == 3 || (t == 0 && poly_deg(Q) > g ))
	{
	    m_modpoly_delref( pring, Cstar );
	    m_modpoly_delref( pring, Astarhat );
	    m_modpoly_delref( pring, Bstarhat );

		IF_DEBUG_FLAG
		( 
			DEBUG_MODPOLY_GCD,
			cay_print( "t = %d   continue 14\n", t );
	    )

	    continue;		/* goto step 8 */
	}

/* STEP 15 : Interpolate */
	IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
		cay_print( "step 15" );
		dmplin();
	)

	Cstarhat = modpoly_integer_mult( pring, pdig, Cstar, cstar );
    m_modpoly_delref (pring, Cstar);

    temp = modpoly_eval_princvar( pring, pdig, Q, d );
	qprime = modint_invert( pdig, temp );

    IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
        cay_print ("\n\n\n\nd = %d    q' = %d", d, qprime);
        cay_print ("\n\nAprime   = "); poly_z_write (pring, Aprime);
        cay_print ("\nAstarhat = ");   poly_z_write (pring, Astarhat);
        cay_print ("\n\nBprime   = "); poly_z_write (pring, Bprime);
        cay_print ("\nBstarhat = ");   poly_z_write (pring, Bstarhat);
        cay_print ("\n\nCprime   = "); poly_z_write (pring, Cprime);
        cay_print ("\nCstarhat = ");   poly_z_write (pring, Cstarhat);
	)

	temp = Cprime;
	Cprime = modpoly_interpolate( pring, pdig, Q, d, qprime, temp, Cstarhat );
	m_modpoly_delref( pring, temp );

	temp = Aprime;
	Aprime = modpoly_interpolate( pring, pdig, Q, d, qprime, temp, Astarhat );
	m_modpoly_delref( pring, temp );

	temp = Bprime;
	Bprime = modpoly_interpolate( pring, pdig, Q, d, qprime, temp, Bstarhat );
	m_modpoly_delref( pring, temp );

        m_modpoly_delref (pring, Astarhat);
        m_modpoly_delref (pring, Bstarhat);
        m_modpoly_delref (pring, Cstarhat);

    IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
        cay_print ("\n\nAprime   = "); poly_z_write (pring, Aprime);
        cay_print ("\nBprime   = "); poly_z_write (pring, Bprime);
        cay_print ("\nCprime   = "); poly_z_write (pring, Cprime);
	)

	if ( d != 0 )
	{
	    m_poly_create_empty( &linph, lpvar, lpvar, 2 );

	    m_poly_coefft( linph, 0 ) = modint_negate( pdig, d );
	    m_poly_expt( linph, 0 ) = 0;
	    m_poly_coefft( linph, 1 ) = 1;
	    m_poly_expt( linph, 1 ) = 1;
	}
	else
	{
	    m_poly_create_empty( &linph, lpvar, lpvar, 1 );

	    m_poly_coefft( linph, 0 ) = 1;
	    m_poly_expt( linph, 0 ) = 1;
	}

	linpoly = m_poly_handle_to_poly( linph );

	IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
	    cay_print( "linpoly = " );
	    poly_z_write( pring, linpoly );
	    cay_print( "\nQ = " );
	    poly_z_write( pring, Q );
	    cay_print( "\nQ *= linpoly\n" );
	)

	temp = Q;
	Q = modpoly_mult( pring, pdig, temp, linpoly );
	m_modpoly_delref( pring, temp );
	m_modpoly_delref( pring, linpoly );

	IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
	    cay_print( "Q = " );
	    poly_z_write( pring, Q );
	    cay_print( "\n" );
	)

/* STEP 16 : Test for completion */
	IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
		cay_print( "step 16" );
		dmplin();
	)

	if ( poly_deg( Q ) <= g )
	{
		IF_DEBUG_FLAG
		( 
			DEBUG_MODPOLY_GCD,
			cay_print( "continue 16a\n" );
	    )

	    continue;		/* goto Step 8 */
	}

	IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
	    cay_print( "poly_deg( Q ) > g\n" );
	)

	cprime = poly_deg_sp_var(Cprime, lpvar );
	aprime = poly_deg_sp_var(Aprime, lpvar );
	bprime = poly_deg_sp_var(Bprime, lpvar );

	cprimeprime = ( aprime > bprime ? aprime : bprime );

	if ( poly_deg( Q ) <= cprime + cprimeprime )
	{
		IF_DEBUG_FLAG
		( 
			DEBUG_MODPOLY_GCD,
			cay_print( "continue 16b\n" );
	    )

	    continue;		/* goto Step 8 */
	}

	IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
	    cay_print( "break\n" );
	)

	/* loop is only restarted at continue statements */
	break;
    }

/* STEP 17 : Remove normalization */
	IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
		cay_print( "step 17" );
		dmplin();
	)

    modpoly_uni_contpp( pring, pdig, Cprime, &cprime, Cpoly );

    cprimehat = modpoly_div( pring, pdig, chat, cprime );
    m_modpoly_delref( pring, cprime );

    *Abar = modpoly_uni_quot( pring, pdig, Aprime, cprimehat );
    *Bbar = modpoly_uni_quot( pring, pdig, Bprime, cprimehat );

    m_modpoly_delref( pring, cprimehat );

    temp = *Cpoly;
    *Cpoly = modpoly_uni_mult( pring, pdig, temp, c );
    m_modpoly_delref( pring, temp );

    abar = modpoly_div( pring, pdig, a, c );

    temp = *Abar;
    *Abar = modpoly_uni_mult( pring, pdig, temp, abar );
    m_modpoly_delref( pring, temp );

    m_modpoly_delref( pring, abar );

    bbar = modpoly_div( pring, pdig, b, c );

    temp = *Bbar;
    *Bbar = modpoly_uni_mult( pring, pdig, temp, bbar );
    m_modpoly_delref( pring, temp );

    m_modpoly_delref( pring, bbar );

EXIT3: ;

	IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
		cay_print( "EXIT3" );
		dmplin();
    )

    m_modpoly_delref( pring, Q );
    m_modpoly_delref( pring, Aprime );
    m_modpoly_delref( pring, Bprime );
    m_modpoly_delref( pring, Cprime );
EXIT2: ;

	IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
		cay_print( "EXIT2" );
		dmplin();
    )

    m_modpoly_delref( pring, a );
    m_modpoly_delref( pring, b );
    m_modpoly_delref( pring, c );
    m_modpoly_delref( pring, ahat );
    m_modpoly_delref( pring, bhat );
    m_modpoly_delref( pring, chat );
    m_modpoly_delref( pring, Ahat );
    m_modpoly_delref( pring, Bhat );

EXIT1: ;

	IF_DEBUG_FLAG
	( 
		DEBUG_MODPOLY_GCD,
		cay_print( "EXIT1" );
		dmplin();
		cay_print( "Exiting modpoly_gcd()\nCp = " );
		poly_z_write( pring, *Cpoly );
		cay_print( "\n" );
    )
#ifdef DEPTH_TESTING
DEpth --;
#endif
}

