#include "defs.h"
#include "mp.e"
#include "mp.h"


mp_float
mp_abs		WITH_2_ARGS(
	mp_float,	x,
	mp_float,	y
)
/*
Sets y to the absolute value of x and returns y.
*/
{
    mp_ptr_type		xp = mp_ptr(x), yp = mp_ptr(y);

    DEBUG_BEGIN(DEBUG_OTHER);
    DEBUG_PRINTF_1("+abs {\n");
    DEBUG_1("x = ", xp);

    mp_check_2("mp_abs", xp, yp);

    mp_copy_ptr(xp, yp);

    if (mp_is_neg(yp))
	mp_set_sign(yp, 1);
    
    DEBUG_1("-} y = ", yp);
    DEBUG_END();
    return y;
}


mp_float
mp_accuracy	WITH_1_ARG(
	mp_float,	x
)
/*
Returns x which is set to the (multiple-precision) machine precision,
that is

        x = 1.01 * (b^(1 - t))   (rounded up) if round = MP_TRUNC,
             0.5 * (b^(1 - t))   (rounded up) if round = MP_RND,
                    b^(1 - t)                 if round = MP_RND_UP or
							 MP_RND_DOWN,

where round is the current rounding type.
x is an upper bound on the smallest positive representable
number such that the relative error in the basic mp operations
(addition, subtraction, multiplication and division) is at
most x (unless the result underflows).
*/
{
    mp_ptr_type		xp = mp_ptr(x);

    mp_int_to_mp(1, x);
    mp_expt(xp) = 2 - mp_t(xp);

    mp_update(xp);

    if (round == MP_RND)
	mp_div_int_eq(x, 2);
    
    else if (round == MP_TRUNC)
    {
	round = MP_RND_UP;
	mp_mul_q_eq(x, 101, 100);
	round = MP_TRUNC;
    }

    return x;
}



mp_float
mp_add_int		WITH_3_ARGS(
	mp_float,	x,
	mp_int,	iy,
	mp_float,	z
)
/*
Adds multiple-precision x to integer iy giving multi-precision z
and returns z.  Accumulator operations are performed.
*/
{
    mp_ptr_type		xp = mp_ptr(x), zp = mp_ptr(z);
    mp_acc_float	int_temp;

    DEBUG_BEGIN(DEBUG_ADD);
    DEBUG_PRINTF_1("+add_int {\n");
    DEBUG_1("x = ", xp);
    DEBUG_PRINTF_2("iy = %d\n\n", iy);


    mp_check_2("mp_add_int", xp, zp);

    mp_acc_float_alloc(mp_b(xp), mp_t(xp), int_temp);
    mp_int_to_mp(iy, int_temp);

    mp_add(x, int_temp, z);

    mp_acc_float_delete(int_temp);

    DEBUG_1("-} z = ", mp_ptr(z));
    DEBUG_END();
    return z;
}


mp_float
mp_ceil		WITH_2_ARGS(
	mp_float,	x,
	mp_float,	y
)
/*
Returns y = ceiling(x), i.e. the smallest integer not less than x.
*/
{
    mp_ptr_type		xp = mp_ptr(x), yp = mp_ptr(y);
    mp_acc_float	temp;


    mp_check_2("mp_ceil", xp, yp);

    mp_acc_float_alloc(mp_b(xp), mp_t(xp), temp);
    mp_to_int_mp(x, temp);


    /*
    If x is positive and not an integer, we need to add 1.
    */

    if (mp_is_pos(xp) && mp_cmp(x, temp))
	mp_add_int_eq(temp, 1);

    mp_copy(temp, y);

    mp_acc_float_delete(temp);

    return y;
}



void
mp_decimal	WITH_3_ARGS(
	mp_int,		dec,
	mp_base_type *,	b,
	mp_length *,	t
)
/*
Sets base *b and number of digits *t to appropriate values for use with
mp numbers with dec decimal places.
*/
{
    mp_base_type	bb;
    mp_length		tt;

    DEBUG_BEGIN(DEBUG_OTHER);
    DEBUG_PRINTF_1("+decimal {\n");
    DEBUG_PRINTF_2("dec = %d\n", dec);

    if (dec <= 0)
	mp_error("illegal value of decimal places (%d)", dec);

    bb = 2;

    while (4 * bb * bb < MAX_EXPT + 1)
	bb <<= 1;


    tt = mp_change_base(bb, 10, dec) + 1;

    if (tt < 2)
	tt = 2;

    DEBUG_PRINTF_3("-} *b = %d, *t = %d\n", bb, tt);
    DEBUG_END();

    *b = bb;
    *t = tt;
}

mp_float
mp_epsilon	WITH_1_ARG(
	mp_float,	x
)
/*
Returns x which is set to epsilon: the smallest number such that
the sum of it and 1 is greater than 1.  This depends on the current
rounding type (as well as the base and number of digits of x).
*/
{
    mp_ptr_type		xp = mp_ptr(x);
    mp_length		t = mp_t(xp);

    switch (round)
    {
	case MP_TRUNC:
	case MP_RND_DOWN:
	    mp_int_to_mp(1, x);
	    mp_expt(xp) = -t;

	    break;

	case MP_RND:
	{
	    mp_base_type	b = mp_b(xp);

	    mp_q_to_mp(b, 2, x);
	    mp_expt(xp) = -t;

	    break;
	}

	case MP_RND_UP:
	    mp_tiny(x);
    }

    return x;
}


mp_sign_type
mp_float_sign WITH_1_ARG(
	mp_float,	x
)
/*
Returns the sign of the float x.
*/
{
    return mp_sign(mp_ptr(x));
}



mp_float
mp_floor	WITH_2_ARGS(
	mp_float,	x,
	mp_float,	y
)
/*
Returns y = floor(x), i.e. the largest integer not greater than x.
*/
{
    mp_ptr_type		xp = mp_ptr(x), yp = mp_ptr(y);
    mp_acc_float	temp;


    mp_check_2("mp_floor", xp, yp);

    mp_acc_float_alloc(mp_b(xp), mp_t(xp), temp);
    mp_to_int_mp(x, temp);

    /*
    If x is negative and not an integer, we need to subtract 1.
    */

    if (mp_is_neg(xp) && mp_cmp(x, temp))
	mp_add_int_eq(temp, -1);

    mp_copy(temp, y);

    mp_acc_float_delete(temp);

    return y;
}

mp_float
mp_huge		WITH_1_ARG(
	mp_float,	x
)
/*
Returns x = largest possible mp number representable in x.
*/
{
    mp_ptr_type		xp = mp_ptr(x);
    mp_int		i;
    mp_digit_type	fill = mp_b(xp) - 1;

    for (i = mp_t(xp) - 1; i >= 0; i--)
	mp_set_digit(xp, i, fill);

    mp_expt(xp) = MAX_EXPT;

    return x;
}


mp_bool
mp_is_int	WITH_1_ARG(
	mp_float,	x
)
/*
Returns whether the mp_float x is an exact integer.
*/
{
    mp_ptr_type		xp = mp_ptr(x);
    mp_acc_float	temp;
    mp_bool		result;

    mp_acc_float_alloc(mp_b(xp), mp_t(xp), temp);
    mp_to_frac(x, temp);

    result = mp_is_zero(mp_acc_float_ptr(temp));
    mp_acc_float_delete(temp);

    return result;
}

mp_float
mp_max		WITH_3_ARGS(
	mp_float,	x,
	mp_float,	y,
	mp_float,	z
)
/*
Returns z = int_max(x, y).
*/
{
    if (mp_cmp(x, y) >= 0)
	mp_copy(x, z);

    else
	mp_copy(y, z);

    return z;
}


void
mp_set_round	WITH_1_ARG(
	mp_round_type,	r
)
/*
Sets the current mp rounding type to r.  All subsequent calculations
will be performed according to this rounding type.
*/
{
    round = r;
}


mp_float
mp_tiny		WITH_1_ARG(
	mp_float,	x
)
/*
Returns x = smallest possible mp number representable in x.
*/
{
    mp_int_to_mp(1, x);
    mp_expt(mp_ptr(x)) = MIN_EXPT;

    return x;
}


void
mp_priv_update	WITH_1_ARG(
	mp_ptr_type,	zp
)
/*
Updates the maximum and minimum exponent indicators of the package with
respect to zp.  No check is made for overflow or underflow.
*/
{
    mp_expt_type	expt = mp_expt(zp);

    DEBUG_BEGIN(DEBUG_OTHER);
    DEBUG_PRINTF_1("+update {\n");
    DEBUG_PRINTF_2("expt = %d\n", expt);

    if (expt < mp_priv_min_expt)
	mp_priv_min_expt = expt;

    else if (expt > mp_priv_max_expt)
	mp_priv_max_expt = expt;
    
    DEBUG_PRINTF_1("-} update\n");
    DEBUG_END();
}
