#include "defs.h"
#include "mp.e"
#include "mp.h"


mp_float
mp_q_to_mp		WITH_3_ARGS(
	mp_int,		i,
	mp_int,		j,
	mp_float,	z
)
/*
Converts the rational number i/j to multi-precision z and returns z.
*/
{
    DEBUG_BEGIN(DEBUG_OTHER);
    DEBUG_PRINTF_1("+q_to_mp {\n");
    DEBUG_PRINTF_3("i = %d, j = %d\n\n", i, j);

    /*
    Make i and j relatively prime.
    */

    mp_int_gcd(&i, &j);


    if (!j)
	mp_error("mp_q_to_mp: denominator j == 0");

    if (j < 0)
    {
	/*
	Make denominator positive to give correct directed rounding.
	*/

	i = -i;
	j = -j;
    }

    mp_int_to_mp(i, z);

    if (j != 1)
	mp_div_int_eq(z, j);

    DEBUG_1("-} z = ", mp_ptr(z));

    DEBUG_END();
    return z;
}



mp_float
mp_add_q		WITH_4_ARGS(
	mp_float,	x,
	mp_int,		i,
	mp_int,		j,
	mp_float,	z
)
/*
Adds multiple-precision x to rational number i/j giving multi-precision z
and returns z.  The effect is the same as converting i/j to multi-precision
using mp_q_to_mp() and then adding to x using mp_add(), so see the comments
in these routines.  Accumulator operations are performed.
*/
{
    mp_ptr_type		xp = mp_ptr(x), zp = mp_ptr(z);
    mp_acc_float	q_temp;


    DEBUG_BEGIN(DEBUG_OTHER);
    DEBUG_PRINTF_1("+add_q {\n");
    DEBUG_1("x = ", xp);
    DEBUG_PRINTF_2("i = %d\n", i);
    DEBUG_PRINTF_2("j = %d\n\n", j);

    /*
    Check for compatible parameters.
    */

    mp_check_2("mp_add_q", xp, zp);


    /*
    Allocate temporary float, compute i/j and add to x.
    */

    mp_acc_float_alloc(mp_b(xp), mp_t(xp), q_temp);

    mp_q_to_mp(i, j, q_temp);
    mp_add(x, q_temp, z);

    mp_acc_float_delete(q_temp);

    DEBUG_1("-} z = ", mp_ptr(z));
    DEBUG_END();
    return z;
}



mp_float
mp_mul_q	WITH_4_ARGS(
	mp_float,	x,
	mp_int,		i,
	mp_int,		j,
	mp_float,	y
)
/*
Multiplies mp x by i / j, giving mp result y.  Accumulator operations
are performed.
*/
{
    mp_ptr_type		xp = mp_ptr(x), yp = mp_ptr(y);


    DEBUG_BEGIN(DEBUG_OTHER);
    DEBUG_PRINTF_1("+mul_q {\n");
    DEBUG_1("x = ", xp);
    DEBUG_PRINTF_3("i = %d, j = %d\n\n", i, j);


    if (!j)
	mp_error("mp_mul_q: division by zero");

    mp_check_2("mp_mul_q", xp, yp);

    if (!i)
    {
	mp_set_sign(yp, 0);

	DEBUG_PRINTF_1("-} y = 0");
	DEBUG_END();
	return y;
    }

    /*
    Reduce to lowest terms.
    */

    mp_int_gcd(&i, &j);

    if (int_abs(i) == 1)

	/*
	Here i = +-1.
	*/

	mp_div_int(x, i * j, y);

    else
    {
	/*
	Check rounding options.
	*/

	if (round == MP_TRUNC)
	{
	    /*
	    Use simple methods.
	    */

	    mp_mul_int(x, i, y);
	    mp_div_int_eq(y, j);
	}

	else
	{
	    /*
	    Use enough guard digits so that multiplication is exact.
	    */

	    mp_base_type	b = mp_b(xp);
	    mp_acc_float	temp;

	    mp_acc_float_alloc(b, mp_t(xp) + mp_guard_digits(i, b), temp);

	    mp_move(x, temp);

	    mp_mul_int_eq(temp, i);

	    if (j != 1)
		mp_div_int_eq(temp, j);

	    mp_move(temp, y);

	    mp_acc_float_delete(temp);
	}
    }

    DEBUG_1("-} y = ", mp_ptr(y));

    DEBUG_END();
    return y;
}



mp_float
mp_mul_double_q		WITH_5_ARGS(
	mp_float,	x,
	mp_int,		i,
	mp_int,		j,
	mp_int,		k,
	mp_int,		l
)
/*
Sets x *= i * j / (k * l).  The routine calls mp_mul_q() once if i *
j and k * l are not too large, otherwise mp_mul_q() is called twice.
Rounding is not the best possible, but directed rounding is in the
correct direction.
*/
{
    mp_ptr_type		xp = mp_ptr(x);


    DEBUG_BEGIN(DEBUG_OTHER);
    DEBUG_PRINTF_1("+mul_double_q {\n");
    DEBUG_1("x = ", xp);
    DEBUG_PRINTF_5("i = %d, j = %d, k = %d, l = %d\n\n", i, j, k, l);


    if (j && int_abs(i) <= MAX_INT / int_abs(j) && l && int_abs(k) <= MAX_INT / int_abs(l))
    {
	/*
	Here it is safe to form i * j and k * l.
	*/

	mp_int		ij = i * j, kl = k * l;

	if (ij == 1)
	    mp_div_int_eq(x, kl);
	
	else if (kl == 1)
	    mp_mul_int_eq(x, ij);
	
	else
	    mp_mul_q_eq(x, ij, kl);

    }

    else
    {
	/*
	Split up into two calls, taking care for directed rounding.
	*/

	mp_mul_q_eq(x, i * int_sign(j), k * int_sign(l));
	mp_mul_q_eq(x, int_abs(j), int_abs(l));
    }

    DEBUG_1("-} x = ", xp);
    DEBUG_END();

    return x;
}
