#include "defs.h"
#include "integer.e"
#include "poly.h"
#include "poly_z_faclst.h"
#include "poly_mat.h"
#include "zm.e"
#include "debug.e"

dyn_arr_handle
poly_u_zm_fact_b4_small WITH_4_ARGS(
    t_handle,      pring,
    integer_big,   pdig,
    t_poly,     apoly,
    t_handle,        vectorsh
)
/*
** poly_u_zm_fact_b4_small: Modular univariate polynomial factorisation using
** the Berlekamp algorithm - Step 4. ( MUPFBL ). Deterministic algorithm.
**
** pdig: prime beta-integer
** vectorsh: matrix of vector entries of nullspace of Qmatrix.
** apoly: univariate polynomial over Zpdig.
** Returns: a list of the factors of apoly in polynomial form.
*/
{
    block_declarations;
    t_handle           aph;              /* Handle for apoly                   */
    t_int     apvar;            /* princvar for apoly                 */
    t_poly        gcdpoly;          /* gcd at various places              */
    t_poly        factorpoly;       /* polynomial used in determining     */
                                       /* factors                            */
    t_poly        diffpoly;         /* polynomial used as poly1 - poly2   */
    t_poly        vkpoly;           /* polynomial converted from the kth  */
                                       /* vector of the null space           */
    t_handle           resmh;            /* result t_handle                      */
    t_int    nterms;           /* number of terms in the vectors     */
    t_handle           vkph;             /* Handle for the dense vector k      */
    t_int    factorno;         /* factor counter                     */
    t_int    upperfactorno;    /* upper limit for looping through    */
                                       /* the already found factors          */
    t_int    fcount;           /* factor counter                     */
    t_int    termno;           /* term counter                       */
    t_int    total_factors;    /* number of vectors in null space    */
                                       /* = total number of factors of apoly */
    t_int    nfactors;         /* number of factors determined       */
    t_int    k;                /* vector counter                     */
    integer_big      s;                /* element of Zpdig                   */
    t_handle           tmph;
    t_poly        temp;
    t_int    factorpower;

    if ( pdig >= 25 )
    {
	return poly_u_zm_fact_b4_large( pring, pdig, apoly, vectorsh );
    }

    aph = m_poly_poly_to_handle( apoly );
    apvar = m_poly_princvar( aph );
    nterms = m_poly_mat_col( vectorsh );

    /* Initalise result table */

    total_factors = m_poly_mat_row( vectorsh );

    resmh = m_poly_z_faclst_alloc( total_factors );
    m_poly_z_faclst_factor( resmh, 0 ) = m_modpoly_incref( pring, apoly );
    m_poly_z_faclst_power( resmh, 0 ) = 1;
    m_poly_z_faclst_len_put( resmh, 1 );
    nfactors = 1;

    k = 1;
    while ( ( nfactors < total_factors ) && ( k <= total_factors ) )
    {
        /* get vkpoly from vectorsh */

        m_poly_create_empty(&vkph, apvar, apvar, nterms );

        for ( termno = 0; termno < nterms; termno++ )
        {
            m_poly_coefft( vkph, termno ) = m_poly_mat_elt( vectorsh, k ,termno+1 );
	    m_poly_expt( vkph, termno ) = termno;
        }

	IF_DEBUG_FLAG(DEBUG_MODUPOLY_FACT_B4_SMALL,
	{
		poly_z_write( pring, m_poly_handle_to_poly( vkph ) );
		dmplin();
	}
	);


        vkpoly = poly_z_clean( pring, m_poly_handle_to_poly( vkph ) );

	IF_DEBUG_FLAG(DEBUG_MODUPOLY_FACT_B4_SMALL,
	{
		poly_z_write( pring, vkpoly );
		dmplin();
	}
	);

        /* for every factor already found */

        upperfactorno = nfactors;
        for ( factorno = 0; factorno < upperfactorno; factorno++ )
        {
            factorpoly = m_poly_z_faclst_factor( resmh, 0 );	/* transfer */
	    factorpower = m_poly_z_faclst_power( resmh, 0 );

            /* remove factorpoly from resmh */

            for ( fcount = 1; fcount < nfactors; fcount++ )
            {
                m_poly_z_faclst_factor( resmh, fcount - 1 ) = m_poly_z_faclst_factor( resmh, fcount );

                /* no need to incref, delref as we are only shuffling the */
                /* polynomials inside the matrix.                         */

                m_poly_z_faclst_power( resmh, fcount - 1 ) = m_poly_z_faclst_power( resmh, fcount );
            }

            nfactors--;
	    m_poly_z_faclst_len_put( resmh, nfactors );

            for ( s = 0; s < pdig; s++ )
            {
		temp = modint_negate( pdig, s );
                m_poly_create_empty(&tmph, apvar, apvar, 1);
		m_poly_coefft (tmph, 0) = temp;
		m_poly_expt (tmph, 0) = 0;
		temp = m_poly_handle_to_poly (tmph);
                diffpoly = modpoly_add( pring, pdig, vkpoly, temp );
		poly_z_handle_delref (pring, tmph);

                gcdpoly = poly_u_zm_gcd( pring, pdig, diffpoly, factorpoly );
		m_modpoly_delref( pring, diffpoly );

                /* gcd( vk(x) - s, factorpoly ) */

                if ( ! poly_z_is_one_poly (pring, gcdpoly))
                {
                    /* Store factor in result */

                    m_poly_z_faclst_factor( resmh, nfactors ) = gcdpoly;
                    m_poly_z_faclst_power( resmh, nfactors ) = factorpower;
                    nfactors++;
		    m_poly_z_faclst_len_put( resmh, nfactors );
                }
		else
		{
		    m_modpoly_delref (pring, gcdpoly);
		}
            }

            m_modpoly_delref( pring, factorpoly );
        }

        m_modpoly_delref( pring, vkpoly );
        k++;
    }

	IF_DEBUG_FLAG(DEBUG_MODUPOLY_FACT_B4_SMALL,
	{
		cay_print( "b4 returns: nfactors = %d\n", nfactors );
		cay_print( "resmh is\n" ); poly_z_faclst_write(pring,  resmh );
		dmplin();
	}
	);

    return  resmh;
}
