#include "defs.h"
#include "integer.e"
#include "error.e"
#include "poly.h"

t_poly
poly_z_mult WITH_3_ARGS(
    t_handle,    pring,
    t_poly,   apoly,
    t_poly,   bpoly
)
/*
*    IPOLY_MULT: integral polynomial product.
*    apoly,bpoly are integral polynomials
*    returns apoly.bpoly
*/
{
    block_declarations;
    t_handle           aph;         /* t_handle to apoly     */
    t_handle           bph;         /* t_handle to bpoly     */
    t_handle           adashph;     /* t_handle to adashpoly */
    t_poly        adashpoly;
    t_poly        respoly;     /* result polynomial   */
    t_poly        bcoefft;
    t_poly        temp;        /* used for swapping   */
    t_int    bexpt;
    t_int    i;           /* loop counter        */
    t_int    j;           /* loop counter        */
    t_int    aterms;      /* no. terms in apoly  */
    t_int    bterms;      /* no. terms in bpoly  */

    if ( m_poly_const( apoly ) && m_poly_const( bpoly ) )
    {
        /* both polys constant */
        return  integer_mult( apoly, bpoly );
    }

    if ((m_poly_const (bpoly)) || (m_poly_const (apoly)))
    {
        /* only one is constant */
        error_internal ("poly_z_mult: polys not lifted\n");
        return  0;
    }

    /* both polynomials non_trivial */

    bph = m_poly_poly_to_handle( bpoly );
    aph = m_poly_poly_to_handle( apoly );

    bterms = m_poly_nterms( bph );
    aterms = m_poly_nterms( aph );

    if ( ( aterms == 0 ) || ( bterms == 0 ) )
    {
        /* zero result */
        error_internal ("poly_z_mult: zero term poly\n");
        return  0;
    }

    if ((m_poly_princvar (aph)   != m_poly_princvar (bph))
    ||  (m_poly_least_pvar (aph) != m_poly_least_pvar (bph)))
    {
        /* polys in different variables. */
        error_internal ("%d %d %d %d poly_z_mult: diff variables\n", m_poly_princvar (aph) ,
m_poly_princvar (bph), m_poly_least_pvar (aph), m_poly_least_pvar (bph));
        return  0;
    }

    /* term by term of bpoly apply multiplication by apoly  */
    /* then add resulting polynomials to obtain product.    */

    m_poly_create_empty(&adashph, m_poly_princvar (aph),
                                      m_poly_least_pvar (aph), aterms );
    adashpoly = m_poly_handle_to_poly (adashph);

    respoly = poly_z_zero_poly (pring, apoly);

    for ( i = 0; i < aterms; i++ )
    {
        /* this should be poly_z_zero_poly, but it will be delreffed */
        /* before it is used, so 0 will do                          */

        m_poly_coefft( adashph, i ) = 0;
    }

    for ( j = 0; j < bterms; j++ )
    {
        bexpt = m_poly_expt( bph, j );
        bcoefft = m_poly_coefft( bph, j );

        for ( i = 0; i < aterms; i++ )
        {
            /* remove old term from temporary polynomial */

	    m_poly_z_delref( pring, m_poly_coefft( adashph, i ) );

            /* place new coefficient in temporary polynomial */

            m_poly_expt( adashph, i ) = bexpt + m_poly_expt( aph, i );
            m_poly_coefft( adashph, i )
		= poly_z_mult( pring, bcoefft, m_poly_coefft( aph, i ) );
        }

	temp = respoly;
        respoly = poly_z_add( pring, temp, adashpoly );
	m_poly_z_delref( pring, temp );

        if ( m_poly_has_other_refs( adashph ) )
        {
            /* need to create new adashpoly unless no more iterations */

            if ( j != bterms - 1 )
            {
                /* allocate new space for adashpoly */

	        m_poly_z_delref( pring, adashpoly );
                m_poly_create_empty(&adashph, m_poly_princvar (aph),
                                        m_poly_least_pvar (aph), aterms );
                adashpoly = m_poly_handle_to_poly( adashph );

                for ( i = 0; i < aterms; i++ )
                {
                    m_poly_coefft( adashph, i ) = 0;
                }
            }
        }
    }

    m_poly_z_delref( pring, adashpoly );
    return  respoly;

} /* poly_z_mult() */



t_poly
poly_z_integer_mult WITH_3_ARGS(
    t_handle,      pring,   /* parent poly ring   */
    t_poly,     apoly,   /* multiplicand       */
    integer_big,   aint     /* multiplier         */
)
/*
*    IPOLY_INTEGER_MULT: integral polynomial integer product.
*    apoly is an integral poly, aint an integer .
*    returns aint.apoly.
*    IPIP
*/
{
    t_handle           resph;       /* t_handle to respoly  */
    t_handle           aph;         /* t_handle to apoly    */
    t_poly        acoefft;
    t_int    termno;      /* loop counter       */
    t_int    nterms;      /* no. terms in apoly */

    if ( m_poly_const( apoly ) )
    {
        /* constant polynomial */
        return  integer_mult( apoly, aint );
    }

    /* general case: non_trivial polynomial */

    if ( aint == 0 )
	return  poly_z_zero_poly (pring, apoly);

    if ( aint == 1 )
	return  m_poly_z_incref (pring, apoly);

    aph = m_poly_poly_to_handle( apoly );
    nterms = m_poly_nterms( aph );
    m_poly_create_empty(&resph, m_poly_princvar (aph),
                                        m_poly_least_pvar (aph), nterms);

    if ( m_poly_univariate (aph))
    {
        for ( termno = 0;termno < nterms; termno++ )
        {
            /* multiply term by term */

            acoefft = m_poly_coefft( aph, termno );
            m_poly_expt( resph, termno ) = m_poly_expt( aph, termno );
            m_poly_coefft( resph, termno ) = integer_mult( acoefft, aint );
        }
    }
    else
    {
        for ( termno = 0;termno < nterms; termno++ )
        {
            /* multiply term by term */

            acoefft = m_poly_coefft( aph, termno );
            m_poly_expt( resph, termno ) = m_poly_expt( aph, termno );
            m_poly_coefft( resph, termno ) = 
                            poly_z_integer_mult( pring, acoefft, aint );
        }
    }

    return  m_poly_handle_to_poly( resph );

} /* poly_z_integer_mult() */

