/*******************************************************************************
                   
anf_order.h
   
Header file for orders in algebraic number fields.
This file should only be #included by anf.h.
 
An order ("order") is defined via a C-structure which (besides other
items) contains informations about a basis (which might be power basis; 
in this case a t_handle to the minimal polynomial is provided) and
the multiplication table (not necessarily over Z).
    
 
History: 
        93-01-25 MJ/KW     additional unit macros:
                           order_units_logs_assure(h)
                           order_reg_assure(h)
        92-09-17 JS        Move to Cayley V4
        92-06-04 JS        order_fincke_*
        92-06-04 JS        split off from anf.h

*******************************************************************************/


/******************************************************************************
/ 
/  Structure definition for orders
/
*******************************************************************************/

typedef enum {
	ORDER_ATT_MAXIMAL,
	ORDER_ATT_BASIS_POWER,
	ORDER_ATT_BASIS_REL,
	ORDER_ATT_BASIS_TABLE,
	ORDER_ATT_UNITS_FUND,
	ORDER_ATT_TOTAL
	} order_att_numbers;
  
typedef struct
	{
		t_ring_common	ring_hdr;

		integer_small	abs_degree;
		integer_small	r1;
		integer_small	r2;
		integer_small	r;

		anf		field;

		int		value[ att_req_length(ORDER_ATT_TOTAL)];
		int		known[ att_req_length(ORDER_ATT_TOTAL)];

		t_poly		poly;

		matrix		tran;
		integer_big	tran_den;
		integer_big	index;
		matrix		invtran;
		order		suborder;
 
		anf_elt		pure_gen;

		dyn_arr_handle	basis_table;

		integer_big	disc;

		dyn_arr_handle	mult_table;
		order		coef_order;
		integer_small	rel_degree;

		integer_small	one_position;
 
		matrix		basis_real;
		matrix		basis_real_inv;
 
		dyn_arr_handle	unit_basis;
		matrix		units_logs;
		anf_elt		torsion_unit;
		integer_small	torsion_rank;
		t_real		reg;
 
		t_handle		reals;
		t_real		sqrt_2;
 
		integer_big	class_number;
       		dyn_arr_handle	class_group_structure;
       		dyn_arr_handle	class_group_gens;
       		matrix		class_group_genexps;
		                                    
		dyn_arr_handle	subfield_order_info;
 
		lattice		units_log_lat;
		lat_enum_env	units_log_lat_env;
 
		dyn_arr_handle	fac_basis;
		matrix		relation_mat;
		vector		relation_numbers;
		matrix		relation_trans_mat;
		integer_small	relation_status;
		integer_small	relation_count;
 
                t_real          fincke_gamma1;
                t_real          fincke_lambda;
                vector          fincke_rbounds;
 
		t_handle		dummy_1;
		t_handle		dummy_2;
	}
	t_order_table;

  
/*******************************************************************************
/
/   Macros for accessing structure items of orders
/
*******************************************************************************/
  
/*
    access macro for table element (internal use)
*/
    
#define order_access(h)	((t_order_table *) ring_access(h))
  
/*
    absolute degree of extension 
    number of real embeddings 
    half number of complex embeddings 
    t_handle of quotient field
*/ 
  
#define order_abs_degree(h)	(order_access(h) -> abs_degree)
#define order_r1(h)		(order_access(h) -> r1)
#define order_r2(h)		(order_access(h) -> r2)
#define order_r(h)		(order_access(h) -> r)
#define order_anf(h)		(order_access(h) -> field)
  
  
/*
    Macro for accessing flags (internal use)
*/
 
#define order_known(h)		(order_access(h) -> known)
#define order_value(h)		(order_access(h) -> value)
  
 
/*
    What do we know about the order and its base?
       ... we know it is maximal?
       ... we know it is not maximal?
       ... the actual basis is power basis?
       ... the actual basis is given relative to a suborder?
       ... the actual basis is given via a table of algebraic numbers/orders?
       ... the actual basis is power basis of a pure polynomial?
*/
 
#define order_is_maximal(h) \
    att3_is_true(order_known(h), order_value(h), ORDER_ATT_MAXIMAL)
#define order_isnt_maximal(h) \
    att3_is_false(order_known(h), order_value(h), ORDER_ATT_MAXIMAL)
 
#define order_basis_is_power(h) \
    att2_is_true(order_value(h), ORDER_ATT_BASIS_POWER)

#define order_basis_is_rel(h) \
    att2_is_true(order_value(h), ORDER_ATT_BASIS_REL)

#define order_basis_is_table(h) \
    att2_is_true(order_value(h), ORDER_ATT_BASIS_TABLE)
 
#define order_basis_is_pure(h)	(order_pure_gen(h) != MEM_NH)
  
/*
    Is the unit system fundamental?
*/
 
#define order_units_are_fund(h) \
    att3_is_true(order_known(h), order_value(h), ORDER_ATT_UNITS_FUND)
 
/*
    Setting the flags
       ... we know it is maximal!
       ... we know it is not maximal!
       ... the actual basis is power basis!
       ... the actual basis is given relative to a suborder!
       ... the actual basis is given via a table of algebraic numbers/orders!
       ... the unit system is fundamental!
      
*/
 
#define order_set_is_maximal(h) \
    att3_set_true(order_known(h), order_value(h), ORDER_ATT_MAXIMAL)
#define order_set_isnt_maximal(h) \
    att3_set_false(order_known(h), order_value(h), ORDER_ATT_MAXIMAL)
 
#define order_set_basis_is_power(h) \
    att2_set_true(order_value(h), ORDER_ATT_BASIS_POWER)

#define order_set_basis_is_rel(h) \
    att2_set_true(order_value(h), ORDER_ATT_BASIS_REL)

#define order_set_basis_is_table(h) \
    att2_set_true(order_value(h), ORDER_ATT_BASIS_TABLE)
 
#define order_set_units_are_fund(h) \
    att3_set_true(order_known(h), order_value(h),  ORDER_ATT_UNITS_FUND)
  
 
/*
    generating polynomial if basis is power basis
*/
 
#define order_poly(h)		(order_access(h) -> poly)
   

/*
    If basis is given referring to a suborder:
       - transformation matrix (order basis in terms of suborder basis)
       - denominator of transformation matrix
       - index of suborder in order 
       - inverse transition matrix
       - the suborder 
*/
 
#define order_tran(h)		(order_access(h) -> tran)
#define order_tran_den(h)	(order_access(h) -> tran_den)
#define order_index(h)		(order_access(h) -> index)
#define order_invtran(h)	(order_access(h) -> invtran)
#define order_suborder(h)	(order_access(h) -> suborder)

  
/*
    General basis: 
       - Table of orders and order elements (internal use)
       - accessing a certain basis element
       - accessing the order referring to whose basis the element is stored
*/
  
#define order_basis_table(h)	(order_access(h) -> basis_table)
#define order_basis_table_anf_elt(h, i) \
		dyn_arr_element(order_basis_table(h), (i)*2 - 2)
#define order_basis_table_order(h, i) \
		dyn_arr_element(order_basis_table(h), (i)*2 - 1)
 
 
/*
    generator of pure power basis
*/
 
#define order_pure_gen(h)	(order_access(h) -> pure_gen)
  
 
/*
    Discriminant of order
    ... is it known?
    ... assure its existence
*/
   
#define order_disc(h)		(order_access(h) -> disc)
#define order_disc_known(h)	(order_disc(h) != MEM_NH)
#define order_disc_assure(h)	if (!order_disc_known(h)) order_disc_calc(h)

  
/*   
     - multiplication table of basis elements 
     - check whether it is given
     - representation matrix of a certain basis element
*/
 
#define order_mult_table(h)	(order_access(h) -> mult_table)
#define order_mult_table_known(h) \
				(order_mult_table(h) != MEM_NH)
#define order_mult_matrix(h, i) \
		dyn_arr_element(order_mult_table(h), (i) - 1)
 
/* 
     - coefficient order
     - relative degree (= dimension of multiplication table)
*/
   
#define order_coef_order(h)	(order_access(h) -> coef_order)
#define order_rel_degree(h)	(order_access(h) -> rel_degree)
 
/* 
     At the moment we can do multiplication only via the 
     multiplication table. But still...
       - Is multiplication possible in the order given?
       - Assure that multiplication is possible

*/
 
#define order_mult_known(h)  (order_mult_table_known(h))
#define order_mult_assure(h) if (!order_mult_known(h)) \
					order_mult_table_create(h)

  
/*   
     - at which position is the 1 (if it is a basis element)?
*/
    
#define order_one_position(h)	(order_access(h) -> one_position)
 
 
/*
    ... the real basis ...
    ... and its inverse...
    ... are they known?
*/ 
 
#define order_basis_real(h)	(order_access(h) -> basis_real)
#define order_basis_real_inv(h)	(order_access(h) -> basis_real_inv)
#define order_basis_real_known(h)  \
				(order_basis_real(h) != MEM_NH)
#define order_reals_assure(h)	if (!order_basis_real_known(h)) \
					order_reals_create(h)
    
 
 
/*
    accessing unit related data of the order
       ... the units (internal use)
       ... a particular unit
       ... the logarithm matrix of the units
       ... a generator of the torsion subgroup
       ... the rank of the torsion subgroup
       ... the regulator of the unit group
       
*/    
 
#define order_unit_basis(h)	(order_access(h) -> unit_basis)
#define order_unit(h, i) 	dyn_arr_element(order_unit_basis(h), (i)-1)
#define order_units_logs(h)	(order_access(h) -> units_logs)
#define order_torsion_unit(h)	(order_access(h) -> torsion_unit)
#define order_torsion_rank(h)	(order_access(h) -> torsion_rank)
#define order_reg(h)		(order_access(h) -> reg)
 
/*
    counts for units:
       ... accessing number of independent units given
       ... setting number of units and allocating/assuring space
*/
    
#define order_units_count(h) \
		(order_unit_basis(h) == MEM_NH \
		  ? 0 \
		  : dyn_arr_curr_length(order_unit_basis(h)) \
		) 
 
#define order_units_count_set(h, count) \
		((order_unit_basis(h) == MEM_NH \
		  ? (void) (order_unit_basis(h) = dyn_arr_alloc((count))) \
		  : (void) (dyn_arr_assure_space_fun(order_unit_basis(h), (count), 1)) \
		), \
		(void) (dyn_arr_curr_length(order_unit_basis(h)) = (count)))
 
/*
    Are units known at all?
    Is the unit system maximal?
    Are the logarithms known?
    Is the torsion subgroup generator known?
    Is the regulator known?
*/
 
#define order_units_known(h) 		(order_units_count(h))
#define order_units_are_maximal(h)  \
		(order_units_are_fund(h) || order_units_count(h) == order_r(h))
#define order_units_logs_known(h) 	(order_units_logs(h) != MEM_NH)
#define order_units_logs_assure(h) 	if (!order_units_logs_known(h)) order_units_logs_calc(h);
#define order_torsion_unit_known(h)  	(order_torsion_unit(h) != MEM_NH)
#define order_reg_known(h)  		(order_reg(h) != MEM_NH)
#define order_reg_assure(h)     	if (!order_reg_known(h)) order_reg_calc(h);
 
       
 
/*
    - accessing the real field over which the real basis, its inverse,
      the logarithms of the units and the regulator are given.
    - is the real field known?
    - accessing the real precision (0 if no reals are given)
*/                                                                  

#define order_reals(h)		(order_access(h) -> reals)
#define order_reals_known(h)	(order_reals(h) != MEM_NH)
#define order_real_prec(h)	((order_reals(h))  \
					? real_dec_prec(order_reals(h)) \
					: 0)
 
 
/*
    accessing the square root of 2 in the actual precision
*/                                                                  

#define order_sqrt_2(h)		(order_access(h) -> sqrt_2)

/*  
    class number 
    class group structure (internal)
    ideals which generate the class group (internal)
    exponents of generators of each cyclic factor, referring to ideals
*/
 
#define order_class_number(h)	(order_access(h) -> class_number)
#define order_class_group_structure(h)	\
		      		(order_access(h) -> class_group_structure)
#define order_class_group_gens(h)    (order_access(h) -> class_group_gens)
#define order_class_group_genexps(h) (order_access(h) -> class_group_genexps)

 
/*
    number of cyclic factors of the class group
    order of a certain cyclic factor of the class group
    number of ideals needed for description of class group
    exponent of a particular ideal of a particular cyclic factor generator
*/

 
#define order_class_group_order(h) \
                (order_class_number(h) == 1 ? 0 \
		: dyn_arr_curr_length(order_class_group_structure(h)))
 
#define order_class_group_factor_order(h, i) \
		dyn_arr_element(order_class_group_structure(h), (i)-1)
 
#define order_class_group_gen_count(h) \
                (order_class_number(h) == 1 ? 0 \
		: dyn_arr_curr_length(order_class_group_gens(h)))
 
#define order_class_group_gen(h, i) \
		dyn_arr_element(order_class_group_gens(h), (i)-1)
 
#define order_class_group_factor_gen_exp(h, i, j) \
		mat_elt(order_class_group_genexps(h), i, j)

  
/*
    is the class number known?
    is the class group structure known?
*/
 
#define order_class_number_known(h) (order_class_number(h))
#define order_class_group_structure_known(h) \
	(order_class_number(h) == 1 || order_class_group_structure(h) != MEM_NH)
 
/*
    setting the number of cyclic factors of the class group
    setting the number of ideals (no care is taken for the genexp matrix!)
*/   
 
#define order_class_group_order_set(h, count) \
		((order_class_group_structure(h) == MEM_NH \
		  ? (void) (order_class_group_structure(h) = dyn_arr_alloc((count))) \
		  : (void) (dyn_arr_assure_space_fun(order_class_group_structure(h), (count), 1)) \
		), \
		(void) (dyn_arr_curr_length(order_class_group_structure(h)) = (count)))

#define order_class_group_gen_count_set(h, count) \
		((order_class_group_gens(h) == MEM_NH \
		  ? (void) (order_class_group_gens(h) = dyn_arr_alloc((count))) \
		  : (void) (dyn_arr_assure_space_fun(order_class_group_gens(h), (count), 1)) \
		), \
		(void) (dyn_arr_curr_length(order_class_group_gens(h)) = (count)))
 
/*
    accessing the information about subfield orders of ord (internal)
*/
 
#define order_subfield_order_info(h)  (order_access(h) -> subfield_order_info)
 
  
/*
       ... accessing number of subfield orders given
       ... setting number of subgfield orders
*/

#define order_subfield_order_count(h) \
		(order_subfield_order_info(h) == MEM_NH \
		  ? 0 \
		  : (dyn_arr_curr_length(order_subfield_order_info(h)) / 4) \
		)
#define order_subfield_order_count_set(h, len) \
		((order_subfield_order_info(h) == MEM_NH \
		  ? (void) (order_subfield_order_info(h) = dyn_arr_alloc((len) * 4)) \
		  : (void) (dyn_arr_assure_space_fun(order_subfield_order_info(h), (len) * 4, 4)) \
		), \
		(void) (dyn_arr_curr_length(order_subfield_order_info(h)) = (len) * 4))
    
/*
    a certain order of a subfield of the quotient field of the order
    (sorry for the triple "of")
       ... the t_handle of the order
       ... the relative degree
       ... the vector representing a basis of 'down' referring to a basis of 'up'
       ... a particular basis element
       ... the transformation matrix of the conjugates ('down' to 'up')
           ... is it known
*/
 
#define order_subfield_order(h, i) \
		dyn_arr_element(order_subfield_order_info(h), (i)*4 - 4)

#define order_subfield_order_rel_degree(h, i) \
		dyn_arr_element(order_subfield_order_info(h), (i)*4 - 3)

#define order_subfield_order_basis(h, i) \
		dyn_arr_element(order_subfield_order_info(h), (i)*4 - 2)
 
#define order_subfield_order_basis_elt(h, i, j) \
		vec_entry(order_subfield_order_basis(h, i), j)
 
#define order_subfield_order_con_tran(h, i) \
		dyn_arr_element(order_subfield_order_info(h), (i)*4 - 1)


#define order_subfield_order_con_tran_known(h, i) \
                (order_subfield_order_con_tran(h, i))
                                                 
                          
/*
    the logarithm lattice of the units
    the enumeration environment
*/
 
#define order_units_log_lat(h)  	(order_access(h) -> units_log_lat)
#define order_units_log_lat_env(h)  	(order_access(h) -> units_log_lat_env)
 
 
/*
    factor basis of prime ideals in order
    relation matrix
    relation numbers 
    a certain relation number
    transformation matrix
    status 
*/
 
#define order_fac_basis(h)  		(order_access(h) -> fac_basis)
#define order_relation_mat(h)		(order_access(h) -> relation_mat)
#define order_relation_numbers(h)	(order_access(h) -> relation_numbers)
#define order_relation_number(h, i)	(vec_entry(order_relation_numbers(h), i))
#define order_relation_trans_mat(h)	(order_access(h) -> relation_trans_mat)
#define order_relation_status(h)	(order_access(h) -> relation_status)
#define order_relation_count(h)		(order_access(h) -> relation_count)
 
/* 
    factor basis macros:
    ... allocating space for factor basis
    ... accessing a certain prime in factor basis
    ... accessing the prime ideal table of a certain prime
    ... actual total number of prime ideals
    ... dto, different format
    ... setting the number
    ... actual length of the factor basis
    ... dto, different format
    ... setting the length
    ... assuring space
    ... assuring space (with given minimal extension)
 
    not yet implemented: Accessing the faclst of prime ideals

*/
 
#define order_fac_basis_alloc(h, len)   	(order_fac_basis(h) = dyn_arr_alloc((len)*2 + 1))
#define order_fac_basis_prime(h, i)	        (dyn_arr_element(order_fac_basis(h), (i)*2-1))
#define order_fac_basis_ideals(h, i)         	(dyn_arr_element(order_fac_basis(h), (i)*2))
#define order_fac_basis_ideals_count(h)        	((order_fac_basis(h)) \
							? dyn_arr_element(order_fac_basis(h), 0) \
                                                        : 0 )
#define order_fac_basis_ideals_count_get(h, c) 	(c = order_fac_basis_ideals_count(h))
#define order_fac_basis_ideals_count_set(h, c) 	(dyn_arr_element(order_fac_basis(h), 0) = c)

#define order_fac_basis_len(h) \
  (order_fac_basis(h) ? ((dyn_arr_curr_length(order_fac_basis(h))-1) / 2) : 0)

#define order_fac_basis_len_get(h, len)      	(len = order_fac_basis_len(h))
#define order_fac_basis_len_set(h, len)      	(dyn_arr_curr_length(order_fac_basis(h)) = 2*(len) + 1)
#define order_fac_basis_assure_space(h, len) 	(dyn_arr_assure_space(order_fac_basis(h), (len)*2 + 1, 10))
#define order_fac_basis_assure_space_size(h, len, size)  (dyn_arr_assure_space(order_fac_basis(h), (len)*2 + 1, size))

/*
    Elements referring to the norm equation method of U. Fincke:
    ... gamma1  (this is Fincke's optimal gamma, divided by K)
    ... lambda  (the zero of Fincke's function)
    ... rbounds (what Fincke calls S_j = bounds for exponents r_j)
    ... accessing a certain rbound
*/
 
#define order_fincke_gamma1(h) 		(order_access(h) -> fincke_gamma1)
#define order_fincke_lambda(h) 		(order_access(h) -> fincke_lambda)
#define order_fincke_rbounds(h) 	(order_access(h) -> fincke_rbounds)
#define order_fincke_rbound(h, i )   	(vec_entry(order_fincke_rbounds(h), i))
 
/* 
    ... are rbounds known?
    ... is gamma1 known?
    ... is lambda known?
*/
 
#define order_fincke_rbounds_known(h)   (order_fincke_rbounds(h))
#define order_fincke_gamma1_known(h)    (order_fincke_gamma1(h))
#define order_fincke_lambda_known(h)    (order_fincke_lambda(h))

/* 
    ... allocating space for rbounds (this should be done by order_create - 
                                      if order_abs_degree(ord) is known)
    ... assuring that rbounds are known
*/
 
#define order_fincke_rbounds_alloc(h)  	(order_fincke_rbounds(h) = vec_new(order_abs_degree(ord)))
#define order_fincke_rbounds_assure(h)	if (!order_fincke_rbounds_known(h)) \
						order_fincke_rbounds_alloc(h)
 

/*  
    incrementing the reference counter of an order
*/
  
#define order_incref(h)		ring_incref(h)
 
 
/*
    Many of the algorithms rely on the fact that the order in question
    is given over Z. The following macro provides a way of checking this easily.
*/
 
#define order_must_be_over_z(h)	\
	if(ring_type(order_coef_order(h)) != RING_Z) 	\
		error_internal("This routine can only t_handle Z-orders.")
 

