/*
 *	Macros to manipulate individual bits of integer arrays.
 *	Those with "att2" as prefix are for 2-valued attribute arrays,
 *	i.e. True/False.
 *	"arr" is a pointer to the integer array.
 *	For "set_all" operations, "n" is the number of bits.
 *	For others, the operation is done on the nth bit, starting from 0.
 *
 *	Those with "att3" as prefix are for 3-valued attribute arrays,
 *	i.e. Unknown/True/False.
 *	"kn" is a pointer to the "Known/Unknown" integer array.
 *	"val" is a pointer to the "True/False" integer array.
 *	"n" is as for 2-valued arrays.
 */

#ifndef _ATT_H_
#define _ATT_H_

#define ATT_INT_SIZE (sizeof(int) << 3)
#define ATT_BIT_MASK (ATT_INT_SIZE - 1)


/*
 *	returns the required integer array length for "n" bits
 */

#define att_req_length(n)	(((n) + ATT_INT_SIZE - 1) / ATT_INT_SIZE)

/**
 **	att2 macros
 **/

#define att2_set_true(arr, n)		\
		(arr [((int) n) / ATT_INT_SIZE] |= 1 << (((int) n) & ATT_BIT_MASK))

#define att2_set_false(arr, n)		\
		(arr [((int) n) / ATT_INT_SIZE] &= ~(1 << (((int) n) & ATT_BIT_MASK)))

/*
 * This sets all the bits to 1 in all the words.
 * "n" is the number of bits.
 * I.E.
 * If the last bit lies in the middle of a word the WHOLE word will be set.
 */
#define att2_set_all_true(arr, n)	\
	{	int att_limit, att_i;		\
		att_limit = (((int) n) - 1) / ATT_INT_SIZE;		\
		for (att_i = 0; att_i <= att_limit; att_i++)	\
			arr [att_i]	= ~0;		\
	}

/*
 * This sets all the bits to 0 in all the words.
 * "n" is the number of bits.
 * I.E.
 * If the last bit lies in the middle of a word the WHOLE word will be cleared.
 */
#define att2_set_all_false(arr, n)	\
	{	int att_limit, att_i;		\
		att_limit = (((int) n) - 1) / ATT_INT_SIZE;		\
		for (att_i = 0; att_i <= att_limit; att_i++)	\
			arr [att_i]	= 0;		\
	}

#define att2_is_true(arr, n)	\
		(arr [((int) n) / ATT_INT_SIZE] & (1 << (((int) n) & ATT_BIT_MASK)))

#define att2_is_false(arr, n) (!(att2_is_true(arr, n)))

/**
 **	att3 macros
 **/

#define att3_set_true(kn, val, n)	\
	{	int att_bit, att_word;	\
		att_bit = 1 << (((int) n) & ATT_BIT_MASK);	\
		att_word = ((int) n) / ATT_INT_SIZE;			\
		kn  [att_word]	|= att_bit;			\
		val [att_word]	|= att_bit;			\
	}

#define att3_set_false(kn, val, n)	\
	{	int att_bit, att_word;	\
		att_bit = 1 << (((int) n) & ATT_BIT_MASK);	\
		att_word = ((int) n) / ATT_INT_SIZE;		\
		kn  [att_word]	|= att_bit;			\
		val [att_word]	&= ~att_bit;		\
	}

/*
 * set to "known & true"
 * This sets all the bits to 1 in all the words.
 * "n" is the number of bits.
 * I.E.
 * If the last bit lies in the middle of a word the WHOLE word
 * will be set.
 */
#define att3_set_all_true(kn, val, n)	\
	{	\
		att2_set_all_true(kn, n);	\
		att2_set_all_true(val, n);	\
	}

/*
 * set to "known & false"
 * This sets all the bits to 0 (or 1) in all the words.
 * "n" is the number of bits.
 * I.E.
 * If the last bit lies in the middle of a word the WHOLE word
 * will be cleared (or set).
 */
#define att3_set_all_false(kn, val, n)	\
	{	\
		att2_set_all_true(kn, n);	\
		att2_set_all_false(val, n);	\
	}

#define att3_set_all_known(kn, val, n)	(att2_set_all_true(kn, n))

#define att3_set_all_unknown(kn, val, n)	\
	{	\
		att2_set_all_false(kn, n);		\
		att2_set_all_false(val, n);		\
	}

/*
 * known & true
 */
#define att3_is_true(kn, val, n)	\
		(att2_is_true(kn, n) && att2_is_true(val, n))

/*
 * known & false
 */
#define att3_is_false(kn, val, n)	\
		(att2_is_true(kn, n) && att2_is_false(val, n))

#endif
