;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;        Copyright (C) 1986 by Douglas A. Young,
;;;        Kent State University, Kent Ohio
;;;        Unrestricted permission is granted to copy, modify
;;;        or redistribute this file.
;;;        Douglas A. Young phone: (415) 857-6478
;;;                         net  : dayoung@hplabs.hp.com
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;   
;;;   
;;;   
;;;   
;;;   SYMBOLIC ALGEBRA INTERFACE: DEVDEP.L
;;;   This file contains the device dependent functions. This includes
;;;   functions to draw boxes and rectangles,
;;;   paint strings, halftones, different cursors, and definitions of
;;;   graphics and display variables. Some of these functions
;;;   are taken from the Tektronix 4404 graphics library and support.
;;;   
(sstatus translink t)
(require  '//user//vaxima//young//devdep//gelib)
(setq tefont  "//user//vaxima//young//devdep//tefont.f")
(declare (macros t)
          (special **graphrect **textrect **visrect **copyform **screen
	           **window_list** **fromcopybb **tocopybb **drawbb
		   **erasebb **maxrect **texterasebb **prev-dstate
		   **current-window** **cursors **oldrow** **oldcol**
		   $linenum linenum ttyheight $gcdisable $linel 
		   **top-level-menu** **select-menu** tefont **font
		   **verify-delete-menu** $outchar lg-character-x
		   **win_number** **prompt-area**)
)
;;;   
;;;   DEFINITIONS OF GLOBAL SCREEN RELATED VARIABLES
;;;     these are primarily used to store screen forms
(defvar **graphrect nil)	; graphics display region
(defvar **textrect **visrect)	; text scrolling region
(defvar **copyform nil)		; intermediate form for drawing
(defvar **copyformbytes	nil)	; number of bytes in **copyform
(setq **window_list** nil)      ;  list of all current windows
(defvar **fromcopybb (make-bbcom destform **screen rule bbsord))
(defvar **tocopybb (make-bbcom rule bbsord))
(defvar **drawbb (make-bbcom destform **screen rule bbsord))
(defvar **erasebb
   (make-bbcom cliprect **maxrect destrect **maxrect rule bbzero))
(defvar **texterasebb
   (make-bbcom destform **screen cliprect **maxrect rule bbzero))
(defvar **prev-dstate nil)	; pointer to previous display state structure
;;;   
;;;   
;;;   
(defmacro exit-graphics-mode ()
   ; restores previous display state and normal text scrolling region
   ; no-op if not in graphics mode
   '(progn
       (if (and (boundp '**prev-dstate)
		(not (null **prev-dstate)))	  
	  then (restore_displaystate **prev-dstate)
	       (setq **prev-dstate nil)
	       (setq **copyform nil)
	       (setq **copyformbytes 0)
	       (set-scrolling-region)
	       (setf (bbcom->destrect **texterasebb) **textrect)
	       (clear_screen)) t))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;   Function: remove
;;;   
;;;      Purpose: Macsyma redefines remove as something else
;;;               so this provides the franz remove function as remove*
;;;      Written By: Douglas A. Young
;;;      Date: Fri Mar 14 22:57:01 1986
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun remove* (item list &optional (cnt -1))
   (let ((head '())
	 (tail nil))
      (do ((l list (cdr l))
	   (newcell))
	  ((null l) head)
	  (cond ((or (not (equal (car l) item))
		     (zerop cnt))
		 (setq newcell (list (car l)))
		 (cond ((null head) (setq head newcell))
		       (t (rplacd tail newcell)))
		 (setq tail newcell))
		(t (setq cnt (1- cnt)))))))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;   Function: draw-line 
;;;   
;;;      Purpose: 
;;;    draws line from point1 to point2
;;;    optional bitblt may be passed (if so, then other key-args ignored)
;;;    default rule is inverse of current video, default width is one
;;;    default cliprect is **graphrect
;;;   
;;;      Written By: Douglas A. Young
;;;      Date: Tue Jan 21 22:29:10 1986
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun  draw-line (point1
		     point2 &key
		     (bb nil) (rule bbsord) (width 1)
		     halftone
		     (destform **screen) (cliprect **graphrect))
   (if (null bb)
      then (setq bb **drawbb)
	   (setf (bbcom->destform bb) destform)
	   (setf (bbcom->cliprect bb) cliprect)
	   (setf (bbcom->rule bb) rule)
	   (setf (bbcom->halftoneform **drawbb) halftone)
	   (setf (bbcom->destrect.w bb) width)
	   (setf (bbcom->destrect.h bb) width))
   (setf (bbcom->destrect.x bb) (point->x point1))
   (setf (bbcom->destrect.y bb) (point->y point1))
   (paint_line bb point2) t)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;   Function: draw-rectangle
;;;   
;;;      Purpose: 
;;;    draws a filled-in rectangle at the region defined by rect
;;;    default destform is **screen, default cliprect is **graphrect
;;;    default rule is inverse of current video
;;;   
;;;      Written By: Douglas A. Young
;;;      Date: Tue Jan 21 22:32:51 1986
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun draw-rectangle (rect &key (destform **screen) bb halftone
			    (cliprect **graphrect) (rule bbs))
   (if bb
      then (setf (bbcom->destrect bb) rect)
	   (bit_blt bb)
      else (setf (bbcom->destform **drawbb) destform)
	   (setf (bbcom->cliprect **drawbb) cliprect)
	   (setf (bbcom->halftoneform **drawbb) halftone)
	   (setf (bbcom->rule **drawbb) rule)
	   (setf (bbcom->destrect **drawbb) rect)
	   (bit_blt **drawbb)
	   (setf (bbcom->halftoneform **drawbb) nil)) t)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;   Function: draw-box
;;;   
;;;      Purpose: 
;;          draws a box around the region specified by rect
;;;         default width is one, default destform is **screen
;;;   
;;;      Written By: Douglas A. Young
;;;      Date: Tue Jan 21 22:33:08 1986
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun draw-box (rect &key (width 1) (destform **screen) halftone
		      (cliprect **graphrect) (rule bbsord))
   (let* ((x (rect->x rect)) (y (rect->y rect))
	  (w (rect->w rect)) (h (rect->h rect))
	  (left (- x width)) (top (- y width))
	  (right (+ x w))    (bottom (+ y h))
	  (pt1 (make-point x left y top))
	  (pt2 (make-point x right y top))
	  (pt3 (make-point x right y bottom))
	  (pt4 (make-point x left y bottom)))
      (draw-line pt1 pt2 :width width :destform destform :cliprect cliprect
		 :rule rule :halftone halftone)
      (draw-line pt2 pt3 :width width :destform destform :cliprect cliprect
		 :rule rule :halftone halftone)
      (draw-line pt3 pt4 :width width :destform destform :cliprect cliprect
		 :rule rule :halftone halftone)
      (draw-line pt4 pt1 :width width :destform destform :cliprect cliprect
		 :rule rule :halftone halftone)
      ) t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;   Function: wait-mouse-click
;;;   
;;;      Purpose: 
;;;              waits for press and release of specified mouse button
;;;              returns list of mouse button, mouse location at the 
;;;              time of button release
;;;   
;;;      Written By: Douglas A. Young
;;;      Date: Tue Jan 21 22:33:37 1986
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun wait-mouse-click (button)
   (do* ((oldmouse 0) (pt (make-point))
	 (newmouse 0))
	((and (zerop newmouse)
	      (if (equal m_any button)
		 then (plusp oldmouse)
		 else (equal oldmouse button))) (values oldmouse pt))
	(setq oldmouse newmouse)
	(setq newmouse (get_buttons))
	(get_mposition pt)))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;   Function: draw_line
;;;   
;;;      Purpose: draws a line from x1,y1 to x2,y2
;;;   
;;;      Written By: Douglas A. Young
;;;      Date: Tue Jan 21 22:33:54 1986
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun draw_line (x1 y1 x2 y2)
   (let(( p1 (make-point x x1 y y1))
	(p2 (make-point x x2 y y2)))
      (draw-line p1 p2))
   )

(defun pause_vaxima () (shell))
(defun clear_line ()
   (setf p1 (make-point x 0 y oldrow))
   (setf p2 (make-point x linel y oldrow))
   (draw-line p1 p2 ':width lg-character-y ':rule bbzero))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;   Function: PAINT-CHAR
;;;   
;;;      Purpose: paint a character the graphics screen
;;;   
;;;      Written By: Douglas A. Young
;;;      Date: Tue Jan 21 22:34:46 1986
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun paint-char (x y ch &optional (font **font) (rule bbsord)
		     (destform **screen))
   (let* ((bb (font->bb font))
	  (form (font->form font))
	  (w (font->header.chwidth font))
	  (first (font->header.first font))
	  )
      (setf (bbcom->rule bb) rule)
      (setf (bbcom->destform bb) destform)
      (setf (bbcom->destrect.y bb) y)
      (setf (bbcom->destrect.x bb) x)
      (setf (bbcom->srcpoint.x bb) (* w (- ch first)))
      (bit_blt bb)) t)

; TEKSTBM  Set Top and Bottom Margins
(defun tekstbm (&optional top bottom)
   (if (null top) then (setq top ""))
   (if bottom
      then (msg "[" top ";" bottom "r")
      else (msg "[" top "r")) t)
; CUP  Cursor Position
(defun CUP (&optional row column)
   (if (null row) then (setq row ""))
   (if column
      then (msg "[" row ";" column "H")
      else (msg "[" row "H")) t)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;   Function: INIT-TERM-EM-FONT
;;;   
;;;      Purpose: Read in the tefont 
;;;   
;;;      Written By: Douglas A. Young
;;;      Date: Tue Jan 21 22:35:25 1986
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun init-term-em-font ()
   (let* ((font_header (make-fontheader first 0. last 255.
				       chwidth 8. chheight 15.))
	  (fontform (read_form tefont))
	  (fontbb (make-bbcom srcform fontform
			      destform **screen
			      destrect (make-rect w 8. h 15.)
			      cliprect **maxrect
			      rule bbs)))
      (make-font header font_header form fontform bb fontbb)))

(defvar **font (init-term-em-font))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;   Function: MOUSE-IN-REGION
;;;   
;;;      Purpose: return true if mouse is the specified rect
;;;   
;;;      Written By: Douglas A. Young
;;;      Date: Tue Jan 21 22:36:26 1986
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun mouse-in-region (rect)
   (let* ((x (rect->x rect))
	  (y (rect->y rect))
	  (pt (make-point))
	  (mouse-pos (get_mposition pt)))
      (and (lessp x (point->x mouse-pos) (+ x (rect->w rect)))
	   (lessp y (point->y mouse-pos) (+ y (rect->h rect))))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;   Function: PAINT-STRING
;;;   
;;;      Purpose: graphically paint a string to the screen
;;;   
;;;      Written By: Douglas A. Young
;;;      Date: Tue Jan 21 22:57:43 1986
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun paint-string (x y str &key (font **font) (rule bbsxord)
		       (destform **screen) (cliprect **maxrect))
   (let ((bb (font->bb font))
	 (form (font->form font))
	 (w (font->header.chwidth font))
	 )
      (setf (bbcom->rule bb) rule)
      (setf (bbcom->destform bb) destform)
      (setf (bbcom->destrect.y bb) y)
      (do* ((chars (exploden str) (cdr chars))
	    (first (font->header.first font))
	    (count 0 (add1 count))
	    (destx x (+ destx w))
	    )
	   ((null chars) t)
	   (setf (bbcom->destrect.x bb) destx)
	   (setf (bbcom->srcpoint.x bb) (* w (- (car chars) first)))
	   (bit_blt bb))) t)

; various halftone forms from smalltalk

(defvar blackhalftone
   (make-form addr (vectori-word
			#xffff #xffff #xffff #xffff
 			#xffff #xffff #xffff #xffff
 			#xffff #xffff #xffff #xffff
 			#xffff #xffff #xffff #xffff)
	      w 16. h 16. offsetw 0 offseth 0 inc 2))

(defvar darkgrayhalftone
   (make-form addr (vectori-word
			#x7777 #xdddd #x7777 #xdddd
 			#x7777 #xdddd #x7777 #xdddd
 			#x7777 #xdddd #x7777 #xdddd
 			#x7777 #xdddd #x7777 #xdddd)
	      w 16. h 16. offsetw 0 offseth 0 inc 2))

(defvar grayhalftone
   (make-form addr (vectori-word
			#x5555 #xaaaa #x5555 #xaaaa
 			#x5555 #xaaaa #x5555 #xaaaa
 			#x5555 #xaaaa #x5555 #xaaaa
 			#x5555 #xaaaa #x5555 #xaaaa)
	      w 16. h 16. offsetw 0 offseth 0 inc 2))

(defvar lightgrayhalftone
   (make-form addr (vectori-word
			#x8888 #x2222 #x8888 #x2222
 			#x8888 #x2222 #x8888 #x2222
 			#x8888 #x2222 #x8888 #x2222
 			#x8888 #x2222 #x8888 #x2222)
	      w 16. h 16. offsetw 0 offseth 0 inc 2))

(defvar verylightgrayhalftone
   (make-form addr (vectori-word
			#x8888 #x0000 #x2222 #x0000
 			#x8888 #x0000 #x2222 #x0000
 			#x8888 #x0000 #x2222 #x0000
 			#x8888 #x0000 #x2222 #x0000)
	      w 16. h 16. offsetw 0 offseth 0 inc 2))
(defvar extralightgrayhalftone
   (make-form addr (vectori-word
			#x8888 #x0000 #x0000 #x0000
			#x8888 #x0000 #x0000 #x0000
			#x8888 #x0000 #x0000 #x0000
			#x8888 #x0000 #x0000 #x0000)			
	      w 16. h 16. offsetw 0 offseth 0 inc 2))

(defvar whitehalftone
   (make-form addr (vectori-word
			#x0000 #x0000 #x0000 #x0000
 			#x0000 #x0000 #x0000 #x0000
 			#x0000 #x0000 #x0000 #x0000
 			#x0000 #x0000 #x0000 #x0000)
	      w 16. h 16. offsetw 0 offseth 0 inc 2))

(defvar **halftones  (list (eval whitehalftone)
			   (eval verylightgrayhalftone)
			   (eval lightgrayhalftone)
			   (eval grayhalftone)
			   (eval darkgrayhalftone)
			   (eval blackhalftone)))

; various cursor images from smalltalk 

(defvar origincursor
   (make-form addr (vectori-word
			#xffff  #xffff  #xc000  #xc000
			#xc000  #xc000  #xc000  #xc000
			#xc000  #xc000  #xc000  #xc000
			#xc000  #xc000  #xc000  #xc000)
	      w 16. h 16. offsetw 0 offseth 0 inc 2))

(defvar cornercursor
   (make-form addr (vectori-word
			#x0003  #x0003  #x0003  #x0003
			#x0003  #x0003  #x0003  #x0003
			#x0003  #x0003  #x0003  #x0003
			#x0003  #x0003  #xffff  #xffff)
	      w 16. h 16. offsetw 16. offseth 16. inc 2))


(defvar waitcursor
   (make-form addr (vectori-word
			#xffff  #x8001  #x4002  #x2004
			#x1c38  #x0f70  #x06e0  #x03c0
			#x03c0  #x05a0  #x0890  #x1188
			#x2344  #x4ff2  #xbffd  #xffff)
	      w 16. h 16. offsetw 8 offseth 8 inc 2))


(defvar normalcursor
   (make-form addr (vectori-word
			#x8000  #xc000  #xe000  #xf000
			#xf800  #xfc00  #xfe00  #xf800
			#xf800  #x9800  #x0c00  #x0c00
			#x0600  #x0600  #x0300  #x0300)
	      w 16. h 16. offsetw 0 offseth 0 inc 2))

(defvar crosshaircursor
   (make-form addr (vectori-word
			#x0100  #x0100  #x0100  #x0100
			#x0100  #x0100  #x0100  #xfffe
			#x0100  #x0100  #x0100  #x0100
			#x0100  #x0100  #x0100  #x0000)
	      w 16. h 16. offsetw 8 offseth 8 inc 2))

(defvar **cursors  (list (eval origincursor)
			 (eval cornercursor)
			 (eval waitcursor)
			 (eval normalcursor)
			 (eval crosshaircursor)
		   ))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;   Function: INIT-FRAMER
;;;   
;;;      Purpose: 
;;;   
;;;      Written By: Douglas A. Young
;;;      Date: Tue Jan 21 22:55:29 1986
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun init-framer ()
   (let ((framer (gensym))
	 (saveform (form_create 640.   480.))
	 (rectform (form_create 640.   480.)))
      (putprop framer saveform ':saveform)
      (putprop framer
	       (make-bbcom srcform **screen destform saveform
			   destrect **visrect
				     cliprect 
				     **visrect
			   rule bbs)
	       ':savebb)
      (putprop framer
	       (make-bbcom srcform saveform destform rectform
			   destrect **visrect cliprect **visrect
			   rule bbs)
	       ':backbb)
      (putprop framer
	       (make-bbcom destform rectform cliprect **visrect
			   halftoneform lightgrayhalftone
			   rule bbsxord)
	       ':rectbb)
      (putprop framer
	       (make-bbcom srcform rectform destform **screen
			   destrect **visrect cliprect **visrect
			   rule bbs)
	       ':showbb)
      framer))

(defvar **framer (init-framer))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;   Function: RECT-FROM-USER
;;;   
;;;      Purpose: Interactively get a rectangle from the user.
;;;               the keyword arguments allow the specification of
;;;                  xx,yy        upper left corner of rect
;;;                  initw,inith  initial size of rectangle
;;;                  minw,minh    a minimum size of rectangle
;;;      Written By: Douglas A. Young
;;;      Date: Tue Jan 21 22:55:07 1986
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun rect-from-user (&key (xx nil)(yy nil)
                            (initw 10.)(inith 10.)(minw 0) (minh 0))
   (if xx then (set_mposition (make-point x xx y yy)))
   				; test
   (do* ((savebb (get **framer ':savebb))
	 (backbb (get **framer ':backbb))
	 (rectbb (get **framer ':rectbb))
	 (showbb (get **framer ':showbb))
	 (saveform (get **framer ':saveform))
	 (mousept (make-point)) mousex mousey buttons
	 (foo (progn (setf (bbcom->srcform savebb) **screen)
		     (setf (bbcom->destform savebb) saveform)
		     (bit_blt savebb)
		     (set_cursor origincursor)
		     (setf (bbcom->destrect.w rectbb) minw)
		     (setf (bbcom->destrect.h rectbb) minh)
		     nil))
	 fixedp boundedp)
	((and fixedp boundedp)
	 (setf (bbcom->srcform savebb) saveform)
	 (setf (bbcom->destform savebb) **screen)
	 (set_cursor normalcursor)
	 (bit_blt savebb)
	 (values
	    (make-rect x (bbcom->destrect.x rectbb)
		       y (bbcom->destrect.y rectbb)
		       w (bbcom->destrect.w rectbb)
		       h (bbcom->destrect.h rectbb))
	    savebb))
	(setq buttons (get_buttons))
	(get_mposition mousept)
	(setq mousex (point->x mousept))
	(setq mousey (point->y mousept))
	(if fixedp
	   then	(setf (bbcom->destrect.w rectbb)
		      (max minw
			   (+ 16. (- mousex (bbcom->destrect.x rectbb)))))
		(setf (bbcom->destrect.h rectbb)
		      (max minh
			   (+ 16. (- mousey (bbcom->destrect.y rectbb)))))
		(if (zerop buttons)
		   then	(setq boundedp t))
	   else	(setf (bbcom->destrect.x rectbb)
		      (if (null xx) then (min mousex (-  640. minw))
			 else xx))
		(setf (bbcom->destrect.y rectbb)
		      (if (null xx) then (min mousey (- 480. minh))
			 else xx))
		(if (equal m_left buttons)
		   then	(setq fixedp t)
			(setf (point->x mousept)
			      (+ minw (minus 16.) (bbcom->destrect.x rectbb)))
			(setf (point->y mousept)
			      (+ minh (minus 16.) (bbcom->destrect.y rectbb)))
			(set_mposition (make-point x (+ initw
			                                (point->x mousept))
			                           y (+ inith
						        (point->y mousept))
					)
			)
			(set_cursor cornercursor)))
	(bit_blt backbb)
	(bit_blt rectbb)
	(bit_blt showbb)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;   Function: FIXED-RECT-FROM-USER
;;;   
;;;      Purpose:Get  a fixed size rectangle of minw and minh from the user
;;;   
;;;      Written By: Douglas A. Young
;;;      Date: Tue Jan 21 22:52:36 1986
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun fixed-rect-from-user (&key (minw 100.) (minh 100.))
   (do* ((savebb (get **framer ':savebb))
	 (backbb (get **framer ':backbb))
	 (rectbb (get **framer ':rectbb))
	 (showbb (get **framer ':showbb))
	 (saveform (get **framer ':saveform))
	 (mousept (make-point)) mousex mousey buttons
	 (foo (progn (setf (bbcom->srcform savebb) **screen)
		     (setf (bbcom->destform savebb) saveform)
		     (bit_blt savebb)
		     (set_cursor origincursor)
		     (setf (bbcom->destrect.w rectbb) minw)
		     (setf (bbcom->destrect.h rectbb) minh)
		     nil))
	 fixedp boundedp)
	((and fixedp boundedp)
	 (setf (bbcom->srcform savebb) saveform)
	 (setf (bbcom->destform savebb) **screen)
	 (set_cursor normalcursor)
         (bit_blt savebb)
	 (values
	    (make-rect x (bbcom->destrect.x rectbb)
		       y (bbcom->destrect.y rectbb)
   		       w (bbcom->destrect.w rectbb)
		       h (bbcom->destrect.h rectbb))
	    savebb))
	(setq buttons (get_buttons))
	(get_mposition mousept)
	(setq mousex (point->x mousept))
	(setq mousey (point->y mousept))
	(if fixedp
	    then (setf (bbcom->destrect.w rectbb) minw)
		(setf (bbcom->destrect.h rectbb) minh)
		(if (zerop buttons)
		   then	(setq boundedp t))
	   else	(setf (bbcom->destrect.x rectbb)
		      (min mousex (- (+ (rect->x **graphrect)
		                         (rect->w **graphrect)) minw 2)))
		(setf (bbcom->destrect.y rectbb)
		      (min mousey (- (+(rect->y **graphrect)
		                       (rect->h **graphrect)) minh 2)))
		(if (equal m_left buttons)
		   then	(setq fixedp t)
			(setf (point->x mousept)
			      (+ minw (bbcom->destrect.x rectbb)))
			(setf (point->y mousept)
			      (+ minh  (bbcom->destrect.y rectbb)))
			(set_mposition mousept)
			(set_cursor cornercursor)))
	(bit_blt backbb)
	(bit_blt rectbb)
	(bit_blt showbb)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;   Function: INIT-GRAPHICS-MODE
;;;   
;;;      Purpose: 
;;;         Initialize the graphics screen, and various graphics variables
;;;   
;;;   
;;;      Written By: Douglas A. Young
;;;      Date: Tue Jan 21 22:06:58 1986
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
 (defun init-graphics-mode (&key lines (viewport-origin **origin)
				(video-normal-p t) (cursor-p t)
				(cursor-loc (make-point x 300. y 200.))
				mouse-bounds	 ; default is **graphrect
				(pan-disk-p nil)( pan-cursor-p nil))
   ; initializes the graphics environment as specified in keyword arguments
   (if (or (not (boundp '**prev-dstate)) (null **prev-dstate))
      then (setq **prev-dstate (save_displaystate (make-dispstate))))
   (setq **screen (init_graphics 1))
   (set_viewport viewport-origin)
   (set_mposition cursor-loc)
   (if cursor-p then (cursor_visible t) else (cursor_visible nil))
   (if (null pan-disk-p) 
       then (pan_disk_enable nil)
       else (pan_disk_enable t))
   (if (null pan-cursor-p) 
        then (pan_cursor_enable nil)
        else (pan_cursor_enable t))
   (if (null video-normal-p) then (video_normal nil))
   (if lines
      then (set-scrolling-region :lines lines)	; sets **graphrect
      else (setq **graphrect **maxrect)
	   (setq **textrect **visrect))
   (let ((gx (rect->x **graphrect))
	 (gy (rect->y **graphrect))
	 (gw (rect->w **graphrect))
	 (gh (rect->h **graphrect)))
      (if mouse-bounds
	 then (set_mbounds (car mouse-bounds) (cadr mouse-bounds))
	 else (set_mbounds (make-point x gx y gy)
			   (make-point x (+ (minus 1) gx gw)
				       y (+ (minus 1) gy gh))))
      (setq **copyform (form_create gw gh))
      (setq **copyformbytes (* (form->inc **copyform) gh))
      (setf (bbcom->destform **tocopybb) **copyform)
      (setf (bbcom->destrect **tocopybb) **graphrect)
      (setf (bbcom->cliprect **tocopybb) **graphrect)
      (setf (bbcom->srcform **fromcopybb) **copyform)
      (setf (bbcom->destrect **fromcopybb) **graphrect)
      (setf (bbcom->cliprect **fromcopybb) **graphrect)
      (setf (bbcom->destrect **texterasebb) **textrect))
   (draw-rectangle **graphrect :halftone grayhalftone)
   t)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;   Function: SET-SCROLLING-REGION
;;;   
;;;      Purpose: 
;;;    sets up a text scrolling region for the terminal emulator
;;;    set **graphrect, the current graphics region and default clipping rect
;;;    set **textrect, the current terminal emulator scrolling region
;;;    lines = 0 limits graphics region to visible portion of screen
;;;    lines = 0>n>32 creates separate n-line text region at bottom of screen
;;;    lines = 32 is default terminal emulator scrolling region
;;;      Written By: Douglas A. Young
;;;      Date: Tue Jan 21 22:08:12 1986
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
 (defun set-scrolling-region (&key (lines 32))
   (let ((pt1 (make-point))
	 (pt2 (make-point)))
      (if (or (equal lines 0) (not (lessp 0 lines 33)))
	 then (setq **graphrect **visrect)
	      (setq **textrect **visrect)
	 else (let ((xx 0) (yy 0)
		    (ww   640.
		              )
		    (hh (- 479. (* lines 15.))))
		 (if (lessp lines 32.)
		    then (draw-line 	; line divides text/display regions
			    (progn (setf (point->x pt1) 0)
				   (setf (point->y pt1) hh) pt1)
			    (progn (setf (point->x pt2) (sub1 ww))
				   (setf (point->y pt2) hh) pt2)
			    :width 3
			    :cliprect **maxrect))
		 (tekstbm (- 33. lines) 32.)
		 (CUP)
		 (setq **graphrect (make-rect x xx y yy w ww h hh))
		 (setq **textrect (make-rect x xx y (+ hh 1)
					     w ww h (- 479. hh)))
		 ))) t)



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;   Function: GET_REL_MPOSITION
;;;   
;;;      Purpose: return the mouse location relative to the origin
;;;               of the current window
;;;   
;;;      Written By: Douglas A. Young
;;;      Date: Sun Feb 02 16:05:18 1986
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun get_rel_mposition  (p1 )
    (get_mposition p1)
    (setf (point->x p1) (- (point->x p1)
                           (rect->x (get **current-window** 'screenrect))))
    (setf (point->y p1) (- (point->y p1)
                           (rect->y (get **current-window** 'screenrect))))
    p1
 )  


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;   Function: SHOW-STATUS
;;;   
;;;      Purpose: display the string s in the status line of the current
;;;               window
;;;   
;;;      Written By: Douglas A. Young
;;;      Date: Sun Feb 02 23:33:20 1986
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun show-status  (s)
   (prog ()
      (cond((or(equal 'shrunk (get **current-window** 'size))
		      (null s)(null **current-window**))(return)))
      (putprop 'current-status s 'string)
      (paint-string (- (+ (rect->x (get **current-window** 'rect))
			  (rect->w (get **current-window** 'rect)))
		       (stringsize s **font) 5)
		    (+ (rect->y (get **current-window** 'rect)) 2)
		    s :rule bbsxord))
   )
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;   Function: CLEAR-STATUS
;;;   
;;;      Purpose: Remove the string from the status line
;;;   
;;;      Written By: Douglas A. Young
;;;      Date: Sun Feb 02 23:40:52 1986
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun clear-status  ()
   (prog ()
      (cond((or(equal 'shrunk (get **current-window** 'size))
		      (null (get 'current-status 'string))
		      (null **current-window**))(return)))
      (paint-string (-(+ (rect->x (get **current-window** 'rect))
			 (rect->w (get **current-window** 'rect)))
			 (stringsize (get 'current-status 'string) **font) 5)
		    (+ (rect->y (get **current-window** 'rect)) 2)
		    (get 'current-status 'string) :rule bbsxord)
      (setplist 'current-status nil))
   )
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;   Function: MSELECT
;;;   
;;;      Purpose: select an expression from the screen with the mouse
;;;   
;;;      Written By: Douglas A. Young
;;;      Date: Mon Feb 03 02:35:34 1986
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun mselect  ()
   (putprop **current-window** 2000 'select-pt-x1)
   (putprop **current-window** 2000 'select-pt-y1)
   (putprop **current-window** -1 'select-pt-x2)
   (putprop **current-window** -1 'select-pt-y2)
   (setf p1 (make-point))
   (setf p2 (make-point))
   ;;;
   ;;;   loop until we either give up or have a valid d-line
   ;;;
   (prog ()
      loop1
      (get_rel_mposition p1)
      (setf x1 (point->x p1))
      (setf y1 (point->y p1))
      (cond((select-d-line (- y1 1)(+ 1 y1))
	    (remember-dline **exptop**  **expbot** ))
			   ((zerop (get_buttons))(return nil))
			   (t(go loop1))))
   ;;;
   ;;;   if we have a valid d-line then start selection
   ;;;
   (do ((buttons (get_buttons)(get_buttons))
	(mouse-loc (get_rel_mposition p2)(get_rel_mposition p2))
	(result (select x1 y1 (point->x p2)
			(point->y p2))
		(select x1 y1 (point->x p2)
			(point->y p2))))
       ((zerop buttons)
	(restore-dline  **exptop**  **expbot**)
	(return (trig_hack result)))
;;;   
;;;   for screen dump
;;;    

       (if (equal buttons 3) then (sdump (form->addr **screen) 
                                     (next-file-name)))
       (prog (t1 track-rect)
	  (setf t1 (make-point))
	  (get_mposition t1)
	  (setf track-rect (make-rect x (- (point->x t1) 3)
				      y (- (point->y t1) 3)
				      h 6
				      w 6))
	  loop2
;;;   
;;;    for screen dump
;;;   
       (if (equal (get_buttons) 3) then (sdump (form->addr **screen) 
                                     (next-file-name)))	  
	  (cond((and(not(zerop(get_buttons)))
		   (mouse-in-region track-rect))(go loop2)))
	  (setf t2 (make-point))
	  (get_mposition t2)
	  (setf xt (point->x t2))
	  (setf yt (point->y t2))
	  (cond((or(<= xt (rect->x track-rect))
		       (<= yt (rect->y track-rect)))
		(restore-dline **exptop** **expbot**)
		(putprop **current-window** 2000 'select-pt-x1)
		(putprop **current-window** 2000 'select-pt-y1)
		(putprop **current-window** -1 'select-pt-x2)
		(putprop **current-window** -1 'select-pt-y2))))

       )
   )

;;;-------------------------------------------------------------------
;;; process_mouse The function to execute a mouse command from
;;; top level editor. Function returns a list of fixnums selected
;;; from the screen, or nil if some other command was processed
;;;------------------------------------------------------------------
 
;(defun process_mouse (button)
;   (prog (result)
;       (setq result (process_mouse_aux button))
;       (return result)))

(defun process_mouse (button)
   (prog (current_mouse_position rect win)
      (cond((zerop button) (return)))
      (setf current_mouse_position (make-point))
      ;;;
      ;;;   check for menu buttons
      ;;;
      (cond((= button m_middle)
	    (eval(menu-choose **top-level-menu**))
	    (return 'refresh))
	       ((= button m_right)
		(show-mac-settings)
		(return 'refresh)))
      ;;;
      ;;;   else if button is M_LEFT
      ;;;
      ;;;   check for no window condition
      ;;;
      (cond((or(null **window_list**)(null **current-window**))
	    (tyo 7)(return)))
      ;;;
      ;;;   wait for click to give user chance to change mind
      ;;;
      (setq rect (get **current-window** 'rect))
      ;;;
      ;;;   check if mouse is in the current window
      ;;;
      (cond((not (mouse-in-region rect))
	    ;;;
	    ;;;   if not, find which window is pointed to
	    ;;;
	    (setq win (mselect_window))
	    (if (null win) then (tyo 7)(return))

	    ;;;
	    ;;;   else if shrunk - grow it
	    ;;;
	    (cond((equal 'shrunk (get win 'size))

		  (grow-window win)(return 'refresh))
			 ;;;
			 ;;;   else pop the window
			 ;;;
			 (t(pop_window win)(return 'refresh))))
		 ;;;
		 ;;;   if mouse is in current window, check if it is shrunk
		 ;;;
		 ((equal 'shrunk (get **current-window** 'size))
		  (grow-window **current-window**))
		 ;;;
		 (t (return(mselect))))))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;   Function: REMEMBER-DLINE
;;;   
;;;      Purpose: Save the contents of the current line in a bitmap form
;;;   
;;;      Written By: Douglas A. Young
;;;      Date: Fri Feb 07 19:29:31 1986
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun remember-dline (y1 y2)
   (let* (( w1 (rect->w (get **current-window**  'screenrect)))
	  (h1 (+ (* 2 lg-character-y) (- y2 y1)))
	  (x1 (rect->x (get **current-window** 'screenrect)))
	  (y1 (+ y1 (minus lg-character-y)
		 (rect->y (get **current-window** 'screenrect)))))
      ;;;
      ;;;   remember displa stuff
      ;;;
      (putprop **current-window** (form_create w1 h1) 'current-dline-display)
      (bit_blt (make-bbcom srcform **screen
			   destform
			   (get **current-window** 'current-dline-display)
			   destrect (make-rect x 0 y 0 w w1 h h1)
			   cliprect **maxrect
			   rule bbs
			   srcpoint (make-point x x1 y y1)))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;   Function: RESTORE-DLINE
;;;   
;;;      Purpose: copy the saved bitmap form back to where it came from
;;;   
;;;      Written By: Douglas A. Young
;;;      Date: Fri Feb 07 19:32:20 1986
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun restore-dline  (y1 y2)
   (prog ()
      (cond((null(get **current-window** 'current-dline-display))(return)))
      (let* (
	       (w1 (rect->w (get **current-window** 'screenrect)))
	       (h1 (+ (* 2 lg-character-y) (- y2 y1)))
	       (x1 (rect->x (get **current-window** 'screenrect)))
	       (y1 (+ y1 (minus lg-character-y)
		      (rect->y (get **current-window** 'screenrect))))
	       (current (get **current-window** 'current-dline-display)))
	 ;;;
	 ;;; set up the transfer for the new location
	 ;;;
	 (bit_blt (make-bbcom srcform  current
			      destform **screen
			      destrect (make-rect x x1 y y1 w w1 h h1)
			      cliprect **maxrect
			      rule bbs
			      srcpoint (make-point x 0 y 0))))))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;   Function: HIGHLIGHT
;;;   
;;;      Purpose: highlight an expression or subexpression
;;;   
;;;      Written By: Douglas A. Young
;;;      Date: Fri Feb 07 19:35:40 1986
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun on_highlight  (x1 y1 x2 y2)
   (prog ()
      (cond((not *highlight-flag*) (return)))
   (cond((< (get **current-window** 'select-pt-x1) x1)
	 (setq x1(get **current-window** 'select-pt-x1)))
	    (t(putprop **current-window** x1 'select-pt-x1)))
   (cond((< (get **current-window** 'select-pt-y1) y1)
	 (setq y1(get **current-window** 'select-pt-y1)))
	    (t(putprop **current-window** y1 'select-pt-y1)))
   (cond((> (get **current-window** 'select-pt-x2) x2)
	 (setq x2 (get **current-window** 'select-pt-x2)))
	    (t(putprop **current-window** x2 'select-pt-x2)))
   (cond((> (get **current-window** 'select-pt-y2) y2)
	 (setq y2 (get **current-window** 'select-pt-y2)))
	    (t(putprop **current-window** y2 'select-pt-y2)))

   (draw-rectangle (make-rect x (+ x1 (minus lg-character-x-2)
				   (rect->x (get **current-window**
						 'screenrect)))
			      y (+ y1 (minus lg-character-y-2)
				   (rect->y (get **current-window**
						 'screenrect)))
			      w (+ lg-character-x (- x2 x1))
			      h (+ lg-character-y (- y2 y1)))
		   :rule bbsord :halftone verylightgrayhalftone)
   ))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;   Function: SETUP_INITIAL_LAYOUT
;;;   
;;;      Purpose: put terminal in graphics mode and open Calc window
;;;   
;;;      Written By: Douglas A. Young
;;;      Date: Thu Jan 30 22:36:24 1986
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun setup_initial_layout  ( )
   (setq gcdisable t)
   (init-graphics-mode :lines 4.)
   (make-window :name 'calc :rectangle (make-rect x 20. y 10. w 570. h 370.))
   (setq gcdisable nil)
   )
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;   Function: SETUP-PROMPT-AREA
;;;   
;;;      Purpose: Draw a box in which prompts can be written
;;;   
;;;      Written By: Douglas A. Young
;;;      Date: Tue Mar 11 23:11:33 1986
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun setup-prompt-area ()
   (let* (( w1 300.)
	  (h1  100.)
	  (x1 50.)
	  (y1 50.))
      (setcursor waitcursor)
      ;;;
      ;;;   remember displa stuff
      ;;;

      (putprop '**prompt-area** (form_create w1 h1) 'prompt-area)
      (bit_blt (make-bbcom srcform **screen
			   destform (get '**prompt-area** 'prompt-area)
			   destrect (make-rect x 0 y 0 w w1 h h1)
			   cliprect **maxrect
			   rule bbs
			   srcpoint (make-point x x1 y y1)))
      (draw-rectangle (make-rect x x1 y y1 w w1 h h1) :rule bbzero)
      (draw-box  (make-rect x (+ 1 x1) y (1+ y1) w (- w1 2) h (- h1 2)))

      ))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;   Function: RESTORE-PROMPT-AREA
;;;   
;;;      Purpose: erase the prompt area and restore the screen
;;;   
;;;      Written By: Douglas A. Young
;;;      Date: Fri Feb 07 19:32:20 1986
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun restore-prompt-area ()
   (let* ((w1 300.)
	  (h1 100.)
	  (x1  50.)
	  (y1  50.)
	  (current (get '**prompt-area** 'prompt-area)))
      (setcursor normalcursor)
      ;;;
      ;;; set up the transfer for the new location
      ;;;
      (bit_blt (make-bbcom srcform  current
			   destform **screen
			   destrect (make-rect x x1 y y1 w w1 h h1)
			   cliprect **maxrect
			   rule bbs
			   srcpoint (make-point x 0 y 0)))))
(defun draw-poly (x1 y1 x2 y2 x3 y3 x4 y4)
   (let(( p1 (make-point x x1 y y1))
	(p2 (make-point x x2 y y2))
	(p3 (make-point x x3 y y3))
	(p4 (make-point x x4 y y4)))
      (draw-line p1 p2)
      (draw-line p2 p3)
      (draw-line p3 p4)
      (draw-line p4 p1)
      )
   )
;;;   
;;;   Make exit clear the graphics mode first
;;;   
(defun $exit ()
   (terminal_enable)(msg "[41")
   (exit-graphics-mode)
   (exit))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;   Function: SET-WINDOW-VALUE
;;;   
;;;      Purpose: Assign an expression to "$%windowname"
;;;   
;;;      Written By: Douglas A. Young
;;;      Date: Sun Mar 09 19:34:25 1986
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun set-window-value  (win)
   (prog (temp)
      (show-status "Select expression with mouse")
      (do () ((not (zerop (get_buttons)))))
      (setq st (reverse(mselect)))
      (setq temp (concat '$ '% win))
      (set temp (car (parse1)))
      (clear-status))
   )
;;;   
;;;   redefine gcbefore to inform us of the garbage collection
;;;   
(putd 'tempgcb (getd 'gcbefore))
(defun gcbefore fexpr (arg)
   (setup-prompt-area)
   (paint-string 65. 88.  "^^^^^^^^^^^^^^^^^^^^^^^^^"
		 :cliprect (make-rect x 50. y 50. w 300. h 100.))
   (paint-string 65. 100. "^^ GARBAGE COLLECTING  ^^"
		 :cliprect (make-rect x 50. y 50. w 300. h 100.))
   (paint-string 65. 114. "^^^^^^^^^^^^^^^^^^^^^^^^^"
		 :cliprect (make-rect x 50. y 50. w 300. h 100.))
   (tempgcb arg))
;;;   
;;;   redefine gcafter to cleanup the GC prompt
;;;   
(putd 'tempgcaft (getd 'gcafter))
(defun gcafter fexpr (arg)
   (cond((equal arg '(nil))(tempgcaft))
		((null arg)(tempgcaft))
		(t(tempgcaft arg)))
   (restore-prompt-area)
   )
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;   Function: SHOW-MAC-SETTINGS
;;;   
;;;      Purpose: A "Options Browser" This menu is created at run time
;;;               so that the current values may be shown
;;;   
;;;      Written By: Douglas A. Young
;;;      Date: Wed Mar 12 22:08:02 1986
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun show-mac-settings  ( )
   (eval
      (menu-choose
	 (make-menu
	    `((,(strcat "Showtime   : " (eval $showtime))
		(toggle-showtime))
	      (,(strcat "Simp       : " (eval $simp))
		(toggle-simp))
	      (,(strcat "Numer     : " (eval $numer))
		(toggle-numer))
	     )
	    :title "Options Menu"
	 ))))
(defun toggle-showtime ()
  (setq $showtime (not $showtime)))
(defun toggle-simp ()
   (setq $simp (not $simp)))
(defun toggle-numer ()
   (setq $numer (not $numer)))

